/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.schemas;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.schemas.model.BadRequestException;
import software.amazon.awssdk.services.schemas.model.ConflictException;
import software.amazon.awssdk.services.schemas.model.CreateDiscovererRequest;
import software.amazon.awssdk.services.schemas.model.CreateDiscovererResponse;
import software.amazon.awssdk.services.schemas.model.CreateRegistryRequest;
import software.amazon.awssdk.services.schemas.model.CreateRegistryResponse;
import software.amazon.awssdk.services.schemas.model.CreateSchemaRequest;
import software.amazon.awssdk.services.schemas.model.CreateSchemaResponse;
import software.amazon.awssdk.services.schemas.model.DeleteDiscovererRequest;
import software.amazon.awssdk.services.schemas.model.DeleteDiscovererResponse;
import software.amazon.awssdk.services.schemas.model.DeleteRegistryRequest;
import software.amazon.awssdk.services.schemas.model.DeleteRegistryResponse;
import software.amazon.awssdk.services.schemas.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.schemas.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.schemas.model.DeleteSchemaRequest;
import software.amazon.awssdk.services.schemas.model.DeleteSchemaResponse;
import software.amazon.awssdk.services.schemas.model.DeleteSchemaVersionRequest;
import software.amazon.awssdk.services.schemas.model.DeleteSchemaVersionResponse;
import software.amazon.awssdk.services.schemas.model.DescribeCodeBindingRequest;
import software.amazon.awssdk.services.schemas.model.DescribeCodeBindingResponse;
import software.amazon.awssdk.services.schemas.model.DescribeDiscovererRequest;
import software.amazon.awssdk.services.schemas.model.DescribeDiscovererResponse;
import software.amazon.awssdk.services.schemas.model.DescribeRegistryRequest;
import software.amazon.awssdk.services.schemas.model.DescribeRegistryResponse;
import software.amazon.awssdk.services.schemas.model.DescribeSchemaRequest;
import software.amazon.awssdk.services.schemas.model.DescribeSchemaResponse;
import software.amazon.awssdk.services.schemas.model.ExportSchemaRequest;
import software.amazon.awssdk.services.schemas.model.ExportSchemaResponse;
import software.amazon.awssdk.services.schemas.model.ForbiddenException;
import software.amazon.awssdk.services.schemas.model.GetCodeBindingSourceRequest;
import software.amazon.awssdk.services.schemas.model.GetCodeBindingSourceResponse;
import software.amazon.awssdk.services.schemas.model.GetDiscoveredSchemaRequest;
import software.amazon.awssdk.services.schemas.model.GetDiscoveredSchemaResponse;
import software.amazon.awssdk.services.schemas.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.schemas.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.schemas.model.GoneException;
import software.amazon.awssdk.services.schemas.model.InternalServerErrorException;
import software.amazon.awssdk.services.schemas.model.ListDiscoverersRequest;
import software.amazon.awssdk.services.schemas.model.ListDiscoverersResponse;
import software.amazon.awssdk.services.schemas.model.ListRegistriesRequest;
import software.amazon.awssdk.services.schemas.model.ListRegistriesResponse;
import software.amazon.awssdk.services.schemas.model.ListSchemaVersionsRequest;
import software.amazon.awssdk.services.schemas.model.ListSchemaVersionsResponse;
import software.amazon.awssdk.services.schemas.model.ListSchemasRequest;
import software.amazon.awssdk.services.schemas.model.ListSchemasResponse;
import software.amazon.awssdk.services.schemas.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.schemas.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.schemas.model.NotFoundException;
import software.amazon.awssdk.services.schemas.model.PreconditionFailedException;
import software.amazon.awssdk.services.schemas.model.PutCodeBindingRequest;
import software.amazon.awssdk.services.schemas.model.PutCodeBindingResponse;
import software.amazon.awssdk.services.schemas.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.schemas.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.schemas.model.SchemasException;
import software.amazon.awssdk.services.schemas.model.SchemasRequest;
import software.amazon.awssdk.services.schemas.model.SearchSchemasRequest;
import software.amazon.awssdk.services.schemas.model.SearchSchemasResponse;
import software.amazon.awssdk.services.schemas.model.ServiceUnavailableException;
import software.amazon.awssdk.services.schemas.model.StartDiscovererRequest;
import software.amazon.awssdk.services.schemas.model.StartDiscovererResponse;
import software.amazon.awssdk.services.schemas.model.StopDiscovererRequest;
import software.amazon.awssdk.services.schemas.model.StopDiscovererResponse;
import software.amazon.awssdk.services.schemas.model.TagResourceRequest;
import software.amazon.awssdk.services.schemas.model.TagResourceResponse;
import software.amazon.awssdk.services.schemas.model.TooManyRequestsException;
import software.amazon.awssdk.services.schemas.model.UnauthorizedException;
import software.amazon.awssdk.services.schemas.model.UntagResourceRequest;
import software.amazon.awssdk.services.schemas.model.UntagResourceResponse;
import software.amazon.awssdk.services.schemas.model.UpdateDiscovererRequest;
import software.amazon.awssdk.services.schemas.model.UpdateDiscovererResponse;
import software.amazon.awssdk.services.schemas.model.UpdateRegistryRequest;
import software.amazon.awssdk.services.schemas.model.UpdateRegistryResponse;
import software.amazon.awssdk.services.schemas.model.UpdateSchemaRequest;
import software.amazon.awssdk.services.schemas.model.UpdateSchemaResponse;
import software.amazon.awssdk.services.schemas.paginators.ListDiscoverersPublisher;
import software.amazon.awssdk.services.schemas.paginators.ListRegistriesPublisher;
import software.amazon.awssdk.services.schemas.paginators.ListSchemaVersionsPublisher;
import software.amazon.awssdk.services.schemas.paginators.ListSchemasPublisher;
import software.amazon.awssdk.services.schemas.paginators.SearchSchemasPublisher;
import software.amazon.awssdk.services.schemas.transform.CreateDiscovererRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.CreateRegistryRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.CreateSchemaRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DeleteDiscovererRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DeleteRegistryRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DeleteSchemaRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DeleteSchemaVersionRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DescribeCodeBindingRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DescribeDiscovererRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DescribeRegistryRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.DescribeSchemaRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.ExportSchemaRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.GetCodeBindingSourceRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.GetDiscoveredSchemaRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.ListDiscoverersRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.ListRegistriesRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.ListSchemaVersionsRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.ListSchemasRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.PutCodeBindingRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.SearchSchemasRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.StartDiscovererRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.StopDiscovererRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.UpdateDiscovererRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.UpdateRegistryRequestMarshaller;
import software.amazon.awssdk.services.schemas.transform.UpdateSchemaRequestMarshaller;
import software.amazon.awssdk.services.schemas.waiters.SchemasAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SchemasAsyncClient}.
 *
 * @see SchemasAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSchemasAsyncClient implements SchemasAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSchemasAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultSchemasAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a discoverer.
     * </p>
     *
     * @param createDiscovererRequest
     * @return A Java Future containing the result of the CreateDiscoverer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>ConflictException 409 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.CreateDiscoverer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/CreateDiscoverer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDiscovererResponse> createDiscoverer(CreateDiscovererRequest createDiscovererRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDiscovererRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDiscoverer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDiscovererResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDiscovererResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDiscovererResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDiscovererRequest, CreateDiscovererResponse>()
                            .withOperationName("CreateDiscoverer")
                            .withMarshaller(new CreateDiscovererRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDiscovererRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDiscovererRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDiscovererResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a registry.
     * </p>
     *
     * @param createRegistryRequest
     * @return A Java Future containing the result of the CreateRegistry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>ConflictException 409 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.CreateRegistry
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/CreateRegistry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRegistryResponse> createRegistry(CreateRegistryRequest createRegistryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRegistryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRegistry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRegistryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRegistryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRegistryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRegistryRequest, CreateRegistryResponse>()
                            .withOperationName("CreateRegistry")
                            .withMarshaller(new CreateRegistryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRegistryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createRegistryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateRegistryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a schema definition.
     * </p>
     * <note>
     * <p>
     * Inactive schemas will be deleted after two years.
     * </p>
     * </note>
     *
     * @param createSchemaRequest
     * @return A Java Future containing the result of the CreateSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.CreateSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/CreateSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSchemaResponse> createSchema(CreateSchemaRequest createSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSchemaRequest, CreateSchemaResponse>()
                            .withOperationName("CreateSchema").withMarshaller(new CreateSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createSchemaRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a discoverer.
     * </p>
     *
     * @param deleteDiscovererRequest
     * @return A Java Future containing the result of the DeleteDiscoverer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DeleteDiscoverer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DeleteDiscoverer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDiscovererResponse> deleteDiscoverer(DeleteDiscovererRequest deleteDiscovererRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDiscovererRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDiscoverer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDiscovererResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDiscovererResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDiscovererResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDiscovererRequest, DeleteDiscovererResponse>()
                            .withOperationName("DeleteDiscoverer")
                            .withMarshaller(new DeleteDiscovererRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDiscovererRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDiscovererRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDiscovererResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Registry.
     * </p>
     *
     * @param deleteRegistryRequest
     * @return A Java Future containing the result of the DeleteRegistry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DeleteRegistry
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DeleteRegistry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRegistryResponse> deleteRegistry(DeleteRegistryRequest deleteRegistryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRegistryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRegistry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRegistryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRegistryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRegistryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRegistryRequest, DeleteRegistryResponse>()
                            .withOperationName("DeleteRegistry")
                            .withMarshaller(new DeleteRegistryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRegistryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteRegistryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteRegistryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete the resource-based policy attached to the specified registry.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DeleteResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DeleteResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy")
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResourcePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteResourcePolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a schema definition.
     * </p>
     *
     * @param deleteSchemaRequest
     * @return A Java Future containing the result of the DeleteSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DeleteSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DeleteSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSchemaResponse> deleteSchema(DeleteSchemaRequest deleteSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSchemaRequest, DeleteSchemaResponse>()
                            .withOperationName("DeleteSchema").withMarshaller(new DeleteSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteSchemaRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete the schema version definition
     * </p>
     *
     * @param deleteSchemaVersionRequest
     * @return A Java Future containing the result of the DeleteSchemaVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DeleteSchemaVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DeleteSchemaVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSchemaVersionResponse> deleteSchemaVersion(
            DeleteSchemaVersionRequest deleteSchemaVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSchemaVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchemaVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSchemaVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSchemaVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSchemaVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSchemaVersionRequest, DeleteSchemaVersionResponse>()
                            .withOperationName("DeleteSchemaVersion")
                            .withMarshaller(new DeleteSchemaVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSchemaVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteSchemaVersionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteSchemaVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe the code binding URI.
     * </p>
     *
     * @param describeCodeBindingRequest
     * @return A Java Future containing the result of the DescribeCodeBinding operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DescribeCodeBinding
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DescribeCodeBinding" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCodeBindingResponse> describeCodeBinding(
            DescribeCodeBindingRequest describeCodeBindingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCodeBindingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCodeBinding");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCodeBindingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCodeBindingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCodeBindingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCodeBindingRequest, DescribeCodeBindingResponse>()
                            .withOperationName("DescribeCodeBinding")
                            .withMarshaller(new DescribeCodeBindingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeCodeBindingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeCodeBindingRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeCodeBindingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the discoverer.
     * </p>
     *
     * @param describeDiscovererRequest
     * @return A Java Future containing the result of the DescribeDiscoverer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DescribeDiscoverer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DescribeDiscoverer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDiscovererResponse> describeDiscoverer(DescribeDiscovererRequest describeDiscovererRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDiscovererRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDiscoverer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDiscovererResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDiscovererResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDiscovererResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDiscovererRequest, DescribeDiscovererResponse>()
                            .withOperationName("DescribeDiscoverer")
                            .withMarshaller(new DescribeDiscovererRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDiscovererRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDiscovererRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeDiscovererResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the registry.
     * </p>
     *
     * @param describeRegistryRequest
     * @return A Java Future containing the result of the DescribeRegistry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DescribeRegistry
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DescribeRegistry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRegistryResponse> describeRegistry(DescribeRegistryRequest describeRegistryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegistryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegistry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRegistryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRegistryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRegistryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRegistryRequest, DescribeRegistryResponse>()
                            .withOperationName("DescribeRegistry")
                            .withMarshaller(new DescribeRegistryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRegistryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeRegistryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeRegistryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the schema definition.
     * </p>
     *
     * @param describeSchemaRequest
     * @return A Java Future containing the result of the DescribeSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.DescribeSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/DescribeSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSchemaResponse> describeSchema(DescribeSchemaRequest describeSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSchemaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSchemaRequest, DescribeSchemaResponse>()
                            .withOperationName("DescribeSchema")
                            .withMarshaller(new DescribeSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeSchemaRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the ExportSchema operation asynchronously.
     *
     * @param exportSchemaRequest
     * @return A Java Future containing the result of the ExportSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ExportSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ExportSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportSchemaResponse> exportSchema(ExportSchemaRequest exportSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportSchemaRequest, ExportSchemaResponse>()
                            .withOperationName("ExportSchema").withMarshaller(new ExportSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = exportSchemaRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ExportSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the code binding source URI.
     * </p>
     *
     * @param getCodeBindingSourceRequest
     * @return A Java Future containing the result of the GetCodeBindingSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.GetCodeBindingSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/GetCodeBindingSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCodeBindingSourceResponse> getCodeBindingSource(
            GetCodeBindingSourceRequest getCodeBindingSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCodeBindingSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCodeBindingSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetCodeBindingSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCodeBindingSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCodeBindingSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCodeBindingSourceRequest, GetCodeBindingSourceResponse>()
                            .withOperationName("GetCodeBindingSource")
                            .withMarshaller(new GetCodeBindingSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCodeBindingSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCodeBindingSourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetCodeBindingSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the discovered schema that was generated based on sampled events.
     * </p>
     *
     * @param getDiscoveredSchemaRequest
     * @return A Java Future containing the result of the GetDiscoveredSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.GetDiscoveredSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/GetDiscoveredSchema" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDiscoveredSchemaResponse> getDiscoveredSchema(
            GetDiscoveredSchemaRequest getDiscoveredSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDiscoveredSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDiscoveredSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDiscoveredSchemaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDiscoveredSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDiscoveredSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDiscoveredSchemaRequest, GetDiscoveredSchemaResponse>()
                            .withOperationName("GetDiscoveredSchema")
                            .withMarshaller(new GetDiscoveredSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDiscoveredSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDiscoveredSchemaRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetDiscoveredSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource-based policy attached to a given registry.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.GetResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/GetResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy")
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourcePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResourcePolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the discoverers.
     * </p>
     *
     * @param listDiscoverersRequest
     * @return A Java Future containing the result of the ListDiscoverers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListDiscoverers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListDiscoverers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDiscoverersResponse> listDiscoverers(ListDiscoverersRequest listDiscoverersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDiscoverersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDiscoverers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDiscoverersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDiscoverersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDiscoverersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDiscoverersRequest, ListDiscoverersResponse>()
                            .withOperationName("ListDiscoverers")
                            .withMarshaller(new ListDiscoverersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDiscoverersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDiscoverersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDiscoverersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the discoverers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDiscoverers(software.amazon.awssdk.services.schemas.model.ListDiscoverersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListDiscoverersPublisher publisher = client.listDiscoverersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListDiscoverersPublisher publisher = client.listDiscoverersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.schemas.model.ListDiscoverersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.schemas.model.ListDiscoverersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDiscoverers(software.amazon.awssdk.services.schemas.model.ListDiscoverersRequest)} operation.</b>
     * </p>
     *
     * @param listDiscoverersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListDiscoverers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListDiscoverers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDiscoverersPublisher listDiscoverersPaginator(ListDiscoverersRequest listDiscoverersRequest) {
        return new ListDiscoverersPublisher(this, applyPaginatorUserAgent(listDiscoverersRequest));
    }

    /**
     * <p>
     * List the registries.
     * </p>
     *
     * @param listRegistriesRequest
     * @return A Java Future containing the result of the ListRegistries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListRegistries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListRegistries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRegistriesResponse> listRegistries(ListRegistriesRequest listRegistriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRegistriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegistries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRegistriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRegistriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRegistriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRegistriesRequest, ListRegistriesResponse>()
                            .withOperationName("ListRegistries")
                            .withMarshaller(new ListRegistriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRegistriesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRegistriesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListRegistriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the registries.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listRegistries(software.amazon.awssdk.services.schemas.model.ListRegistriesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListRegistriesPublisher publisher = client.listRegistriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListRegistriesPublisher publisher = client.listRegistriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.schemas.model.ListRegistriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.schemas.model.ListRegistriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRegistries(software.amazon.awssdk.services.schemas.model.ListRegistriesRequest)} operation.</b>
     * </p>
     *
     * @param listRegistriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListRegistries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListRegistries" target="_top">AWS API
     *      Documentation</a>
     */
    public ListRegistriesPublisher listRegistriesPaginator(ListRegistriesRequest listRegistriesRequest) {
        return new ListRegistriesPublisher(this, applyPaginatorUserAgent(listRegistriesRequest));
    }

    /**
     * <p>
     * Provides a list of the schema versions and related information.
     * </p>
     *
     * @param listSchemaVersionsRequest
     * @return A Java Future containing the result of the ListSchemaVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListSchemaVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListSchemaVersions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemaVersionsResponse> listSchemaVersions(ListSchemaVersionsRequest listSchemaVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemaVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemaVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemaVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSchemaVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemaVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemaVersionsRequest, ListSchemaVersionsResponse>()
                            .withOperationName("ListSchemaVersions")
                            .withMarshaller(new ListSchemaVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchemaVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listSchemaVersionsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListSchemaVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of the schema versions and related information.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSchemaVersions(software.amazon.awssdk.services.schemas.model.ListSchemaVersionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListSchemaVersionsPublisher publisher = client.listSchemaVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListSchemaVersionsPublisher publisher = client.listSchemaVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.schemas.model.ListSchemaVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.schemas.model.ListSchemaVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchemaVersions(software.amazon.awssdk.services.schemas.model.ListSchemaVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSchemaVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListSchemaVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListSchemaVersions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSchemaVersionsPublisher listSchemaVersionsPaginator(ListSchemaVersionsRequest listSchemaVersionsRequest) {
        return new ListSchemaVersionsPublisher(this, applyPaginatorUserAgent(listSchemaVersionsRequest));
    }

    /**
     * <p>
     * List the schemas.
     * </p>
     *
     * @param listSchemasRequest
     * @return A Java Future containing the result of the ListSchemas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListSchemas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemasResponse> listSchemas(ListSchemasRequest listSchemasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSchemasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemasRequest, ListSchemasResponse>()
                            .withOperationName("ListSchemas").withMarshaller(new ListSchemasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchemasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listSchemasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListSchemasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the schemas.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSchemas(software.amazon.awssdk.services.schemas.model.ListSchemasRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListSchemasPublisher publisher = client.listSchemasPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.ListSchemasPublisher publisher = client.listSchemasPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.schemas.model.ListSchemasResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.schemas.model.ListSchemasResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchemas(software.amazon.awssdk.services.schemas.model.ListSchemasRequest)} operation.</b>
     * </p>
     *
     * @param listSchemasRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListSchemas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSchemasPublisher listSchemasPaginator(ListSchemasRequest listSchemasRequest) {
        return new ListSchemasPublisher(this, applyPaginatorUserAgent(listSchemasRequest));
    }

    /**
     * <p>
     * Get tags for resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Put code binding URI
     * </p>
     *
     * @param putCodeBindingRequest
     * @return A Java Future containing the result of the PutCodeBinding operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>GoneException 410 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.PutCodeBinding
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/PutCodeBinding" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutCodeBindingResponse> putCodeBinding(PutCodeBindingRequest putCodeBindingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCodeBindingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCodeBinding");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutCodeBindingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutCodeBindingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutCodeBindingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutCodeBindingRequest, PutCodeBindingResponse>()
                            .withOperationName("PutCodeBinding")
                            .withMarshaller(new PutCodeBindingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putCodeBindingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putCodeBindingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutCodeBindingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The name of the policy.
     * </p>
     *
     * @param putResourcePolicyRequest
     *        The name of the policy.
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>PreconditionFailedException 412 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.PutResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/PutResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy")
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putResourcePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putResourcePolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Search the schemas
     * </p>
     *
     * @param searchSchemasRequest
     * @return A Java Future containing the result of the SearchSchemas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.SearchSchemas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/SearchSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchSchemasResponse> searchSchemas(SearchSchemasRequest searchSchemasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchSchemasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchSchemas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchSchemasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchSchemasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchSchemasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchSchemasRequest, SearchSchemasResponse>()
                            .withOperationName("SearchSchemas")
                            .withMarshaller(new SearchSchemasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchSchemasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchSchemasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SearchSchemasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Search the schemas
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #searchSchemas(software.amazon.awssdk.services.schemas.model.SearchSchemasRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.SearchSchemasPublisher publisher = client.searchSchemasPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.schemas.paginators.SearchSchemasPublisher publisher = client.searchSchemasPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.schemas.model.SearchSchemasResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.schemas.model.SearchSchemasResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchSchemas(software.amazon.awssdk.services.schemas.model.SearchSchemasRequest)} operation.</b>
     * </p>
     *
     * @param searchSchemasRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.SearchSchemas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/SearchSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    public SearchSchemasPublisher searchSchemasPaginator(SearchSchemasRequest searchSchemasRequest) {
        return new SearchSchemasPublisher(this, applyPaginatorUserAgent(searchSchemasRequest));
    }

    /**
     * <p>
     * Starts the discoverer
     * </p>
     *
     * @param startDiscovererRequest
     * @return A Java Future containing the result of the StartDiscoverer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.StartDiscoverer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/StartDiscoverer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartDiscovererResponse> startDiscoverer(StartDiscovererRequest startDiscovererRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDiscovererRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDiscoverer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDiscovererResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDiscovererResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDiscovererResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDiscovererRequest, StartDiscovererResponse>()
                            .withOperationName("StartDiscoverer")
                            .withMarshaller(new StartDiscovererRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startDiscovererRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startDiscovererRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartDiscovererResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the discoverer
     * </p>
     *
     * @param stopDiscovererRequest
     * @return A Java Future containing the result of the StopDiscoverer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.StopDiscoverer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/StopDiscoverer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopDiscovererResponse> stopDiscoverer(StopDiscovererRequest stopDiscovererRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDiscovererRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDiscoverer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopDiscovererResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopDiscovererResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopDiscovererResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopDiscovererRequest, StopDiscovererResponse>()
                            .withOperationName("StopDiscoverer")
                            .withMarshaller(new StopDiscovererRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopDiscovererRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopDiscovererRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopDiscovererResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add tags to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the discoverer
     * </p>
     *
     * @param updateDiscovererRequest
     * @return A Java Future containing the result of the UpdateDiscoverer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.UpdateDiscoverer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/UpdateDiscoverer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDiscovererResponse> updateDiscoverer(UpdateDiscovererRequest updateDiscovererRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDiscovererRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDiscoverer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDiscovererResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDiscovererResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDiscovererResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDiscovererRequest, UpdateDiscovererResponse>()
                            .withOperationName("UpdateDiscoverer")
                            .withMarshaller(new UpdateDiscovererRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDiscovererRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDiscovererRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateDiscovererResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a registry.
     * </p>
     *
     * @param updateRegistryRequest
     *        Updates the registry.
     * @return A Java Future containing the result of the UpdateRegistry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>UnauthorizedException 401 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.UpdateRegistry
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/UpdateRegistry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRegistryResponse> updateRegistry(UpdateRegistryRequest updateRegistryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRegistryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRegistry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRegistryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRegistryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRegistryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRegistryRequest, UpdateRegistryResponse>()
                            .withOperationName("UpdateRegistry")
                            .withMarshaller(new UpdateRegistryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRegistryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRegistryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateRegistryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the schema definition
     * </p>
     * <note>
     * <p>
     * Inactive schemas will be deleted after two years.
     * </p>
     * </note>
     *
     * @param updateSchemaRequest
     * @return A Java Future containing the result of the UpdateSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException 400 response</li>
     *         <li>InternalServerErrorException 500 response</li>
     *         <li>ForbiddenException 403 response</li>
     *         <li>NotFoundException 404 response</li>
     *         <li>ServiceUnavailableException 503 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SchemasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SchemasAsyncClient.UpdateSchema
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/schemas-2019-12-02/UpdateSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSchemaResponse> updateSchema(UpdateSchemaRequest updateSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "schemas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSchemaRequest, UpdateSchemaResponse>()
                            .withOperationName("UpdateSchema").withMarshaller(new UpdateSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateSchemaRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SchemasException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GoneException").exceptionBuilderSupplier(GoneException::builder)
                                .httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionFailedException")
                                .exceptionBuilderSupplier(PreconditionFailedException::builder).httpStatusCode(412).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SchemasRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public SchemasAsyncWaiter waiter() {
        return SchemasAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }
}
