/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3vectors.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The attributes of a vector to add to a vector index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutInputVector implements SdkPojo, Serializable, ToCopyableBuilder<PutInputVector.Builder, PutInputVector> {
    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("key")
            .getter(getter(PutInputVector::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("key").build()).build();

    private static final SdkField<VectorData> DATA_FIELD = SdkField.<VectorData> builder(MarshallingType.SDK_POJO)
            .memberName("data").getter(getter(PutInputVector::data)).setter(setter(Builder::data))
            .constructor(VectorData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("data").build()).build();

    private static final SdkField<Document> METADATA_FIELD = SdkField.<Document> builder(MarshallingType.DOCUMENT)
            .memberName("metadata").getter(getter(PutInputVector::metadata)).setter(setter(Builder::metadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metadata").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_FIELD, DATA_FIELD,
            METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String key;

    private final VectorData data;

    private final Document metadata;

    private PutInputVector(BuilderImpl builder) {
        this.key = builder.key;
        this.data = builder.data;
        this.metadata = builder.metadata;
    }

    /**
     * <p>
     * The name of the vector. The key uniquely identifies the vector in a vector index.
     * </p>
     * 
     * @return The name of the vector. The key uniquely identifies the vector in a vector index.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * The vector data of the vector.
     * </p>
     * <p>
     * Vector dimensions must match the dimension count that's configured for the vector index.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>cosine</code> distance metric, zero vectors (vectors containing all zeros) aren't allowed.
     * </p>
     * </li>
     * <li>
     * <p>
     * For both <code>cosine</code> and <code>euclidean</code> distance metrics, vector data must contain only valid
     * floating-point values. Invalid values such as NaN (Not a Number) or Infinity aren't allowed.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The vector data of the vector. </p>
     *         <p>
     *         Vector dimensions must match the dimension count that's configured for the vector index.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>cosine</code> distance metric, zero vectors (vectors containing all zeros) aren't allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For both <code>cosine</code> and <code>euclidean</code> distance metrics, vector data must contain only
     *         valid floating-point values. Invalid values such as NaN (Not a Number) or Infinity aren't allowed.
     *         </p>
     *         </li>
     */
    public final VectorData data() {
        return data;
    }

    /**
     * <p>
     * Metadata about the vector. All metadata entries undergo validation to ensure they meet the format requirements
     * for size and data types.
     * </p>
     * 
     * @return Metadata about the vector. All metadata entries undergo validation to ensure they meet the format
     *         requirements for size and data types.
     */
    public final Document metadata() {
        return metadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(metadata());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutInputVector)) {
            return false;
        }
        PutInputVector other = (PutInputVector) obj;
        return Objects.equals(key(), other.key()) && Objects.equals(data(), other.data())
                && Objects.equals(metadata(), other.metadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutInputVector").add("Key", key()).add("Data", data()).add("Metadata", metadata()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "key":
            return Optional.ofNullable(clazz.cast(key()));
        case "data":
            return Optional.ofNullable(clazz.cast(data()));
        case "metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("key", KEY_FIELD);
        map.put("data", DATA_FIELD);
        map.put("metadata", METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PutInputVector, T> g) {
        return obj -> g.apply((PutInputVector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PutInputVector> {
        /**
         * <p>
         * The name of the vector. The key uniquely identifies the vector in a vector index.
         * </p>
         * 
         * @param key
         *        The name of the vector. The key uniquely identifies the vector in a vector index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * The vector data of the vector.
         * </p>
         * <p>
         * Vector dimensions must match the dimension count that's configured for the vector index.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>cosine</code> distance metric, zero vectors (vectors containing all zeros) aren't allowed.
         * </p>
         * </li>
         * <li>
         * <p>
         * For both <code>cosine</code> and <code>euclidean</code> distance metrics, vector data must contain only valid
         * floating-point values. Invalid values such as NaN (Not a Number) or Infinity aren't allowed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param data
         *        The vector data of the vector. </p>
         *        <p>
         *        Vector dimensions must match the dimension count that's configured for the vector index.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>cosine</code> distance metric, zero vectors (vectors containing all zeros) aren't
         *        allowed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For both <code>cosine</code> and <code>euclidean</code> distance metrics, vector data must contain
         *        only valid floating-point values. Invalid values such as NaN (Not a Number) or Infinity aren't
         *        allowed.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(VectorData data);

        /**
         * <p>
         * The vector data of the vector.
         * </p>
         * <p>
         * Vector dimensions must match the dimension count that's configured for the vector index.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>cosine</code> distance metric, zero vectors (vectors containing all zeros) aren't allowed.
         * </p>
         * </li>
         * <li>
         * <p>
         * For both <code>cosine</code> and <code>euclidean</code> distance metrics, vector data must contain only valid
         * floating-point values. Invalid values such as NaN (Not a Number) or Infinity aren't allowed.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link VectorData.Builder} avoiding the need to
         * create one manually via {@link VectorData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VectorData.Builder#build()} is called immediately and its result
         * is passed to {@link #data(VectorData)}.
         * 
         * @param data
         *        a consumer that will call methods on {@link VectorData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #data(VectorData)
         */
        default Builder data(Consumer<VectorData.Builder> data) {
            return data(VectorData.builder().applyMutation(data).build());
        }

        /**
         * <p>
         * Metadata about the vector. All metadata entries undergo validation to ensure they meet the format
         * requirements for size and data types.
         * </p>
         * 
         * @param metadata
         *        Metadata about the vector. All metadata entries undergo validation to ensure they meet the format
         *        requirements for size and data types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(Document metadata);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private VectorData data;

        private Document metadata;

        private BuilderImpl() {
        }

        private BuilderImpl(PutInputVector model) {
            key(model.key);
            data(model.data);
            metadata(model.metadata);
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final VectorData.Builder getData() {
            return data != null ? data.toBuilder() : null;
        }

        public final void setData(VectorData.BuilderImpl data) {
            this.data = data != null ? data.build() : null;
        }

        @Override
        public final Builder data(VectorData data) {
            this.data = data;
            return this;
        }

        public final Document getMetadata() {
            return metadata;
        }

        public final void setMetadata(Document metadata) {
            this.metadata = metadata;
        }

        @Override
        public final Builder metadata(Document metadata) {
            this.metadata = metadata;
            return this;
        }

        @Override
        public PutInputVector build() {
            return new PutInputVector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
