/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3outposts.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * S3 on Outposts access points simplify managing data access at scale for shared datasets in Amazon S3 on Outposts. S3
 * on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual private
 * cloud (VPC).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointArn").getter(getter(Endpoint::endpointArn)).setter(setter(Builder::endpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointArn").build()).build();

    private static final SdkField<String> OUTPOSTS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutpostsId").getter(getter(Endpoint::outpostsId)).setter(setter(Builder::outpostsId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostsId").build()).build();

    private static final SdkField<String> CIDR_BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CidrBlock").getter(getter(Endpoint::cidrBlock)).setter(setter(Builder::cidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CidrBlock").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Endpoint::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Endpoint::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaces")
            .getter(getter(Endpoint::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_ARN_FIELD,
            OUTPOSTS_ID_FIELD, CIDR_BLOCK_FIELD, STATUS_FIELD, CREATION_TIME_FIELD, NETWORK_INTERFACES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String endpointArn;

    private final String outpostsId;

    private final String cidrBlock;

    private final String status;

    private final Instant creationTime;

    private final List<NetworkInterface> networkInterfaces;

    private Endpoint(BuilderImpl builder) {
        this.endpointArn = builder.endpointArn;
        this.outpostsId = builder.outpostsId;
        this.cidrBlock = builder.cidrBlock;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.networkInterfaces = builder.networkInterfaces;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the endpoint.
     */
    public final String endpointArn() {
        return endpointArn;
    }

    /**
     * <p>
     * The ID of the AWS Outpost.
     * </p>
     * 
     * @return The ID of the AWS Outpost.
     */
    public final String outpostsId() {
        return outpostsId;
    }

    /**
     * <p>
     * The VPC CIDR committed by this endpoint.
     * </p>
     * 
     * @return The VPC CIDR committed by this endpoint.
     */
    public final String cidrBlock() {
        return cidrBlock;
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.
     * @see EndpointStatus
     */
    public final EndpointStatus status() {
        return EndpointStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.
     * @see EndpointStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time the endpoint was created.
     * </p>
     * 
     * @return The time the endpoint was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * Returns true if the NetworkInterfaces property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network interface of the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNetworkInterfaces()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The network interface of the endpoint.
     */
    public final List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(outpostsId());
        hashCode = 31 * hashCode + Objects.hashCode(cidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(endpointArn(), other.endpointArn()) && Objects.equals(outpostsId(), other.outpostsId())
                && Objects.equals(cidrBlock(), other.cidrBlock()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationTime(), other.creationTime()) && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Endpoint").add("EndpointArn", endpointArn()).add("OutpostsId", outpostsId())
                .add("CidrBlock", cidrBlock()).add("Status", statusAsString()).add("CreationTime", creationTime())
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EndpointArn":
            return Optional.ofNullable(clazz.cast(endpointArn()));
        case "OutpostsId":
            return Optional.ofNullable(clazz.cast(outpostsId()));
        case "CidrBlock":
            return Optional.ofNullable(clazz.cast(cidrBlock()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "NetworkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the endpoint.
         * </p>
         * 
         * @param endpointArn
         *        The Amazon Resource Name (ARN) of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointArn(String endpointArn);

        /**
         * <p>
         * The ID of the AWS Outpost.
         * </p>
         * 
         * @param outpostsId
         *        The ID of the AWS Outpost.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostsId(String outpostsId);

        /**
         * <p>
         * The VPC CIDR committed by this endpoint.
         * </p>
         * 
         * @param cidrBlock
         *        The VPC CIDR committed by this endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrBlock(String cidrBlock);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param status
         *        The status of the endpoint.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param status
         *        The status of the endpoint.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder status(EndpointStatus status);

        /**
         * <p>
         * The time the endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        The time the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The network interface of the endpoint.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interface of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interface of the endpoint.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interface of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interface of the endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NetworkInterface>.Builder} avoiding the
         * need to create one manually via {@link List<NetworkInterface>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkInterface>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on {@link List<NetworkInterface>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(List<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);
    }

    static final class BuilderImpl implements Builder {
        private String endpointArn;

        private String outpostsId;

        private String cidrBlock;

        private String status;

        private Instant creationTime;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            endpointArn(model.endpointArn);
            outpostsId(model.outpostsId);
            cidrBlock(model.cidrBlock);
            status(model.status);
            creationTime(model.creationTime);
            networkInterfaces(model.networkInterfaces);
        }

        public final String getEndpointArn() {
            return endpointArn;
        }

        @Override
        public final Builder endpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
            return this;
        }

        public final void setEndpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
        }

        public final String getOutpostsId() {
            return outpostsId;
        }

        @Override
        public final Builder outpostsId(String outpostsId) {
            this.outpostsId = outpostsId;
            return this;
        }

        public final void setOutpostsId(String outpostsId) {
            this.outpostsId = outpostsId;
        }

        public final String getCidrBlock() {
            return cidrBlock;
        }

        @Override
        public final Builder cidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
            return this;
        }

        public final void setCidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(EndpointStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Collection<NetworkInterface.Builder> getNetworkInterfaces() {
            if (networkInterfaces instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaces != null ? networkInterfaces.stream().map(NetworkInterface::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copyFromBuilder(networkInterfaces);
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
