/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration parameters for a Set Object Tagging operation. Amazon S3 Batch Operations passes each
 * value through to the underlying PUT Object tagging API. For more information about the parameters for this operation,
 * see <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectPUTtagging.html">PUT Object tagging</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3SetObjectTaggingOperation implements SdkPojo, Serializable,
        ToCopyableBuilder<S3SetObjectTaggingOperation.Builder, S3SetObjectTaggingOperation> {
    private static final SdkField<List<S3Tag>> TAG_SET_FIELD = SdkField
            .<List<S3Tag>> builder(MarshallingType.LIST)
            .getter(getter(S3SetObjectTaggingOperation::tagSet))
            .setter(setter(Builder::tagSet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("TagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TAG_SET_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<S3Tag> tagSet;

    private S3SetObjectTaggingOperation(BuilderImpl builder) {
        this.tagSet = builder.tagSet;
    }

    /**
     * Returns true if the TagSet property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTagSet() {
        return tagSet != null && !(tagSet instanceof SdkAutoConstructList);
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTagSet()} to see if a value was sent in this field.
     * </p>
     * 
     * @return
     */
    public List<S3Tag> tagSet() {
        return tagSet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(tagSet());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3SetObjectTaggingOperation)) {
            return false;
        }
        S3SetObjectTaggingOperation other = (S3SetObjectTaggingOperation) obj;
        return Objects.equals(tagSet(), other.tagSet());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3SetObjectTaggingOperation").add("TagSet", tagSet()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TagSet":
            return Optional.ofNullable(clazz.cast(tagSet()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3SetObjectTaggingOperation, T> g) {
        return obj -> g.apply((S3SetObjectTaggingOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3SetObjectTaggingOperation> {
        /**
         * <p/>
         * 
         * @param tagSet
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSet(Collection<S3Tag> tagSet);

        /**
         * <p/>
         * 
         * @param tagSet
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSet(S3Tag... tagSet);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link List<S3Tag>.Builder} avoiding the need to create
         * one manually via {@link List<S3Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tagSet(List<S3Tag>)}.
         * 
         * @param tagSet
         *        a consumer that will call methods on {@link List<S3Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSet(List<S3Tag>)
         */
        Builder tagSet(Consumer<S3Tag.Builder>... tagSet);
    }

    static final class BuilderImpl implements Builder {
        private List<S3Tag> tagSet = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3SetObjectTaggingOperation model) {
            tagSet(model.tagSet);
        }

        public final Collection<S3Tag.Builder> getTagSet() {
            return tagSet != null ? tagSet.stream().map(S3Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tagSet(Collection<S3Tag> tagSet) {
            this.tagSet = S3TagSetCopier.copy(tagSet);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSet(S3Tag... tagSet) {
            tagSet(Arrays.asList(tagSet));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSet(Consumer<S3Tag.Builder>... tagSet) {
            tagSet(Stream.of(tagSet).map(c -> S3Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTagSet(Collection<S3Tag.BuilderImpl> tagSet) {
            this.tagSet = S3TagSetCopier.copyFromBuilder(tagSet);
        }

        @Override
        public S3SetObjectTaggingOperation build() {
            return new S3SetObjectTaggingOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
