/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.Response;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.xml.AwsXmlProtocolFactory;
import software.amazon.awssdk.protocols.xml.XmlOperationMetadata;
import software.amazon.awssdk.services.s3control.model.BadRequestException;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.CreateJobRequest;
import software.amazon.awssdk.services.s3control.model.CreateJobResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.DescribeJobRequest;
import software.amazon.awssdk.services.s3control.model.DescribeJobResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.IdempotencyException;
import software.amazon.awssdk.services.s3control.model.InternalServiceException;
import software.amazon.awssdk.services.s3control.model.InvalidNextTokenException;
import software.amazon.awssdk.services.s3control.model.InvalidRequestException;
import software.amazon.awssdk.services.s3control.model.JobStatusException;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse;
import software.amazon.awssdk.services.s3control.model.ListJobsRequest;
import software.amazon.awssdk.services.s3control.model.ListJobsResponse;
import software.amazon.awssdk.services.s3control.model.NoSuchPublicAccessBlockConfigurationException;
import software.amazon.awssdk.services.s3control.model.NotFoundException;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.S3ControlException;
import software.amazon.awssdk.services.s3control.model.S3ControlRequest;
import software.amazon.awssdk.services.s3control.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3control.model.TooManyTagsException;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityResponse;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusResponse;
import software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable;
import software.amazon.awssdk.services.s3control.paginators.ListJobsIterable;
import software.amazon.awssdk.services.s3control.transform.CreateAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeletePublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyStatusRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetPublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListAccessPointsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutPublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.UpdateJobPriorityRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.UpdateJobStatusRequestMarshaller;

/**
 * Internal implementation of {@link S3ControlClient}.
 *
 * @see S3ControlClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3ControlClient implements S3ControlClient {
    private final SyncClientHandler clientHandler;

    private final AwsXmlProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3ControlClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an access point and associates it with the specified bucket.
     * </p>
     *
     * @param createAccessPointRequest
     * @return Result of the CreateAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.CreateAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAccessPointResponse createAccessPoint(CreateAccessPointRequest createAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<CreateAccessPointResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                CreateAccessPointResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<CreateAccessPointRequest, CreateAccessPointResponse>()
                .withOperationName("CreateAccessPoint").withCombinedResponseHandler(responseHandler)
                .withInput(createAccessPointRequest).withMarshaller(new CreateAccessPointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * You can use Amazon S3 Batch Operations to perform large-scale Batch Operations on Amazon S3 objects. Amazon S3
     * Batch Operations can execute a single operation or action on lists of Amazon S3 objects that you specify. For
     * more information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">Amazon S3
     * Batch Operations</a> in the Amazon Simple Storage Service Developer Guide.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createJobRequest
     * @return Result of the CreateJob operation returned by the service.
     * @throws TooManyRequestsException
     * @throws BadRequestException
     * @throws IdempotencyException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws TooManyRequestsException, BadRequestException,
            IdempotencyException, InternalServiceException, AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<CreateJobResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                CreateJobResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                .withOperationName("CreateJob").withCombinedResponseHandler(responseHandler).withInput(createJobRequest)
                .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified access point.
     * </p>
     *
     * @param deleteAccessPointRequest
     * @return Result of the DeleteAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAccessPointResponse deleteAccessPoint(DeleteAccessPointRequest deleteAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<DeleteAccessPointResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                DeleteAccessPointResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<DeleteAccessPointRequest, DeleteAccessPointResponse>()
                .withOperationName("DeleteAccessPoint").withCombinedResponseHandler(responseHandler)
                .withInput(deleteAccessPointRequest).withMarshaller(new DeleteAccessPointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the access point policy for the specified access point.
     * </p>
     *
     * @param deleteAccessPointPolicyRequest
     * @return Result of the DeleteAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAccessPointPolicyResponse deleteAccessPointPolicy(DeleteAccessPointPolicyRequest deleteAccessPointPolicyRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<DeleteAccessPointPolicyResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(DeleteAccessPointPolicyResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<DeleteAccessPointPolicyRequest, DeleteAccessPointPolicyResponse>()
                .withOperationName("DeleteAccessPointPolicy").withCombinedResponseHandler(responseHandler)
                .withInput(deleteAccessPointPolicyRequest)
                .withMarshaller(new DeleteAccessPointPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the entire tag set from the specified Amazon S3 Batch Operations job. To use this operation, you must
     * have permission to perform the <code>s3:DeleteJobTagging</code> action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html#batch-ops-job-tags">Using Job
     * Tags</a> in the Amazon Simple Storage Service Developer Guide.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetJobTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>PutJobTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteJobTaggingRequest
     * @return Result of the DeleteJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteJobTaggingResponse deleteJobTagging(DeleteJobTaggingRequest deleteJobTaggingRequest)
            throws InternalServiceException, TooManyRequestsException, NotFoundException, AwsServiceException,
            SdkClientException, S3ControlException {

        HttpResponseHandler<Response<DeleteJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                DeleteJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<DeleteJobTaggingRequest, DeleteJobTaggingResponse>()
                .withOperationName("DeleteJobTagging").withCombinedResponseHandler(responseHandler)
                .withInput(deleteJobTaggingRequest).withMarshaller(new DeleteJobTaggingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param deletePublicAccessBlockRequest
     * @return Result of the DeletePublicAccessBlock operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeletePublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeletePublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePublicAccessBlockResponse deletePublicAccessBlock(DeletePublicAccessBlockRequest deletePublicAccessBlockRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<DeletePublicAccessBlockResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(DeletePublicAccessBlockResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<DeletePublicAccessBlockRequest, DeletePublicAccessBlockResponse>()
                .withOperationName("DeletePublicAccessBlock").withCombinedResponseHandler(responseHandler)
                .withInput(deletePublicAccessBlockRequest)
                .withMarshaller(new DeletePublicAccessBlockRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the configuration parameters and status for a Batch Operations job. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">Amazon S3 Batch Operations</a> in
     * the Amazon Simple Storage Service Developer Guide.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeJobRequest
     * @return Result of the DescribeJob operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobResponse describeJob(DescribeJobRequest describeJobRequest) throws BadRequestException,
            TooManyRequestsException, NotFoundException, InternalServiceException, AwsServiceException, SdkClientException,
            S3ControlException {

        HttpResponseHandler<Response<DescribeJobResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                DescribeJobResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                .withOperationName("DescribeJob").withCombinedResponseHandler(responseHandler).withInput(describeJobRequest)
                .withMarshaller(new DescribeJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns configuration information about the specified access point.
     * </p>
     *
     * @param getAccessPointRequest
     * @return Result of the GetAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAccessPointResponse getAccessPoint(GetAccessPointRequest getAccessPointRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {

        HttpResponseHandler<Response<GetAccessPointResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                GetAccessPointResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<GetAccessPointRequest, GetAccessPointResponse>()
                .withOperationName("GetAccessPoint").withCombinedResponseHandler(responseHandler)
                .withInput(getAccessPointRequest).withMarshaller(new GetAccessPointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the access point policy associated with the specified access point.
     * </p>
     *
     * @param getAccessPointPolicyRequest
     * @return Result of the GetAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAccessPointPolicyResponse getAccessPointPolicy(GetAccessPointPolicyRequest getAccessPointPolicyRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<GetAccessPointPolicyResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(GetAccessPointPolicyResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<GetAccessPointPolicyRequest, GetAccessPointPolicyResponse>()
                .withOperationName("GetAccessPointPolicy").withCombinedResponseHandler(responseHandler)
                .withInput(getAccessPointPolicyRequest)
                .withMarshaller(new GetAccessPointPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Indicates whether the specified access point currently has a policy that allows public access. For more
     * information about public access through access points, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html">Managing Data Access with Amazon S3
     * Access Points</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     *
     * @param getAccessPointPolicyStatusRequest
     * @return Result of the GetAccessPointPolicyStatus operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPointPolicyStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAccessPointPolicyStatusResponse getAccessPointPolicyStatus(
            GetAccessPointPolicyStatusRequest getAccessPointPolicyStatusRequest) throws AwsServiceException, SdkClientException,
            S3ControlException {

        HttpResponseHandler<Response<GetAccessPointPolicyStatusResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(GetAccessPointPolicyStatusResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler
                .execute(new ClientExecutionParams<GetAccessPointPolicyStatusRequest, GetAccessPointPolicyStatusResponse>()
                        .withOperationName("GetAccessPointPolicyStatus").withCombinedResponseHandler(responseHandler)
                        .withInput(getAccessPointPolicyStatusRequest)
                        .withMarshaller(new GetAccessPointPolicyStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the tags on an Amazon S3 Batch Operations job. To use this operation, you must have permission to perform
     * the <code>s3:GetJobTagging</code> action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html#batch-ops-job-tags">Using Job
     * Tags</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>PutJobTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteJobTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getJobTaggingRequest
     * @return Result of the GetJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobTaggingResponse getJobTagging(GetJobTaggingRequest getJobTaggingRequest) throws InternalServiceException,
            TooManyRequestsException, NotFoundException, AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<GetJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                GetJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<GetJobTaggingRequest, GetJobTaggingResponse>()
                .withOperationName("GetJobTagging").withCombinedResponseHandler(responseHandler).withInput(getJobTaggingRequest)
                .withMarshaller(new GetJobTaggingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param getPublicAccessBlockRequest
     * @return Result of the GetPublicAccessBlock operation returned by the service.
     * @throws NoSuchPublicAccessBlockConfigurationException
     *         Amazon S3 throws this exception if you make a <code>GetPublicAccessBlock</code> request against an
     *         account that doesn't have a <code>PublicAccessBlockConfiguration</code> set.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetPublicAccessBlockResponse getPublicAccessBlock(GetPublicAccessBlockRequest getPublicAccessBlockRequest)
            throws NoSuchPublicAccessBlockConfigurationException, AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<GetPublicAccessBlockResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(GetPublicAccessBlockResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<GetPublicAccessBlockRequest, GetPublicAccessBlockResponse>()
                .withOperationName("GetPublicAccessBlock").withCombinedResponseHandler(responseHandler)
                .withInput(getPublicAccessBlockRequest)
                .withMarshaller(new GetPublicAccessBlockRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1,000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), the response will include a continuation token that you can use to
     * list the additional access points.
     * </p>
     *
     * @param listAccessPointsRequest
     * @return Result of the ListAccessPoints operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAccessPointsResponse listAccessPoints(ListAccessPointsRequest listAccessPointsRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {

        HttpResponseHandler<Response<ListAccessPointsResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                ListAccessPointsResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<ListAccessPointsRequest, ListAccessPointsResponse>()
                .withOperationName("ListAccessPoints").withCombinedResponseHandler(responseHandler)
                .withInput(listAccessPointsRequest).withMarshaller(new ListAccessPointsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1,000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), the response will include a continuation token that you can use to
     * list the additional access points.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client.listAccessPointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client
     *             .listAccessPointsPaginator(request);
     *     for (software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client.listAccessPointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation.</b>
     * </p>
     *
     * @param listAccessPointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAccessPointsIterable listAccessPointsPaginator(ListAccessPointsRequest listAccessPointsRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        return new ListAccessPointsIterable(this, applyPaginatorUserAgent(listAccessPointsRequest));
    }

    /**
     * <p>
     * Lists current Amazon S3 Batch Operations jobs and jobs that have ended within the last 30 days for the AWS
     * account making the request. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">Amazon S3 Batch Operations</a> in
     * the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <p/>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws InvalidRequestException
     * @throws InternalServiceException
     * @throws InvalidNextTokenException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws InvalidRequestException, InternalServiceException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<ListJobsResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                ListJobsResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                .withCombinedResponseHandler(responseHandler).withInput(listJobsRequest)
                .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists current Amazon S3 Batch Operations jobs and jobs that have ended within the last 30 days for the AWS
     * account making the request. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">Amazon S3 Batch Operations</a> in
     * the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <p/>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.s3control.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     * @throws InternalServiceException
     * @throws InvalidNextTokenException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws InvalidRequestException,
            InternalServiceException, InvalidNextTokenException, AwsServiceException, SdkClientException, S3ControlException {
        return new ListJobsIterable(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Associates an access policy with the specified access point. Each access point can have only one policy, so a
     * request made to this API replaces any existing policy associated with the specified access point.
     * </p>
     *
     * @param putAccessPointPolicyRequest
     * @return Result of the PutAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutAccessPointPolicyResponse putAccessPointPolicy(PutAccessPointPolicyRequest putAccessPointPolicyRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<PutAccessPointPolicyResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(PutAccessPointPolicyResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<PutAccessPointPolicyRequest, PutAccessPointPolicyResponse>()
                .withOperationName("PutAccessPointPolicy").withCombinedResponseHandler(responseHandler)
                .withInput(putAccessPointPolicyRequest)
                .withMarshaller(new PutAccessPointPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Set the supplied tag-set on an Amazon S3 Batch Operations job.
     * </p>
     * <p>
     * A tag is a key-value pair. You can associate Amazon S3 Batch Operations tags with any job by sending a PUT
     * request against the tagging subresource that is associated with the job. To modify the existing tag set, you can
     * either replace the existing tag set entirely, or make changes within the existing tag set by retrieving the
     * existing tag set using <a>GetJobTagging</a>, modify that tag set, and use this API action to replace the tag set
     * with the one you have modified.. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html#batch-ops-job-tags">Using Job
     * Tags</a> in the Amazon Simple Storage Service Developer Guide.
     * </p>
     * <p/>
     * <note>
     * <ul>
     * <li>
     * <p>
     * If you send this request with an empty tag set, Amazon S3 deletes the existing tag set on the Batch Operations
     * job. If you use this method, you will be charged for a Tier 1 Request (PUT). For more information, see <a
     * href="http://aws.amazon.com/s3/pricing/">Amazon S3 pricing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For deleting existing tags for your batch operations job, <a>DeleteJobTagging</a> request is preferred because it
     * achieves the same result without incurring charges.
     * </p>
     * </li>
     * <li>
     * <p>
     * A few things to consider about using tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon S3 limits the maximum number of tags to 50 tags per job.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can associate up to 50 tags with a job as long as they have unique tag keys.
     * </p>
     * </li>
     * <li>
     * <p>
     * A tag key can be up to 128 Unicode characters in length, and tag values can be up to 256 Unicode characters in
     * length.
     * </p>
     * </li>
     * <li>
     * <p>
     * The key and values are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * For tagging-related restrictions related to characters and encodings, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/allocation-tag-restrictions.html">User-Defined
     * Tag Restrictions</a>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </note>
     * <p/>
     * <p>
     * To use this operation, you must have permission to perform the <code>s3:PutJobTagging</code> action.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetJobTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteJobTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putJobTaggingRequest
     * @return Result of the PutJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutJobTaggingResponse putJobTagging(PutJobTaggingRequest putJobTaggingRequest) throws InternalServiceException,
            TooManyRequestsException, NotFoundException, TooManyTagsException, AwsServiceException, SdkClientException,
            S3ControlException {

        HttpResponseHandler<Response<PutJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                PutJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<PutJobTaggingRequest, PutJobTaggingResponse>()
                .withOperationName("PutJobTagging").withCombinedResponseHandler(responseHandler).withInput(putJobTaggingRequest)
                .withMarshaller(new PutJobTaggingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or modifies the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param putPublicAccessBlockRequest
     * @return Result of the PutPublicAccessBlock operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutPublicAccessBlockResponse putPublicAccessBlock(PutPublicAccessBlockRequest putPublicAccessBlockRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<PutPublicAccessBlockResponse>> responseHandler = protocolFactory
                .createCombinedResponseHandler(PutPublicAccessBlockResponse::builder,
                        new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<PutPublicAccessBlockRequest, PutPublicAccessBlockResponse>()
                .withOperationName("PutPublicAccessBlock").withCombinedResponseHandler(responseHandler)
                .withInput(putPublicAccessBlockRequest)
                .withMarshaller(new PutPublicAccessBlockRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an existing Amazon S3 Batch Operations job's priority. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">Amazon S3 Batch Operations</a> in
     * the Amazon Simple Storage Service Developer Guide.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateJobPriorityRequest
     * @return Result of the UpdateJobPriority operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.UpdateJobPriority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobPriority" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateJobPriorityResponse updateJobPriority(UpdateJobPriorityRequest updateJobPriorityRequest)
            throws BadRequestException, TooManyRequestsException, NotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, S3ControlException {

        HttpResponseHandler<Response<UpdateJobPriorityResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                UpdateJobPriorityResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<UpdateJobPriorityRequest, UpdateJobPriorityResponse>()
                .withOperationName("UpdateJobPriority").withCombinedResponseHandler(responseHandler)
                .withInput(updateJobPriorityRequest).withMarshaller(new UpdateJobPriorityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the status for the specified job. Use this operation to confirm that you want to run a job or to cancel
     * an existing job. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">Amazon S3 Batch Operations</a> in
     * the Amazon Simple Storage Service Developer Guide.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateJobStatusRequest
     * @return Result of the UpdateJobStatus operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws JobStatusException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.UpdateJobStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateJobStatusResponse updateJobStatus(UpdateJobStatusRequest updateJobStatusRequest) throws BadRequestException,
            TooManyRequestsException, NotFoundException, JobStatusException, InternalServiceException, AwsServiceException,
            SdkClientException, S3ControlException {

        HttpResponseHandler<Response<UpdateJobStatusResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                UpdateJobStatusResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

        return clientHandler.execute(new ClientExecutionParams<UpdateJobStatusRequest, UpdateJobStatusResponse>()
                .withOperationName("UpdateJobStatus").withCombinedResponseHandler(responseHandler)
                .withInput(updateJobStatusRequest).withMarshaller(new UpdateJobStatusRequestMarshaller(protocolFactory)));
    }

    private AwsXmlProtocolFactory init() {
        return AwsXmlProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("JobStatusException")
                                .exceptionBuilderSupplier(JobStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotencyException")
                                .exceptionBuilderSupplier(IdempotencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchPublicAccessBlockConfiguration")
                                .exceptionBuilderSupplier(NoSuchPublicAccessBlockConfigurationException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(S3ControlException::builder).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends S3ControlRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
