/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.endpoints.internal;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.s3.endpoints.S3EndpointParams;
import software.amazon.awssdk.services.s3.endpoints.S3EndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.MapUtils;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultS3EndpointProvider implements S3EndpointProvider {
    private static final EndpointRuleset ENDPOINT_RULE_SET = ruleSet();

    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(S3EndpointParams endpointParams) {
        Value res = new DefaultRuleEngine().evaluate(ENDPOINT_RULE_SET, toIdentifierValueMap(endpointParams));
        try {
            return CompletableFuture.completedFuture(AwsEndpointProviderUtils.valueAsEndpointOrThrow(res));
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static Map<Identifier, Value> toIdentifierValueMap(S3EndpointParams params) {
        Map<Identifier, Value> paramsMap = new HashMap<>();
        if (params.bucket() != null) {
            paramsMap.put(Identifier.of("Bucket"), Value.fromStr(params.bucket()));
        }
        if (params.region() != null) {
            paramsMap.put(Identifier.of("Region"), Value.fromStr(params.region().id()));
        }
        if (params.useFips() != null) {
            paramsMap.put(Identifier.of("UseFIPS"), Value.fromBool(params.useFips()));
        }
        if (params.useDualStack() != null) {
            paramsMap.put(Identifier.of("UseDualStack"), Value.fromBool(params.useDualStack()));
        }
        if (params.endpoint() != null) {
            paramsMap.put(Identifier.of("Endpoint"), Value.fromStr(params.endpoint()));
        }
        if (params.forcePathStyle() != null) {
            paramsMap.put(Identifier.of("ForcePathStyle"), Value.fromBool(params.forcePathStyle()));
        }
        if (params.accelerate() != null) {
            paramsMap.put(Identifier.of("Accelerate"), Value.fromBool(params.accelerate()));
        }
        if (params.useGlobalEndpoint() != null) {
            paramsMap.put(Identifier.of("UseGlobalEndpoint"), Value.fromBool(params.useGlobalEndpoint()));
        }
        if (params.useObjectLambdaEndpoint() != null) {
            paramsMap.put(Identifier.of("UseObjectLambdaEndpoint"), Value.fromBool(params.useObjectLambdaEndpoint()));
        }
        if (params.disableAccessPoints() != null) {
            paramsMap.put(Identifier.of("DisableAccessPoints"), Value.fromBool(params.disableAccessPoints()));
        }
        if (params.disableMultiRegionAccessPoints() != null) {
            paramsMap.put(Identifier.of("DisableMultiRegionAccessPoints"),
                    Value.fromBool(params.disableMultiRegionAccessPoints()));
        }
        if (params.useArnRegion() != null) {
            paramsMap.put(Identifier.of("UseArnRegion"), Value.fromBool(params.useArnRegion()));
        }
        return paramsMap;
    }

    private static Rule endpointRule_8() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .error("Expected a endpoint to be specified but no endpoint was found");
    }

    private static Rule endpointRule_9() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.ec2.{url#authority}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-outposts"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_7() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("regionPrefix")), Expr.of("beta"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_8(), endpointRule_9()));
    }

    private static Rule endpointRule_10() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.ec2.s3-outposts.{Region}.{regionPartition#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3-outposts"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_6() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("hardwareType")), Expr.of("e"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_7(), endpointRule_10()));
    }

    private static Rule endpointRule_13() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .error("Expected a endpoint to be specified but no endpoint was found");
    }

    private static Rule endpointRule_14() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.op-{outpostId}.{url#authority}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-outposts"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_12() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("regionPrefix")), Expr.of("beta"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_13(), endpointRule_14()));
    }

    private static Rule endpointRule_15() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.op-{outpostId}.s3-outposts.{Region}.{regionPartition#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3-outposts"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_11() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("hardwareType")), Expr.of("o"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_12(), endpointRule_15()));
    }

    private static Rule endpointRule_16() {
        return Rule.builder().error("Unrecognized hardware type: \"Expected hardware type o or e but got {hardwareType}\"");
    }

    private static Rule endpointRule_5() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_6(), endpointRule_11(), endpointRule_16()));
    }

    private static Rule endpointRule_4() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("outpostId")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_5()));
    }

    private static Rule endpointRule_17() {
        return Rule.builder().error("Invalid ARN: The outpost Id must only contain a-z, A-Z, 0-9 and `-`.");
    }

    private static Rule endpointRule_3() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("substring")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(49), Expr.of(50),
                                                Expr.of(true))).build().validate()).result("hardwareType").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("substring")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(8), Expr.of(12),
                                                Expr.of(true))).build().validate()).result("regionPrefix").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("substring")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(0), Expr.of(7),
                                                Expr.of(true))).build().validate()).result("abbaSuffix").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("substring")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(32), Expr.of(49),
                                                Expr.of(true))).build().validate()).result("outpostId").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("regionPartition").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("abbaSuffix")), Expr.of("--op-s3"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_4(), endpointRule_17()));
    }

    private static Rule endpointRule_19() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("isSet")
                                                .argv(Arrays.asList(FnNode.builder().fn("parseURL")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                                        .validate())).build().validate())).build().validate()).build())
                .error("Custom endpoint `{Endpoint}` was not a valid URI");
    }

    private static Rule endpointRule_23() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.parseArn").argv(Arrays.asList(Expr.ref(Identifier.of("Bucket"))))
                                        .build().validate()).build())
                .error("Path-style addressing cannot be used with ARN buckets");
    }

    private static Rule endpointRule_25() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .error("Cannot set dual-stack in combination with a custom endpoint.");
    }

    private static Rule endpointRule_33() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.us-east-1.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_34() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.us-east-1.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_36() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3.dualstack.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_35() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_36()));
    }

    private static Rule endpointRule_37() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_38() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_39() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_41() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_42() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_40() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_41(), endpointRule_42()));
    }

    private static Rule endpointRule_43() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_44() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_45() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_47() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_48() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_46() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_47(), endpointRule_48()));
    }

    private static Rule endpointRule_49() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_32() {
        return Rule.builder().treeRule(
                Arrays.asList(endpointRule_33(), endpointRule_34(), endpointRule_35(), endpointRule_37(), endpointRule_38(),
                        endpointRule_39(), endpointRule_40(), endpointRule_43(), endpointRule_44(), endpointRule_45(),
                        endpointRule_46(), endpointRule_49()));
    }

    private static Rule endpointRule_31() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_32()));
    }

    private static Rule endpointRule_50() {
        return Rule.builder().error("Path-style addressing cannot be used with FIPS");
    }

    private static Rule endpointRule_30() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_31(), endpointRule_50()));
    }

    private static Rule endpointRule_29() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_30()));
    }

    private static Rule endpointRule_51() {
        return Rule.builder().error("Path-style addressing cannot be used with S3 Accelerate");
    }

    private static Rule endpointRule_28() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_29(), endpointRule_51()));
    }

    private static Rule endpointRule_27() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_28()));
    }

    private static Rule endpointRule_52() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_26() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_27(), endpointRule_52()));
    }

    private static Rule endpointRule_24() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("uriEncode").argv(Arrays.asList(Expr.ref(Identifier.of("Bucket"))))
                                        .build().validate()).result("uri_encoded_bucket").build())
                .treeRule(Arrays.asList(endpointRule_25(), endpointRule_26()));
    }

    private static Rule endpointRule_22() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_23(), endpointRule_24()));
    }

    private static Rule endpointRule_21() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("ForcePathStyle"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("ForcePathStyle")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_22()));
    }

    private static Rule endpointRule_58() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-cn"))).build()
                                        .validate()).build()).error("Partition does not support FIPS");
    }

    private static Rule endpointRule_60() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).error("Accelerate cannot be used with FIPS");
    }

    private static Rule endpointRule_62() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-cn"))).build()
                                        .validate()).build()).error("S3 Accelerate cannot be used in this region");
    }

    private static Rule endpointRule_64() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Host override cannot be combined with Dualstack, FIPS, or S3 Accelerate");
    }

    private static Rule endpointRule_66() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Host override cannot be combined with Dualstack, FIPS, or S3 Accelerate");
    }

    private static Rule endpointRule_68() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Host override cannot be combined with Dualstack, FIPS, or S3 Accelerate");
    }

    private static Rule endpointRule_70() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-fips.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_71() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-fips.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_73() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.s3-fips.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_72() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_73()));
    }

    private static Rule endpointRule_74() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-fips.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_75() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-fips.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_76() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-fips.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_78() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.s3-fips.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_77() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_78()));
    }

    private static Rule endpointRule_79() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-fips.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_80() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_81() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_83() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.s3-accelerate.dualstack.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_82() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_83()));
    }

    private static Rule endpointRule_84() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.dualstack.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_85() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_86() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_88() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.s3.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_87() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_88()));
    }

    private static Rule endpointRule_89() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_90() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(true))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{Bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_91() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{Bucket}.{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_92() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(true))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{Bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_93() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{Bucket}.{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_95() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{Bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_96() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{Bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_94() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(true))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_95(), endpointRule_96()));
    }

    private static Rule endpointRule_98() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{Bucket}.{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_99() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{Bucket}.{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_97() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_98(), endpointRule_99()));
    }

    private static Rule endpointRule_100() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(true))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{Bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_101() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("isIp")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{Bucket}.{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_102() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_103() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_105() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_106() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.s3-accelerate.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_104() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_105(), endpointRule_106()));
    }

    private static Rule endpointRule_107() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3-accelerate.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_108() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_109() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_111() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_112() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{Bucket}.s3.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_110() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_111(), endpointRule_112()));
    }

    private static Rule endpointRule_113() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{Bucket}.s3.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_69() {
        return Rule.builder().treeRule(
                Arrays.asList(endpointRule_70(), endpointRule_71(), endpointRule_72(), endpointRule_74(), endpointRule_75(),
                        endpointRule_76(), endpointRule_77(), endpointRule_79(), endpointRule_80(), endpointRule_81(),
                        endpointRule_82(), endpointRule_84(), endpointRule_85(), endpointRule_86(), endpointRule_87(),
                        endpointRule_89(), endpointRule_90(), endpointRule_91(), endpointRule_92(), endpointRule_93(),
                        endpointRule_94(), endpointRule_97(), endpointRule_100(), endpointRule_101(), endpointRule_102(),
                        endpointRule_103(), endpointRule_104(), endpointRule_107(), endpointRule_108(), endpointRule_109(),
                        endpointRule_110(), endpointRule_113()));
    }

    private static Rule endpointRule_67() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_68(), endpointRule_69()));
    }

    private static Rule endpointRule_65() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_66(), endpointRule_67()));
    }

    private static Rule endpointRule_63() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_64(), endpointRule_65()));
    }

    private static Rule endpointRule_61() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_62(), endpointRule_63()));
    }

    private static Rule endpointRule_59() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_60(), endpointRule_61()));
    }

    private static Rule endpointRule_57() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_58(), endpointRule_59()));
    }

    private static Rule endpointRule_56() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_57()));
    }

    private static Rule endpointRule_114() {
        return Rule.builder().error("Invalid region: region was not a valid DNS name.");
    }

    private static Rule endpointRule_55() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_56(), endpointRule_114()));
    }

    private static Rule endpointRule_54() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_55()));
    }

    private static Rule endpointRule_115() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_53() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.isVirtualHostableS3Bucket")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_54(), endpointRule_115()));
    }

    private static Rule endpointRule_120() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{Bucket}.{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_119() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_120()));
    }

    private static Rule endpointRule_121() {
        return Rule.builder().error("Invalid region: region was not a valid DNS name.");
    }

    private static Rule endpointRule_118() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_119(), endpointRule_121()));
    }

    private static Rule endpointRule_117() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_118()));
    }

    private static Rule endpointRule_122() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_116() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("url")), Expr.of("scheme")))
                                                        .build().validate(), Expr.of("http"))).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.isVirtualHostableS3Bucket")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(true))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_117(), endpointRule_122()));
    }

    private static Rule endpointRule_131() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Object Lambda does not support Dual-stack");
    }

    private static Rule endpointRule_133() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Object Lambda does not support S3 Accelerate");
    }

    private static Rule endpointRule_137() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("DisableAccessPoints")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("DisableAccessPoints")), Expr.of(true)))
                                        .build().validate()).build()).error("Access points are not supported for this operation");
    }

    private static Rule endpointRule_141() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("UseArnRegion"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseArnRegion")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                        Expr.of("region"))).build().validate(),
                                                        Expr.of("{Region}"))).build().validate())).build().validate()).build())
                .error("Invalid configuration: region from ARN `{bucketArn#region}` does not match client region `{Region}` and UseArnRegion is `false`");
    }

    private static Rule endpointRule_151() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("accountId"))).build().validate(), Expr.of(""))).build()
                                        .validate()).build()).error("Invalid ARN: Missing account id");
    }

    private static Rule endpointRule_157() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketPartition")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-cn"))).build()
                                        .validate()).build()).error("Partition does not support FIPS");
    }

    private static Rule endpointRule_159() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{accessPointName}-{bucketArn#accountId}.{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-object-lambda"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{bucketArn#region}")))))).build());
    }

    private static Rule endpointRule_160() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.s3-object-lambda-fips.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-object-lambda"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{bucketArn#region}")))))).build());
    }

    private static Rule endpointRule_161() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.s3-object-lambda.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-object-lambda"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{bucketArn#region}")))))).build());
    }

    private static Rule endpointRule_158() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_159(), endpointRule_160(), endpointRule_161()));
    }

    private static Rule endpointRule_156() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_157(), endpointRule_158()));
    }

    private static Rule endpointRule_155() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("accessPointName")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_156()));
    }

    private static Rule endpointRule_162() {
        return Rule.builder().error(
                "Invalid ARN: The access point name may only contain a-z, A-Z, 0-9 and `-`. Found: `{accessPointName}`");
    }

    private static Rule endpointRule_154() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_155(), endpointRule_162()));
    }

    private static Rule endpointRule_153() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("accountId"))).build().validate(), Expr.of(false)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_154()));
    }

    private static Rule endpointRule_163() {
        return Rule.builder().error(
                "Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `{bucketArn#accountId}`");
    }

    private static Rule endpointRule_152() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_153(), endpointRule_163()));
    }

    private static Rule endpointRule_150() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_151(), endpointRule_152()));
    }

    private static Rule endpointRule_149() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("region"))).build().validate(), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_150()));
    }

    private static Rule endpointRule_164() {
        return Rule.builder().error("Invalid region in ARN: `{bucketArn#region}` (invalid DNS name)");
    }

    private static Rule endpointRule_148() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_149(), endpointRule_164()));
    }

    private static Rule endpointRule_147() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketPartition")),
                                                                Expr.of("name"))).build().validate(),
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate())).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_148()));
    }

    private static Rule endpointRule_165() {
        return Rule.builder().error(
                "Client was configured for partition `{partitionResult#name}` but ARN (`{Bucket}`) has `{bucketPartition#name}`");
    }

    private static Rule endpointRule_146() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_147(), endpointRule_165()));
    }

    private static Rule endpointRule_145() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_146()));
    }

    private static Rule endpointRule_166() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_144() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_145(), endpointRule_166()));
    }

    private static Rule endpointRule_143() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("aws.partition")
                                        .argv(Arrays.asList(FnNode.builder().fn("getAttr")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("region")))
                                                .build().validate())).build().validate()).result("bucketPartition").build())
                .treeRule(Arrays.asList(endpointRule_144()));
    }

    private static Rule endpointRule_167() {
        return Rule.builder().error("Could not load partition for ARN region `{bucketArn#region}`");
    }

    private static Rule endpointRule_142() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_143(), endpointRule_167()));
    }

    private static Rule endpointRule_140() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_141(), endpointRule_142()));
    }

    private static Rule endpointRule_139() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("isSet")
                                                .argv(Arrays.asList(FnNode
                                                        .builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("resourceId[2]"))).build().validate())).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_140()));
    }

    private static Rule endpointRule_168() {
        return Rule.builder().error("Invalid ARN: The ARN may only contain a single resource component after `accesspoint`.");
    }

    private static Rule endpointRule_138() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_139(), endpointRule_168()));
    }

    private static Rule endpointRule_136() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_137(), endpointRule_138()));
    }

    private static Rule endpointRule_135() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                        Expr.of("region"))).build().validate(), Expr.of("")))
                                                .build().validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_136()));
    }

    private static Rule endpointRule_169() {
        return Rule.builder().error("Invalid ARN: bucket ARN is missing a region");
    }

    private static Rule endpointRule_134() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_135(), endpointRule_169()));
    }

    private static Rule endpointRule_132() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_133(), endpointRule_134()));
    }

    private static Rule endpointRule_130() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_131(), endpointRule_132()));
    }

    private static Rule endpointRule_129() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("resourceId[1]")))
                                        .build().validate()).result("accessPointName").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("accessPointName")), Expr.of("")))
                                                .build().validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_130()));
    }

    private static Rule endpointRule_170() {
        return Rule.builder().error(
                "Invalid ARN: Expected a resource of the format `accesspoint:<accesspoint name>` but no name was provided");
    }

    private static Rule endpointRule_128() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_129(), endpointRule_170()));
    }

    private static Rule endpointRule_127() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("accesspoint"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_128()));
    }

    private static Rule endpointRule_171() {
        return Rule.builder().error(
                "Invalid ARN: Object Lambda ARNs only support `accesspoint` arn types, but found: `{arnType}`");
    }

    private static Rule endpointRule_126() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("service"))).build().validate(),
                                                Expr.of("s3-object-lambda"))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_127(), endpointRule_171()));
    }

    private static Rule endpointRule_179() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("DisableAccessPoints")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("DisableAccessPoints")), Expr.of(true)))
                                        .build().validate()).build()).error("Access points are not supported for this operation");
    }

    private static Rule endpointRule_183() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("UseArnRegion"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseArnRegion")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                        Expr.of("region"))).build().validate(),
                                                        Expr.of("{Region}"))).build().validate())).build().validate()).build())
                .error("Invalid configuration: region from ARN `{bucketArn#region}` does not match client region `{Region}` and UseArnRegion is `false`");
    }

    private static Rule endpointRule_199() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build()).error("Access Points do not support S3 Accelerate");
    }

    private static Rule endpointRule_201() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketPartition")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-cn"))).build()
                                        .validate()).build()).error("Partition does not support FIPS");
    }

    private static Rule endpointRule_203() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .error("DualStack cannot be combined with a Host override (PrivateLink)");
    }

    private static Rule endpointRule_205() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.s3-accesspoint-fips.dualstack.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{bucketArn#region}"))))))
                                .build());
    }

    private static Rule endpointRule_206() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.s3-accesspoint-fips.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{bucketArn#region}"))))))
                                .build());
    }

    private static Rule endpointRule_207() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.s3-accesspoint.dualstack.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{bucketArn#region}"))))))
                                .build());
    }

    private static Rule endpointRule_208() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{accessPointName}-{bucketArn#accountId}.{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{bucketArn#region}"))))))
                                .build());
    }

    private static Rule endpointRule_209() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.s3-accesspoint.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{bucketArn#region}"))))))
                                .build());
    }

    private static Rule endpointRule_204() {
        return Rule.builder()
                .treeRule(
                        Arrays.asList(endpointRule_205(), endpointRule_206(), endpointRule_207(), endpointRule_208(),
                                endpointRule_209()));
    }

    private static Rule endpointRule_202() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_203(), endpointRule_204()));
    }

    private static Rule endpointRule_200() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_201(), endpointRule_202()));
    }

    private static Rule endpointRule_198() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_199(), endpointRule_200()));
    }

    private static Rule endpointRule_197() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("accessPointName")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_198()));
    }

    private static Rule endpointRule_210() {
        return Rule.builder().error(
                "Invalid ARN: The access point name may only contain a-z, A-Z, 0-9 and `-`. Found: `{accessPointName}`");
    }

    private static Rule endpointRule_196() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_197(), endpointRule_210()));
    }

    private static Rule endpointRule_195() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("accountId"))).build().validate(), Expr.of(false)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_196()));
    }

    private static Rule endpointRule_211() {
        return Rule.builder().error(
                "Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `{bucketArn#accountId}`");
    }

    private static Rule endpointRule_194() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_195(), endpointRule_211()));
    }

    private static Rule endpointRule_193() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("service"))).build().validate(), Expr.of("s3"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_194()));
    }

    private static Rule endpointRule_212() {
        return Rule.builder().error("Invalid ARN: The ARN was not for the S3 service, found: {bucketArn#service}");
    }

    private static Rule endpointRule_192() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_193(), endpointRule_212()));
    }

    private static Rule endpointRule_191() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("region"))).build().validate(), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_192()));
    }

    private static Rule endpointRule_213() {
        return Rule.builder().error("Invalid region in ARN: `{bucketArn#region}` (invalid DNS name)");
    }

    private static Rule endpointRule_190() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_191(), endpointRule_213()));
    }

    private static Rule endpointRule_189() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketPartition")),
                                                                Expr.of("name"))).build().validate(),
                                                Expr.of("{partitionResult#name}"))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_190()));
    }

    private static Rule endpointRule_214() {
        return Rule.builder().error(
                "Client was configured for partition `{partitionResult#name}` but ARN (`{Bucket}`) has `{bucketPartition#name}`");
    }

    private static Rule endpointRule_188() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_189(), endpointRule_214()));
    }

    private static Rule endpointRule_187() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_188()));
    }

    private static Rule endpointRule_215() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_186() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_187(), endpointRule_215()));
    }

    private static Rule endpointRule_185() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("aws.partition")
                                        .argv(Arrays.asList(FnNode.builder().fn("getAttr")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("region")))
                                                .build().validate())).build().validate()).result("bucketPartition").build())
                .treeRule(Arrays.asList(endpointRule_186()));
    }

    private static Rule endpointRule_216() {
        return Rule.builder().error("Could not load partition for ARN region `{bucketArn#region}`");
    }

    private static Rule endpointRule_184() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_185(), endpointRule_216()));
    }

    private static Rule endpointRule_182() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_183(), endpointRule_184()));
    }

    private static Rule endpointRule_181() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("isSet")
                                                .argv(Arrays.asList(FnNode
                                                        .builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("resourceId[2]"))).build().validate())).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_182()));
    }

    private static Rule endpointRule_217() {
        return Rule.builder().error("Invalid ARN: The ARN may only contain a single resource component after `accesspoint`.");
    }

    private static Rule endpointRule_180() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_181(), endpointRule_217()));
    }

    private static Rule endpointRule_178() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_179(), endpointRule_180()));
    }

    private static Rule endpointRule_177() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                        Expr.of("region"))).build().validate(), Expr.of("")))
                                                .build().validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_178()));
    }

    private static Rule endpointRule_218() {
        return Rule.builder().error("Invalid ARN: bucket ARN is missing a region");
    }

    private static Rule endpointRule_176() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("accesspoint"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_177(), endpointRule_218()));
    }

    private static Rule endpointRule_175() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                        Expr.of("region"))).build().validate(), Expr.of("")))
                                                .build().validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_176()));
    }

    private static Rule endpointRule_222() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 MRAP does not support dual-stack");
    }

    private static Rule endpointRule_224() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 MRAP does not support FIPS");
    }

    private static Rule endpointRule_226() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 MRAP does not support S3 Accelerate");
    }

    private static Rule endpointRule_228() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("DisableMultiRegionAccessPoints")),
                                                Expr.of(true))).build().validate()).build())
                .error("Invalid configuration: Multi-Region Access Point ARNs are disabled.");
    }

    private static Rule endpointRule_233() {
        return Rule.builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{accessPointName}.accesspoint.s3-global.{mrapPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4a"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegionSet"),
                                                Literal.fromTuple(Arrays.asList(Literal.fromStr("*")))))))).build());
    }

    private static Rule endpointRule_232() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("mrapPartition")),
                                                                Expr.of("name"))).build().validate(),
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("partition"))).build().validate())).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_233()));
    }

    private static Rule endpointRule_234() {
        return Rule
                .builder()
                .error("Client was configured for partition `{mrapPartition#name}` but bucket referred to partition `{bucketArn#partition}`");
    }

    private static Rule endpointRule_231() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_232(), endpointRule_234()));
    }

    private static Rule endpointRule_230() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("mrapPartition").build())
                .treeRule(Arrays.asList(endpointRule_231()));
    }

    private static Rule endpointRule_235() {
        return Rule.builder().error("{Region} was not a valid region");
    }

    private static Rule endpointRule_229() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_230(), endpointRule_235()));
    }

    private static Rule endpointRule_227() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_228(), endpointRule_229()));
    }

    private static Rule endpointRule_225() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_226(), endpointRule_227()));
    }

    private static Rule endpointRule_223() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_224(), endpointRule_225()));
    }

    private static Rule endpointRule_221() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_222(), endpointRule_223()));
    }

    private static Rule endpointRule_220() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("accessPointName")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_221()));
    }

    private static Rule endpointRule_236() {
        return Rule.builder().error("Invalid Access Point Name");
    }

    private static Rule endpointRule_219() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_220(), endpointRule_236()));
    }

    private static Rule endpointRule_174() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_175(), endpointRule_219()));
    }

    private static Rule endpointRule_173() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("resourceId[1]")))
                                        .build().validate()).result("accessPointName").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("accessPointName")), Expr.of("")))
                                                .build().validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_174()));
    }

    private static Rule endpointRule_237() {
        return Rule.builder().error(
                "Invalid ARN: Expected a resource of the format `accesspoint:<accesspoint name>` but no name was provided");
    }

    private static Rule endpointRule_172() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("accesspoint"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_173(), endpointRule_237()));
    }

    private static Rule endpointRule_239() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Outposts does not support Dual-stack");
    }

    private static Rule endpointRule_241() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Outposts does not support FIPS");
    }

    private static Rule endpointRule_243() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Outposts does not support S3 Accelerate");
    }

    private static Rule endpointRule_245() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isSet")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("getAttr")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                        Expr.of("resourceId[4]"))).build().validate())).build().validate())
                                .build()).error("Invalid Arn: Outpost Access Point ARN contains sub resources");
    }

    private static Rule endpointRule_251() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("UseArnRegion"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseArnRegion")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                        Expr.of("region"))).build().validate(),
                                                        Expr.of("{Region}"))).build().validate())).build().validate()).build())
                .error("Invalid configuration: region from ARN `{bucketArn#region}` does not match client region `{Region}` and UseArnRegion is `false`");
    }

    private static Rule endpointRule_269() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://{accessPointName}-{bucketArn#accountId}.{outpostId}.{url#authority}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-outposts"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{bucketArn#region}")))))).build());
    }

    private static Rule endpointRule_270() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{accessPointName}-{bucketArn#accountId}.{outpostId}.s3-outposts.{bucketArn#region}.{bucketPartition#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-outposts"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{bucketArn#region}")))))).build());
    }

    private static Rule endpointRule_268() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_269(), endpointRule_270()));
    }

    private static Rule endpointRule_267() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("outpostType")), Expr.of("accesspoint")))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_268()));
    }

    private static Rule endpointRule_271() {
        return Rule.builder().error("Expected an outpost type `accesspoint`, found {outpostType}");
    }

    private static Rule endpointRule_266() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_267(), endpointRule_271()));
    }

    private static Rule endpointRule_265() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("resourceId[3]")))
                                        .build().validate()).result("accessPointName").build())
                .treeRule(Arrays.asList(endpointRule_266()));
    }

    private static Rule endpointRule_272() {
        return Rule.builder().error("Invalid ARN: expected an access point name");
    }

    private static Rule endpointRule_264() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_265(), endpointRule_272()));
    }

    private static Rule endpointRule_263() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("resourceId[2]")))
                                        .build().validate()).result("outpostType").build())
                .treeRule(Arrays.asList(endpointRule_264()));
    }

    private static Rule endpointRule_273() {
        return Rule.builder().error("Invalid ARN: Expected a 4-component resource");
    }

    private static Rule endpointRule_262() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_263(), endpointRule_273()));
    }

    private static Rule endpointRule_261() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("accountId"))).build().validate(), Expr.of(false)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_262()));
    }

    private static Rule endpointRule_274() {
        return Rule.builder().error(
                "Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `{bucketArn#accountId}`");
    }

    private static Rule endpointRule_260() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_261(), endpointRule_274()));
    }

    private static Rule endpointRule_259() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("region"))).build().validate(), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_260()));
    }

    private static Rule endpointRule_275() {
        return Rule.builder().error("Invalid region in ARN: `{bucketArn#region}` (invalid DNS name)");
    }

    private static Rule endpointRule_258() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_259(), endpointRule_275()));
    }

    private static Rule endpointRule_257() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketPartition")),
                                                                Expr.of("name"))).build().validate(),
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate())).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_258()));
    }

    private static Rule endpointRule_276() {
        return Rule.builder().error(
                "Client was configured for partition `{partitionResult#name}` but ARN (`{Bucket}`) has `{bucketPartition#name}`");
    }

    private static Rule endpointRule_256() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_257(), endpointRule_276()));
    }

    private static Rule endpointRule_255() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_256()));
    }

    private static Rule endpointRule_277() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_254() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_255(), endpointRule_277()));
    }

    private static Rule endpointRule_253() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("aws.partition")
                                        .argv(Arrays.asList(FnNode.builder().fn("getAttr")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("region")))
                                                .build().validate())).build().validate()).result("bucketPartition").build())
                .treeRule(Arrays.asList(endpointRule_254()));
    }

    private static Rule endpointRule_278() {
        return Rule.builder().error("Could not load partition for ARN region {bucketArn#region}");
    }

    private static Rule endpointRule_252() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_253(), endpointRule_278()));
    }

    private static Rule endpointRule_250() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_251(), endpointRule_252()));
    }

    private static Rule endpointRule_249() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("outpostId")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_250()));
    }

    private static Rule endpointRule_279() {
        return Rule.builder().error("Invalid ARN: The outpost Id may only contain a-z, A-Z, 0-9 and `-`. Found: `{outpostId}`");
    }

    private static Rule endpointRule_248() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_249(), endpointRule_279()));
    }

    private static Rule endpointRule_247() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("resourceId[1]")))
                                        .build().validate()).result("outpostId").build())
                .treeRule(Arrays.asList(endpointRule_248()));
    }

    private static Rule endpointRule_280() {
        return Rule.builder().error("Invalid ARN: The Outpost Id was not set");
    }

    private static Rule endpointRule_246() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_247(), endpointRule_280()));
    }

    private static Rule endpointRule_244() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_245(), endpointRule_246()));
    }

    private static Rule endpointRule_242() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_243(), endpointRule_244()));
    }

    private static Rule endpointRule_240() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_241(), endpointRule_242()));
    }

    private static Rule endpointRule_238() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")),
                                                                Expr.of("service"))).build().validate(), Expr.of("s3-outposts")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_239(), endpointRule_240()));
    }

    private static Rule endpointRule_281() {
        return Rule.builder().error("Invalid ARN: Unrecognized format: {Bucket} (type: {arnType})");
    }

    private static Rule endpointRule_125() {
        return Rule.builder().treeRule(
                Arrays.asList(endpointRule_126(), endpointRule_172(), endpointRule_238(), endpointRule_281()));
    }

    private static Rule endpointRule_124() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("bucketArn")), Expr.of("resourceId[0]")))
                                        .build().validate()).result("arnType").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of(""))).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_125()));
    }

    private static Rule endpointRule_282() {
        return Rule.builder().error("Invalid ARN: No ARN type specified");
    }

    private static Rule endpointRule_123() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.parseArn").argv(Arrays.asList(Expr.ref(Identifier.of("Bucket"))))
                                        .build().validate()).result("bucketArn").build())
                .treeRule(Arrays.asList(endpointRule_124(), endpointRule_282()));
    }

    private static Rule endpointRule_283() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("substring")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")), Expr.of(0), Expr.of(4),
                                                Expr.of(false))).build().validate()).result("arnPrefix").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnPrefix")), Expr.of("arn:"))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("isSet")
                                                .argv(Arrays.asList(FnNode.builder().fn("aws.parseArn")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")))).build()
                                                        .validate())).build().validate())).build().validate()).build())
                .error("Invalid ARN: `{Bucket}` was not a valid ARN");
    }

    private static Rule endpointRule_285() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .error("Cannot set dual-stack in combination with a custom endpoint.");
    }

    private static Rule endpointRule_293() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.us-east-1.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_294() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.us-east-1.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_296() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3.dualstack.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_295() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_296()));
    }

    private static Rule endpointRule_297() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_298() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_299() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_301() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_302() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_300() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_301(), endpointRule_302()));
    }

    private static Rule endpointRule_303() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#normalizedPath}{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_304() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_305() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_307() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_308() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_306() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_307(), endpointRule_308()));
    }

    private static Rule endpointRule_309() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{Region}.{partitionResult#dnsSuffix}/{uri_encoded_bucket}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_292() {
        return Rule.builder().treeRule(
                Arrays.asList(endpointRule_293(), endpointRule_294(), endpointRule_295(), endpointRule_297(), endpointRule_298(),
                        endpointRule_299(), endpointRule_300(), endpointRule_303(), endpointRule_304(), endpointRule_305(),
                        endpointRule_306(), endpointRule_309()));
    }

    private static Rule endpointRule_291() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_292()));
    }

    private static Rule endpointRule_310() {
        return Rule.builder().error("Path-style addressing cannot be used with FIPS");
    }

    private static Rule endpointRule_290() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_291(), endpointRule_310()));
    }

    private static Rule endpointRule_289() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_290()));
    }

    private static Rule endpointRule_311() {
        return Rule.builder().error("Path-style addressing cannot be used with S3 Accelerate");
    }

    private static Rule endpointRule_288() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_289(), endpointRule_311()));
    }

    private static Rule endpointRule_287() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_288()));
    }

    private static Rule endpointRule_312() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_286() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_287(), endpointRule_312()));
    }

    private static Rule endpointRule_284() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("uriEncode").argv(Arrays.asList(Expr.ref(Identifier.of("Bucket"))))
                                        .build().validate()).result("uri_encoded_bucket").build())
                .treeRule(Arrays.asList(endpointRule_285(), endpointRule_286()));
    }

    private static Rule endpointRule_20() {
        return Rule.builder().treeRule(
                Arrays.asList(endpointRule_21(), endpointRule_53(), endpointRule_116(), endpointRule_123(), endpointRule_283(),
                        endpointRule_284()));
    }

    private static Rule endpointRule_18() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_19(), endpointRule_20()));
    }

    private static Rule endpointRule_318() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Object Lambda does not support Dual-stack");
    }

    private static Rule endpointRule_320() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Accelerate")), Expr.of(true))).build()
                                        .validate()).build()).error("S3 Object Lambda does not support S3 Accelerate");
    }

    private static Rule endpointRule_322() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-cn"))).build()
                                        .validate()).build()).error("Partition does not support FIPS");
    }

    private static Rule endpointRule_324() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-object-lambda"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_325() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-object-lambda-fips.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                                Literal.fromStr("s3-object-lambda"), Identifier.of("disableDoubleEncoding"),
                                                Literal.fromBool(true), Identifier.of("signingRegion"),
                                                Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_326() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3-object-lambda.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"),
                                        Literal.fromStr("s3-object-lambda"), Identifier.of("disableDoubleEncoding"),
                                        Literal.fromBool(true), Identifier.of("signingRegion"), Literal.fromStr("{Region}"))))))
                        .build());
    }

    private static Rule endpointRule_323() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_324(), endpointRule_325(), endpointRule_326()));
    }

    private static Rule endpointRule_321() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_322(), endpointRule_323()));
    }

    private static Rule endpointRule_319() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_320(), endpointRule_321()));
    }

    private static Rule endpointRule_317() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_318(), endpointRule_319()));
    }

    private static Rule endpointRule_316() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of(true))).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_317()));
    }

    private static Rule endpointRule_327() {
        return Rule.builder().error("Invalid region: region was not a valid DNS name.");
    }

    private static Rule endpointRule_315() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_316(), endpointRule_327()));
    }

    private static Rule endpointRule_314() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_315()));
    }

    private static Rule endpointRule_328() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_313() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseObjectLambdaEndpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseObjectLambdaEndpoint")), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_314(), endpointRule_328()));
    }

    private static Rule endpointRule_334() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-cn"))).build()
                                        .validate()).build()).error("Partition does not support FIPS");
    }

    private static Rule endpointRule_336() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_337() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_339() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_338() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_339()));
    }

    private static Rule endpointRule_340() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_341() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-fips.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_342() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-fips.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_344() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3-fips.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_343() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_344()));
    }

    private static Rule endpointRule_345() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-fips.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_346() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_347() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_349() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_348() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_349()));
    }

    private static Rule endpointRule_350() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_351() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-fips.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_352() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-fips.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_354() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3-fips.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_353() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_354()));
    }

    private static Rule endpointRule_355() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3-fips.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_356() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_357() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_359() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_358() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_359()));
    }

    private static Rule endpointRule_360() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_361() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_362() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.us-east-1.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_364() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_363() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_364()));
    }

    private static Rule endpointRule_365() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.dualstack.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_366() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_367() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_369() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_370() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_368() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_369(), endpointRule_370()));
    }

    private static Rule endpointRule_371() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("{url#scheme}://{url#authority}{url#path}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_372() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_373() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_375() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("us-east-1"))).build()
                                        .validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_376() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://s3.{Region}.{partitionResult#dnsSuffix}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                        Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                        Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_374() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_375(), endpointRule_376()));
    }

    private static Rule endpointRule_377() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(false))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("aws-global")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseGlobalEndpoint")), Expr.of(false)))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr.of("https://s3.{Region}.{partitionResult#dnsSuffix}"))
                                .addProperty(
                                        Identifier.of("authSchemes"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                                Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("s3"),
                                                Identifier.of("disableDoubleEncoding"), Literal.fromBool(true),
                                                Identifier.of("signingRegion"), Literal.fromStr("{Region}")))))).build());
    }

    private static Rule endpointRule_335() {
        return Rule.builder().treeRule(
                Arrays.asList(endpointRule_336(), endpointRule_337(), endpointRule_338(), endpointRule_340(), endpointRule_341(),
                        endpointRule_342(), endpointRule_343(), endpointRule_345(), endpointRule_346(), endpointRule_347(),
                        endpointRule_348(), endpointRule_350(), endpointRule_351(), endpointRule_352(), endpointRule_353(),
                        endpointRule_355(), endpointRule_356(), endpointRule_357(), endpointRule_358(), endpointRule_360(),
                        endpointRule_361(), endpointRule_362(), endpointRule_363(), endpointRule_365(), endpointRule_366(),
                        endpointRule_367(), endpointRule_368(), endpointRule_371(), endpointRule_372(), endpointRule_373(),
                        endpointRule_374(), endpointRule_377()));
    }

    private static Rule endpointRule_333() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_334(), endpointRule_335()));
    }

    private static Rule endpointRule_332() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isValidHostLabel")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of(true))).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_333()));
    }

    private static Rule endpointRule_378() {
        return Rule.builder().error("Invalid region: region was not a valid DNS name.");
    }

    private static Rule endpointRule_331() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_332(), endpointRule_378()));
    }

    private static Rule endpointRule_330() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_331()));
    }

    private static Rule endpointRule_379() {
        return Rule.builder().error("A valid partition could not be determined");
    }

    private static Rule endpointRule_329() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Bucket")))).build().validate()))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_330(), endpointRule_379()));
    }

    private static Rule endpointRule_2() {
        return Rule.builder()
                .treeRule(Arrays.asList(endpointRule_3(), endpointRule_18(), endpointRule_313(), endpointRule_329()));
    }

    private static Rule endpointRule_1() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_2()));
    }

    private static Rule endpointRule_380() {
        return Rule.builder().error("A region must be set when sending requests to S3.");
    }

    private static Rule endpointRule_0() {
        return Rule.builder().treeRule(Arrays.asList(endpointRule_1(), endpointRule_380()));
    }

    private static EndpointRuleset ruleSet() {
        return EndpointRuleset
                .builder()
                .version("1.0")
                .serviceId(null)
                .parameters(
                        Parameters
                                .builder()
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("Bucket")
                                                .type(ParameterType.fromValue("String"))
                                                .required(false)
                                                .documentation(
                                                        "The S3 bucket used to send the request. This is an optional parameter that will be set automatically for operations that are scoped to an S3 bucket.")
                                                .build())
                                .addParameter(
                                        Parameter.builder().name("Region").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("AWS::Region")
                                                .documentation("The AWS region used to dispatch the request.").build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseFIPS")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseFIPS")
                                                .documentation(
                                                        "When true, send this request to the FIPS-compliant regional endpoint. If the configured endpoint does not have a FIPS compliant endpoint, dispatching the request will return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseDualStack")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseDualStack")
                                                .documentation(
                                                        "When true, use the dual-stack endpoint. If the configured endpoint does not support dual-stack, dispatching the request MAY return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter.builder().name("Endpoint").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("SDK::Endpoint")
                                                .documentation("Override the endpoint used to send this request").build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("ForcePathStyle")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(false)
                                                .builtIn("AWS::S3::ForcePathStyle")
                                                .documentation(
                                                        "When true, force a path-style endpoint to be used where the bucket name is part of the path.")
                                                .build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("Accelerate")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::S3::Accelerate")
                                                .documentation(
                                                        "When true, use S3 Accelerate. NOTE: Not all regions support S3 accelerate.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseGlobalEndpoint")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::S3::UseGlobalEndpoint")
                                                .documentation(
                                                        "Whether the global endpoint should be used, rather then the regional endpoint for us-east-1.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseObjectLambdaEndpoint")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(false)
                                                .documentation(
                                                        "Internal parameter to use object lambda endpoint for an operation (eg: WriteGetObjectResponse)")
                                                .build())
                                .addParameter(
                                        Parameter.builder().name("DisableAccessPoints").type(ParameterType.fromValue("Boolean"))
                                                .required(false)
                                                .documentation("Internal parameter to disable Access Point Buckets").build())
                                .addParameter(
                                        Parameter.builder().name("DisableMultiRegionAccessPoints")
                                                .type(ParameterType.fromValue("Boolean")).required(true)
                                                .builtIn("AWS::S3::DisableMultiRegionAccessPoints")
                                                .documentation("Whether multi-region access points (MRAP) should be disabled.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseArnRegion")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(false)
                                                .builtIn("AWS::S3::UseArnRegion")
                                                .documentation(
                                                        "When an Access Point ARN is provided and this flag is enabled, the SDK MUST use the ARN's region when constructing the endpoint instead of the client's configured region.")
                                                .build()).build()).addRule(endpointRule_0()).build();
    }
}
