/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rtbfabric.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.rtbfabric.RtbFabricAsyncClient;
import software.amazon.awssdk.services.rtbfabric.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.rtbfabric.model.GetInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.RtbFabricRequest;
import software.amazon.awssdk.services.rtbfabric.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultRtbFabricAsyncWaiter implements RtbFabricAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final RtbFabricAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetInboundExternalLinkResponse> inboundExternalLinkActiveWaiter;

    private final AsyncWaiter<GetLinkResponse> linkAcceptedWaiter;

    private final AsyncWaiter<GetLinkResponse> linkActiveWaiter;

    private final AsyncWaiter<GetOutboundExternalLinkResponse> outboundExternalLinkActiveWaiter;

    private final AsyncWaiter<GetRequesterGatewayResponse> requesterGatewayActiveWaiter;

    private final AsyncWaiter<GetRequesterGatewayResponse> requesterGatewayDeletedWaiter;

    private final AsyncWaiter<GetResponderGatewayResponse> responderGatewayActiveWaiter;

    private final AsyncWaiter<GetResponderGatewayResponse> responderGatewayDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultRtbFabricAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = RtbFabricAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.inboundExternalLinkActiveWaiter = AsyncWaiter.builder(GetInboundExternalLinkResponse.class)
                .acceptors(inboundExternalLinkActiveWaiterAcceptors())
                .overrideConfiguration(inboundExternalLinkActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.linkAcceptedWaiter = AsyncWaiter.builder(GetLinkResponse.class).acceptors(linkAcceptedWaiterAcceptors())
                .overrideConfiguration(linkAcceptedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.linkActiveWaiter = AsyncWaiter.builder(GetLinkResponse.class).acceptors(linkActiveWaiterAcceptors())
                .overrideConfiguration(linkActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.outboundExternalLinkActiveWaiter = AsyncWaiter.builder(GetOutboundExternalLinkResponse.class)
                .acceptors(outboundExternalLinkActiveWaiterAcceptors())
                .overrideConfiguration(outboundExternalLinkActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.requesterGatewayActiveWaiter = AsyncWaiter.builder(GetRequesterGatewayResponse.class)
                .acceptors(requesterGatewayActiveWaiterAcceptors())
                .overrideConfiguration(requesterGatewayActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.requesterGatewayDeletedWaiter = AsyncWaiter.builder(GetRequesterGatewayResponse.class)
                .acceptors(requesterGatewayDeletedWaiterAcceptors())
                .overrideConfiguration(requesterGatewayDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.responderGatewayActiveWaiter = AsyncWaiter.builder(GetResponderGatewayResponse.class)
                .acceptors(responderGatewayActiveWaiterAcceptors())
                .overrideConfiguration(responderGatewayActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.responderGatewayDeletedWaiter = AsyncWaiter.builder(GetResponderGatewayResponse.class)
                .acceptors(responderGatewayDeletedWaiterAcceptors())
                .overrideConfiguration(responderGatewayDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetInboundExternalLinkResponse>> waitUntilInboundExternalLinkActive(
            GetInboundExternalLinkRequest getInboundExternalLinkRequest) {
        return inboundExternalLinkActiveWaiter.runAsync(() -> client
                .getInboundExternalLink(applyWaitersUserAgent(getInboundExternalLinkRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetInboundExternalLinkResponse>> waitUntilInboundExternalLinkActive(
            GetInboundExternalLinkRequest getInboundExternalLinkRequest, WaiterOverrideConfiguration overrideConfig) {
        return inboundExternalLinkActiveWaiter.runAsync(
                () -> client.getInboundExternalLink(applyWaitersUserAgent(getInboundExternalLinkRequest)),
                inboundExternalLinkActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLinkResponse>> waitUntilLinkAccepted(GetLinkRequest getLinkRequest) {
        return linkAcceptedWaiter.runAsync(() -> client.getLink(applyWaitersUserAgent(getLinkRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLinkResponse>> waitUntilLinkAccepted(GetLinkRequest getLinkRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return linkAcceptedWaiter.runAsync(() -> client.getLink(applyWaitersUserAgent(getLinkRequest)),
                linkAcceptedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLinkResponse>> waitUntilLinkActive(GetLinkRequest getLinkRequest) {
        return linkActiveWaiter.runAsync(() -> client.getLink(applyWaitersUserAgent(getLinkRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLinkResponse>> waitUntilLinkActive(GetLinkRequest getLinkRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return linkActiveWaiter.runAsync(() -> client.getLink(applyWaitersUserAgent(getLinkRequest)),
                linkActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetOutboundExternalLinkResponse>> waitUntilOutboundExternalLinkActive(
            GetOutboundExternalLinkRequest getOutboundExternalLinkRequest) {
        return outboundExternalLinkActiveWaiter.runAsync(() -> client
                .getOutboundExternalLink(applyWaitersUserAgent(getOutboundExternalLinkRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetOutboundExternalLinkResponse>> waitUntilOutboundExternalLinkActive(
            GetOutboundExternalLinkRequest getOutboundExternalLinkRequest, WaiterOverrideConfiguration overrideConfig) {
        return outboundExternalLinkActiveWaiter.runAsync(
                () -> client.getOutboundExternalLink(applyWaitersUserAgent(getOutboundExternalLinkRequest)),
                outboundExternalLinkActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRequesterGatewayResponse>> waitUntilRequesterGatewayActive(
            GetRequesterGatewayRequest getRequesterGatewayRequest) {
        return requesterGatewayActiveWaiter.runAsync(() -> client
                .getRequesterGateway(applyWaitersUserAgent(getRequesterGatewayRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRequesterGatewayResponse>> waitUntilRequesterGatewayActive(
            GetRequesterGatewayRequest getRequesterGatewayRequest, WaiterOverrideConfiguration overrideConfig) {
        return requesterGatewayActiveWaiter.runAsync(
                () -> client.getRequesterGateway(applyWaitersUserAgent(getRequesterGatewayRequest)),
                requesterGatewayActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRequesterGatewayResponse>> waitUntilRequesterGatewayDeleted(
            GetRequesterGatewayRequest getRequesterGatewayRequest) {
        return requesterGatewayDeletedWaiter.runAsync(() -> client
                .getRequesterGateway(applyWaitersUserAgent(getRequesterGatewayRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRequesterGatewayResponse>> waitUntilRequesterGatewayDeleted(
            GetRequesterGatewayRequest getRequesterGatewayRequest, WaiterOverrideConfiguration overrideConfig) {
        return requesterGatewayDeletedWaiter.runAsync(
                () -> client.getRequesterGateway(applyWaitersUserAgent(getRequesterGatewayRequest)),
                requesterGatewayDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetResponderGatewayResponse>> waitUntilResponderGatewayActive(
            GetResponderGatewayRequest getResponderGatewayRequest) {
        return responderGatewayActiveWaiter.runAsync(() -> client
                .getResponderGateway(applyWaitersUserAgent(getResponderGatewayRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetResponderGatewayResponse>> waitUntilResponderGatewayActive(
            GetResponderGatewayRequest getResponderGatewayRequest, WaiterOverrideConfiguration overrideConfig) {
        return responderGatewayActiveWaiter.runAsync(
                () -> client.getResponderGateway(applyWaitersUserAgent(getResponderGatewayRequest)),
                responderGatewayActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetResponderGatewayResponse>> waitUntilResponderGatewayDeleted(
            GetResponderGatewayRequest getResponderGatewayRequest) {
        return responderGatewayDeletedWaiter.runAsync(() -> client
                .getResponderGateway(applyWaitersUserAgent(getResponderGatewayRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetResponderGatewayResponse>> waitUntilResponderGatewayDeleted(
            GetResponderGatewayRequest getResponderGatewayRequest, WaiterOverrideConfiguration overrideConfig) {
        return responderGatewayDeletedWaiter.runAsync(
                () -> client.getResponderGateway(applyWaitersUserAgent(getResponderGatewayRequest)),
                responderGatewayDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetInboundExternalLinkResponse>> inboundExternalLinkActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetInboundExternalLinkResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "REJECTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=REJECTED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "ISOLATED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ISOLATED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLinkResponse>> linkAcceptedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLinkResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACCEPTED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "REJECTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=REJECTED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLinkResponse>> linkActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLinkResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "REJECTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=REJECTED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetOutboundExternalLinkResponse>> outboundExternalLinkActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetOutboundExternalLinkResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "REJECTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=REJECTED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "ISOLATED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ISOLATED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRequesterGatewayResponse>> requesterGatewayActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRequesterGatewayResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRequesterGatewayResponse>> requesterGatewayDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRequesterGatewayResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetResponderGatewayResponse>> responderGatewayActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetResponderGatewayResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetResponderGatewayResponse>> responderGatewayDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetResponderGatewayResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration inboundExternalLinkActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration linkAcceptedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration linkActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration outboundExternalLinkActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration requesterGatewayActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration requesterGatewayDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration responderGatewayActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration responderGatewayDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static RtbFabricAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends RtbFabricRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements RtbFabricAsyncWaiter.Builder {
        private RtbFabricAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public RtbFabricAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public RtbFabricAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public RtbFabricAsyncWaiter.Builder client(RtbFabricAsyncClient client) {
            this.client = client;
            return this;
        }

        public RtbFabricAsyncWaiter build() {
            return new DefaultRtbFabricAsyncWaiter(this);
        }
    }
}
