/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.InternalServiceErrorException;
import software.amazon.awssdk.services.route53resolver.model.InvalidNextTokenException;
import software.amazon.awssdk.services.route53resolver.model.InvalidParameterException;
import software.amazon.awssdk.services.route53resolver.model.InvalidPolicyDocumentException;
import software.amazon.awssdk.services.route53resolver.model.InvalidRequestException;
import software.amazon.awssdk.services.route53resolver.model.InvalidTagException;
import software.amazon.awssdk.services.route53resolver.model.LimitExceededException;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.ResourceExistsException;
import software.amazon.awssdk.services.route53resolver.model.ResourceInUseException;
import software.amazon.awssdk.services.route53resolver.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53resolver.model.ResourceUnavailableException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.ThrottlingException;
import software.amazon.awssdk.services.route53resolver.model.UnknownResourceException;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointIpAddressesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRuleAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRulesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.PutResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverRuleRequestMarshaller;

/**
 * Internal implementation of {@link Route53ResolverClient}.
 *
 * @see Route53ResolverClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53ResolverClient implements Route53ResolverClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53ResolverClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds IP addresses to an inbound or an outbound resolver endpoint. If you want to adding more than one IP address,
     * submit one <code>AssociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To remove an IP address from an endpoint, see <a>DisassociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param associateResolverEndpointIpAddressRequest
     * @return Result of the AssociateResolverEndpointIpAddress operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.AssociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResolverEndpointIpAddressResponse associateResolverEndpointIpAddress(
            AssociateResolverEndpointIpAddressRequest associateResolverEndpointIpAddressRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, ResourceExistsException,
            InternalServiceErrorException, LimitExceededException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResolverEndpointIpAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<AssociateResolverEndpointIpAddressRequest, AssociateResolverEndpointIpAddressResponse>()
                        .withOperationName("AssociateResolverEndpointIpAddress").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(associateResolverEndpointIpAddressRequest)
                        .withMarshaller(new AssociateResolverEndpointIpAddressRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Associates a resolver rule with a VPC. When you associate a rule with a VPC, Resolver forwards all DNS queries
     * for the domain name that is specified in the rule and that originate in the VPC. The queries are forwarded to the
     * IP addresses for the DNS resolvers that are specified in the rule. For more information about rules, see
     * <a>CreateResolverRule</a>.
     * </p>
     *
     * @param associateResolverRuleRequest
     * @return Result of the AssociateResolverRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceUnavailableException
     *         The specified resource isn't available.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.AssociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResolverRuleResponse associateResolverRule(AssociateResolverRuleRequest associateResolverRuleRequest)
            throws ResourceNotFoundException, InvalidRequestException, InvalidParameterException, ResourceUnavailableException,
            ResourceExistsException, InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AssociateResolverRuleRequest, AssociateResolverRuleResponse>()
                .withOperationName("AssociateResolverRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(associateResolverRuleRequest)
                .withMarshaller(new AssociateResolverRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a resolver endpoint. There are two types of resolver endpoints, inbound and outbound:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An <i>inbound resolver endpoint</i> forwards DNS queries to the DNS service for a VPC from your network or
     * another VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * An <i>outbound resolver endpoint</i> forwards DNS queries from the DNS service for a VPC to your network or
     * another VPC.
     * </p>
     * </li>
     * </ul>
     *
     * @param createResolverEndpointRequest
     * @return Result of the CreateResolverEndpoint operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.CreateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResolverEndpointResponse createResolverEndpoint(CreateResolverEndpointRequest createResolverEndpointRequest)
            throws InvalidParameterException, ResourceNotFoundException, InvalidRequestException, ResourceExistsException,
            LimitExceededException, InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateResolverEndpointRequest, CreateResolverEndpointResponse>()
                .withOperationName("CreateResolverEndpoint").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createResolverEndpointRequest)
                .withMarshaller(new CreateResolverEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For DNS queries that originate in your VPCs, specifies which resolver endpoint the queries pass through, one
     * domain name that you want to forward to your network, and the IP addresses of the DNS resolvers in your network.
     * </p>
     *
     * @param createResolverRuleRequest
     * @return Result of the CreateResolverRule operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws ResourceUnavailableException
     *         The specified resource isn't available.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.CreateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResolverRuleResponse createResolverRule(CreateResolverRuleRequest createResolverRuleRequest)
            throws InvalidParameterException, InvalidRequestException, LimitExceededException, ResourceNotFoundException,
            ResourceExistsException, ResourceUnavailableException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateResolverRuleRequest, CreateResolverRuleResponse>()
                .withOperationName("CreateResolverRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createResolverRuleRequest)
                .withMarshaller(new CreateResolverRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a resolver endpoint. The effect of deleting a resolver endpoint depends on whether it's an inbound or an
     * outbound resolver endpoint:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Inbound</b>: DNS queries from your network or another VPC are no longer routed to the DNS service for the
     * specified VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Outbound</b>: DNS queries from a VPC are no longer routed to your network or to another VPC.
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteResolverEndpointRequest
     * @return Result of the DeleteResolverEndpoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DeleteResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResolverEndpointResponse deleteResolverEndpoint(DeleteResolverEndpointRequest deleteResolverEndpointRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteResolverEndpointRequest, DeleteResolverEndpointResponse>()
                .withOperationName("DeleteResolverEndpoint").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteResolverEndpointRequest)
                .withMarshaller(new DeleteResolverEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a resolver rule. Before you can delete a resolver rule, you must disassociate it from all the VPCs that
     * you associated the resolver rule with. For more infomation, see <a>DisassociateResolverRule</a>.
     * </p>
     *
     * @param deleteResolverRuleRequest
     * @return Result of the DeleteResolverRule operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceInUseException
     *         The resource that you tried to update or delete is currently in use.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DeleteResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResolverRuleResponse deleteResolverRule(DeleteResolverRuleRequest deleteResolverRuleRequest)
            throws InvalidParameterException, ResourceNotFoundException, ResourceInUseException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteResolverRuleRequest, DeleteResolverRuleResponse>()
                .withOperationName("DeleteResolverRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteResolverRuleRequest)
                .withMarshaller(new DeleteResolverRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes IP addresses from an inbound or an outbound resolver endpoint. If you want to remove more than one IP
     * address, submit one <code>DisassociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To add an IP address to an endpoint, see <a>AssociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param disassociateResolverEndpointIpAddressRequest
     * @return Result of the DisassociateResolverEndpointIpAddress operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DisassociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResolverEndpointIpAddressResponse disassociateResolverEndpointIpAddress(
            DisassociateResolverEndpointIpAddressRequest disassociateResolverEndpointIpAddressRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, ResourceExistsException,
            InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateResolverEndpointIpAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisassociateResolverEndpointIpAddressRequest, DisassociateResolverEndpointIpAddressResponse>()
                        .withOperationName("DisassociateResolverEndpointIpAddress").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResolverEndpointIpAddressRequest)
                        .withMarshaller(new DisassociateResolverEndpointIpAddressRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the association between a specified resolver rule and a specified VPC.
     * </p>
     * <important>
     * <p>
     * If you disassociate a resolver rule from a VPC, Resolver stops forwarding DNS queries for the domain name that
     * you specified in the resolver rule.
     * </p>
     * </important>
     *
     * @param disassociateResolverRuleRequest
     * @return Result of the DisassociateResolverRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DisassociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResolverRuleResponse disassociateResolverRule(
            DisassociateResolverRuleRequest disassociateResolverRuleRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, ThrottlingException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisassociateResolverRuleRequest, DisassociateResolverRuleResponse>()
                        .withOperationName("DisassociateResolverRule").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResolverRuleRequest)
                        .withMarshaller(new DisassociateResolverRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about a specified resolver endpoint, such as whether it's an inbound or an outbound resolver
     * endpoint, and the current status of the endpoint.
     * </p>
     *
     * @param getResolverEndpointRequest
     * @return Result of the GetResolverEndpoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverEndpointResponse getResolverEndpoint(GetResolverEndpointRequest getResolverEndpointRequest)
            throws ResourceNotFoundException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetResolverEndpointRequest, GetResolverEndpointResponse>()
                .withOperationName("GetResolverEndpoint").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getResolverEndpointRequest)
                .withMarshaller(new GetResolverEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about a specified resolver rule, such as the domain name that the rule forwards DNS queries for
     * and the ID of the outbound resolver endpoint that the rule is associated with.
     * </p>
     *
     * @param getResolverRuleRequest
     * @return Result of the GetResolverRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverRuleResponse getResolverRule(GetResolverRuleRequest getResolverRuleRequest)
            throws ResourceNotFoundException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetResolverRuleRequest, GetResolverRuleResponse>()
                .withOperationName("GetResolverRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getResolverRuleRequest)
                .withMarshaller(new GetResolverRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about an association between a specified resolver rule and a VPC. You associate a resolver rule
     * and a VPC using <a>AssociateResolverRule</a>.
     * </p>
     *
     * @param getResolverRuleAssociationRequest
     * @return Result of the GetResolverRuleAssociation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverRuleAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRuleAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverRuleAssociationResponse getResolverRuleAssociation(
            GetResolverRuleAssociationRequest getResolverRuleAssociationRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, ThrottlingException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverRuleAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverRuleAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetResolverRuleAssociationRequest, GetResolverRuleAssociationResponse>()
                        .withOperationName("GetResolverRuleAssociation").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getResolverRuleAssociationRequest)
                        .withMarshaller(new GetResolverRuleAssociationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about a resolver rule policy. A resolver rule policy specifies the Resolver operations and
     * resources that you want to allow another AWS account to be able to use.
     * </p>
     *
     * @param getResolverRulePolicyRequest
     * @return Result of the GetResolverRulePolicy operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws UnknownResourceException
     *         The specified resource doesn't exist.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverRulePolicyResponse getResolverRulePolicy(GetResolverRulePolicyRequest getResolverRulePolicyRequest)
            throws InvalidParameterException, UnknownResourceException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverRulePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetResolverRulePolicyRequest, GetResolverRulePolicyResponse>()
                .withOperationName("GetResolverRulePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getResolverRulePolicyRequest)
                .withMarshaller(new GetResolverRulePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the IP addresses for a specified resolver endpoint.
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return Result of the ListResolverEndpointIpAddresses operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointIpAddressesResponse listResolverEndpointIpAddresses(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverEndpointIpAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverEndpointIpAddressesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListResolverEndpointIpAddressesRequest, ListResolverEndpointIpAddressesResponse>()
                        .withOperationName("ListResolverEndpointIpAddresses").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listResolverEndpointIpAddressesRequest)
                        .withMarshaller(new ListResolverEndpointIpAddressesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the IP addresses for a specified resolver endpoint.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable responses = client.listResolverEndpointIpAddressesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable responses = client
     *             .listResolverEndpointIpAddressesPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable responses = client.listResolverEndpointIpAddressesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointIpAddressesIterable listResolverEndpointIpAddressesPaginator(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverEndpointIpAddressesIterable(this, applyPaginatorUserAgent(listResolverEndpointIpAddressesRequest));
    }

    /**
     * <p>
     * Lists all the resolver endpoints that were created using the current AWS account.
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return Result of the ListResolverEndpoints operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointsResponse listResolverEndpoints(ListResolverEndpointsRequest listResolverEndpointsRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListResolverEndpointsRequest, ListResolverEndpointsResponse>()
                .withOperationName("ListResolverEndpoints").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listResolverEndpointsRequest)
                .withMarshaller(new ListResolverEndpointsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all the resolver endpoints that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable responses = client.listResolverEndpointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable responses = client
     *             .listResolverEndpointsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable responses = client.listResolverEndpointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointsIterable listResolverEndpointsPaginator(ListResolverEndpointsRequest listResolverEndpointsRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverEndpointsIterable(this, applyPaginatorUserAgent(listResolverEndpointsRequest));
    }

    /**
     * <p>
     * Lists the associations that were created between resolver rules and VPCs using the current AWS account.
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return Result of the ListResolverRuleAssociations operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRuleAssociationsResponse listResolverRuleAssociations(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) throws InvalidNextTokenException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverRuleAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverRuleAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListResolverRuleAssociationsRequest, ListResolverRuleAssociationsResponse>()
                        .withOperationName("ListResolverRuleAssociations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listResolverRuleAssociationsRequest)
                        .withMarshaller(new ListResolverRuleAssociationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the associations that were created between resolver rules and VPCs using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable responses = client.listResolverRuleAssociationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable responses = client
     *             .listResolverRuleAssociationsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable responses = client.listResolverRuleAssociationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRuleAssociationsIterable listResolverRuleAssociationsPaginator(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) throws InvalidNextTokenException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverRuleAssociationsIterable(this, applyPaginatorUserAgent(listResolverRuleAssociationsRequest));
    }

    /**
     * <p>
     * Lists the resolver rules that were created using the current AWS account.
     * </p>
     *
     * @param listResolverRulesRequest
     * @return Result of the ListResolverRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRulesResponse listResolverRules(ListResolverRulesRequest listResolverRulesRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResolverRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListResolverRulesRequest, ListResolverRulesResponse>()
                .withOperationName("ListResolverRules").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listResolverRulesRequest)
                .withMarshaller(new ListResolverRulesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the resolver rules that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable responses = client.listResolverRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable responses = client
     *             .listResolverRulesPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable responses = client.listResolverRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRulesIterable listResolverRulesPaginator(ListResolverRulesRequest listResolverRulesRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverRulesIterable(this, applyPaginatorUserAgent(listResolverRulesRequest));
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidNextTokenException, InvalidRequestException,
            InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Specifies the Resolver operations and resources that you want to allow another AWS account to be able to use.
     * </p>
     *
     * @param putResolverRulePolicyRequest
     * @return Result of the PutResolverRulePolicy operation returned by the service.
     * @throws InvalidPolicyDocumentException
     *         The specified resolver rule policy is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws UnknownResourceException
     *         The specified resource doesn't exist.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.PutResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/PutResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutResolverRulePolicyResponse putResolverRulePolicy(PutResolverRulePolicyRequest putResolverRulePolicyRequest)
            throws InvalidPolicyDocumentException, InvalidParameterException, UnknownResourceException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutResolverRulePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutResolverRulePolicyRequest, PutResolverRulePolicyResponse>()
                .withOperationName("PutResolverRulePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putResolverRulePolicyRequest)
                .withMarshaller(new PutResolverRulePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds one or more tags to a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidTagException
     *         The specified tag is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws LimitExceededException,
            ResourceNotFoundException, InvalidParameterException, InvalidTagException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes one or more tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, ThrottlingException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the name of an inbound or an outbound resolver endpoint.
     * </p>
     *
     * @param updateResolverEndpointRequest
     * @return Result of the UpdateResolverEndpoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UpdateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResolverEndpointResponse updateResolverEndpoint(UpdateResolverEndpointRequest updateResolverEndpointRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateResolverEndpointRequest, UpdateResolverEndpointResponse>()
                .withOperationName("UpdateResolverEndpoint").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateResolverEndpointRequest)
                .withMarshaller(new UpdateResolverEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates settings for a specified resolver rule. <code>ResolverRuleId</code> is required, and all other parameters
     * are optional. If you don't specify a parameter, it retains its current value.
     * </p>
     *
     * @param updateResolverRuleRequest
     * @return Result of the UpdateResolverRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceUnavailableException
     *         The specified resource isn't available.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UpdateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResolverRuleResponse updateResolverRule(UpdateResolverRuleRequest updateResolverRuleRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ResourceUnavailableException,
            LimitExceededException, InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateResolverRuleRequest, UpdateResolverRuleResponse>()
                .withOperationName("UpdateResolverRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateResolverRuleRequest)
                .withMarshaller(new UpdateResolverRuleRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53ResolverException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyDocument")
                                .exceptionBuilderSupplier(InvalidPolicyDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends Route53ResolverRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
