/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.InternalServiceErrorException;
import software.amazon.awssdk.services.route53resolver.model.InvalidNextTokenException;
import software.amazon.awssdk.services.route53resolver.model.InvalidParameterException;
import software.amazon.awssdk.services.route53resolver.model.InvalidPolicyDocumentException;
import software.amazon.awssdk.services.route53resolver.model.InvalidRequestException;
import software.amazon.awssdk.services.route53resolver.model.InvalidTagException;
import software.amazon.awssdk.services.route53resolver.model.LimitExceededException;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.ResourceExistsException;
import software.amazon.awssdk.services.route53resolver.model.ResourceInUseException;
import software.amazon.awssdk.services.route53resolver.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53resolver.model.ResourceUnavailableException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.ThrottlingException;
import software.amazon.awssdk.services.route53resolver.model.UnknownResourceException;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesPublisher;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointIpAddressesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRuleAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRulesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.PutResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverRuleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Route53ResolverAsyncClient}.
 *
 * @see Route53ResolverAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53ResolverAsyncClient implements Route53ResolverAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRoute53ResolverAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53ResolverAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds IP addresses to an inbound or an outbound resolver endpoint. If you want to adding more than one IP address,
     * submit one <code>AssociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To remove an IP address from an endpoint, see <a>DisassociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param associateResolverEndpointIpAddressRequest
     * @return A Java Future containing the result of the AssociateResolverEndpointIpAddress operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.AssociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResolverEndpointIpAddressResponse> associateResolverEndpointIpAddress(
            AssociateResolverEndpointIpAddressRequest associateResolverEndpointIpAddressRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateResolverEndpointIpAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResolverEndpointIpAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverEndpointIpAddressRequest, AssociateResolverEndpointIpAddressResponse>()
                            .withOperationName("AssociateResolverEndpointIpAddress")
                            .withMarshaller(new AssociateResolverEndpointIpAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateResolverEndpointIpAddressRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a resolver rule with a VPC. When you associate a rule with a VPC, Resolver forwards all DNS queries
     * for the domain name that is specified in the rule and that originate in the VPC. The queries are forwarded to the
     * IP addresses for the DNS resolvers that are specified in the rule. For more information about rules, see
     * <a>CreateResolverRule</a>.
     * </p>
     *
     * @param associateResolverRuleRequest
     * @return A Java Future containing the result of the AssociateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceUnavailableException The specified resource isn't available.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.AssociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResolverRuleResponse> associateResolverRule(
            AssociateResolverRuleRequest associateResolverRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverRuleRequest, AssociateResolverRuleResponse>()
                            .withOperationName("AssociateResolverRule")
                            .withMarshaller(new AssociateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateResolverRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a resolver endpoint. There are two types of resolver endpoints, inbound and outbound:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An <i>inbound resolver endpoint</i> forwards DNS queries to the DNS service for a VPC from your network or
     * another VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * An <i>outbound resolver endpoint</i> forwards DNS queries from the DNS service for a VPC to your network or
     * another VPC.
     * </p>
     * </li>
     * </ul>
     *
     * @param createResolverEndpointRequest
     * @return A Java Future containing the result of the CreateResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.CreateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResolverEndpointResponse> createResolverEndpoint(
            CreateResolverEndpointRequest createResolverEndpointRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResolverEndpointRequest, CreateResolverEndpointResponse>()
                            .withOperationName("CreateResolverEndpoint")
                            .withMarshaller(new CreateResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createResolverEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For DNS queries that originate in your VPCs, specifies which resolver endpoint the queries pass through, one
     * domain name that you want to forward to your network, and the IP addresses of the DNS resolvers in your network.
     * </p>
     *
     * @param createResolverRuleRequest
     * @return A Java Future containing the result of the CreateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>ResourceUnavailableException The specified resource isn't available.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.CreateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResolverRuleResponse> createResolverRule(CreateResolverRuleRequest createResolverRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResolverRuleRequest, CreateResolverRuleResponse>()
                            .withOperationName("CreateResolverRule")
                            .withMarshaller(new CreateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createResolverRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resolver endpoint. The effect of deleting a resolver endpoint depends on whether it's an inbound or an
     * outbound resolver endpoint:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Inbound</b>: DNS queries from your network or another VPC are no longer routed to the DNS service for the
     * specified VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Outbound</b>: DNS queries from a VPC are no longer routed to your network or to another VPC.
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteResolverEndpointRequest
     * @return A Java Future containing the result of the DeleteResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DeleteResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResolverEndpointResponse> deleteResolverEndpoint(
            DeleteResolverEndpointRequest deleteResolverEndpointRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverEndpointRequest, DeleteResolverEndpointResponse>()
                            .withOperationName("DeleteResolverEndpoint")
                            .withMarshaller(new DeleteResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteResolverEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resolver rule. Before you can delete a resolver rule, you must disassociate it from all the VPCs that
     * you associated the resolver rule with. For more infomation, see <a>DisassociateResolverRule</a>.
     * </p>
     *
     * @param deleteResolverRuleRequest
     * @return A Java Future containing the result of the DeleteResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceInUseException The resource that you tried to update or delete is currently in use.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DeleteResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResolverRuleResponse> deleteResolverRule(DeleteResolverRuleRequest deleteResolverRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverRuleRequest, DeleteResolverRuleResponse>()
                            .withOperationName("DeleteResolverRule")
                            .withMarshaller(new DeleteResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteResolverRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes IP addresses from an inbound or an outbound resolver endpoint. If you want to remove more than one IP
     * address, submit one <code>DisassociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To add an IP address to an endpoint, see <a>AssociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param disassociateResolverEndpointIpAddressRequest
     * @return A Java Future containing the result of the DisassociateResolverEndpointIpAddress operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DisassociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResolverEndpointIpAddressResponse> disassociateResolverEndpointIpAddress(
            DisassociateResolverEndpointIpAddressRequest disassociateResolverEndpointIpAddressRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateResolverEndpointIpAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResolverEndpointIpAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverEndpointIpAddressRequest, DisassociateResolverEndpointIpAddressResponse>()
                            .withOperationName("DisassociateResolverEndpointIpAddress")
                            .withMarshaller(new DisassociateResolverEndpointIpAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateResolverEndpointIpAddressRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association between a specified resolver rule and a specified VPC.
     * </p>
     * <important>
     * <p>
     * If you disassociate a resolver rule from a VPC, Resolver stops forwarding DNS queries for the domain name that
     * you specified in the resolver rule.
     * </p>
     * </important>
     *
     * @param disassociateResolverRuleRequest
     * @return A Java Future containing the result of the DisassociateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DisassociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResolverRuleResponse> disassociateResolverRule(
            DisassociateResolverRuleRequest disassociateResolverRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverRuleRequest, DisassociateResolverRuleResponse>()
                            .withOperationName("DisassociateResolverRule")
                            .withMarshaller(new DisassociateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateResolverRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified resolver endpoint, such as whether it's an inbound or an outbound resolver
     * endpoint, and the current status of the endpoint.
     * </p>
     *
     * @param getResolverEndpointRequest
     * @return A Java Future containing the result of the GetResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverEndpointResponse> getResolverEndpoint(
            GetResolverEndpointRequest getResolverEndpointRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverEndpointRequest, GetResolverEndpointResponse>()
                            .withOperationName("GetResolverEndpoint")
                            .withMarshaller(new GetResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResolverEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified resolver rule, such as the domain name that the rule forwards DNS queries for
     * and the ID of the outbound resolver endpoint that the rule is associated with.
     * </p>
     *
     * @param getResolverRuleRequest
     * @return A Java Future containing the result of the GetResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverRuleResponse> getResolverRule(GetResolverRuleRequest getResolverRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRuleRequest, GetResolverRuleResponse>()
                            .withOperationName("GetResolverRule")
                            .withMarshaller(new GetResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResolverRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about an association between a specified resolver rule and a VPC. You associate a resolver rule
     * and a VPC using <a>AssociateResolverRule</a>.
     * </p>
     *
     * @param getResolverRuleAssociationRequest
     * @return A Java Future containing the result of the GetResolverRuleAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverRuleAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRuleAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverRuleAssociationResponse> getResolverRuleAssociation(
            GetResolverRuleAssociationRequest getResolverRuleAssociationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverRuleAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverRuleAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverRuleAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRuleAssociationRequest, GetResolverRuleAssociationResponse>()
                            .withOperationName("GetResolverRuleAssociation")
                            .withMarshaller(new GetResolverRuleAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResolverRuleAssociationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a resolver rule policy. A resolver rule policy specifies the Resolver operations and
     * resources that you want to allow another AWS account to be able to use.
     * </p>
     *
     * @param getResolverRulePolicyRequest
     * @return A Java Future containing the result of the GetResolverRulePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>UnknownResourceException The specified resource doesn't exist.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverRulePolicyResponse> getResolverRulePolicy(
            GetResolverRulePolicyRequest getResolverRulePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverRulePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverRulePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRulePolicyRequest, GetResolverRulePolicyResponse>()
                            .withOperationName("GetResolverRulePolicy")
                            .withMarshaller(new GetResolverRulePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResolverRulePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the IP addresses for a specified resolver endpoint.
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return A Java Future containing the result of the ListResolverEndpointIpAddresses operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverEndpointIpAddressesResponse> listResolverEndpointIpAddresses(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverEndpointIpAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverEndpointIpAddressesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverEndpointIpAddressesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverEndpointIpAddressesRequest, ListResolverEndpointIpAddressesResponse>()
                            .withOperationName("ListResolverEndpointIpAddresses")
                            .withMarshaller(new ListResolverEndpointIpAddressesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listResolverEndpointIpAddressesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the IP addresses for a specified resolver endpoint.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesPublisher publisher = client.listResolverEndpointIpAddressesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesPublisher publisher = client.listResolverEndpointIpAddressesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverEndpointIpAddressesPublisher listResolverEndpointIpAddressesPaginator(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) {
        return new ListResolverEndpointIpAddressesPublisher(this, applyPaginatorUserAgent(listResolverEndpointIpAddressesRequest));
    }

    /**
     * <p>
     * Lists all the resolver endpoints that were created using the current AWS account.
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return A Java Future containing the result of the ListResolverEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverEndpointsResponse> listResolverEndpoints(
            ListResolverEndpointsRequest listResolverEndpointsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverEndpointsRequest, ListResolverEndpointsResponse>()
                            .withOperationName("ListResolverEndpoints")
                            .withMarshaller(new ListResolverEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listResolverEndpointsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the resolver endpoints that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsPublisher publisher = client.listResolverEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsPublisher publisher = client.listResolverEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverEndpointsPublisher listResolverEndpointsPaginator(ListResolverEndpointsRequest listResolverEndpointsRequest) {
        return new ListResolverEndpointsPublisher(this, applyPaginatorUserAgent(listResolverEndpointsRequest));
    }

    /**
     * <p>
     * Lists the associations that were created between resolver rules and VPCs using the current AWS account.
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return A Java Future containing the result of the ListResolverRuleAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverRuleAssociationsResponse> listResolverRuleAssociations(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverRuleAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverRuleAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverRuleAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverRuleAssociationsRequest, ListResolverRuleAssociationsResponse>()
                            .withOperationName("ListResolverRuleAssociations")
                            .withMarshaller(new ListResolverRuleAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listResolverRuleAssociationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the associations that were created between resolver rules and VPCs using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsPublisher publisher = client.listResolverRuleAssociationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsPublisher publisher = client.listResolverRuleAssociationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverRuleAssociationsPublisher listResolverRuleAssociationsPaginator(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) {
        return new ListResolverRuleAssociationsPublisher(this, applyPaginatorUserAgent(listResolverRuleAssociationsRequest));
    }

    /**
     * <p>
     * Lists the resolver rules that were created using the current AWS account.
     * </p>
     *
     * @param listResolverRulesRequest
     * @return A Java Future containing the result of the ListResolverRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverRulesResponse> listResolverRules(ListResolverRulesRequest listResolverRulesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverRulesRequest, ListResolverRulesResponse>()
                            .withOperationName("ListResolverRules")
                            .withMarshaller(new ListResolverRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listResolverRulesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resolver rules that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesPublisher publisher = client.listResolverRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesPublisher publisher = client.listResolverRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverRulesPublisher listResolverRulesPaginator(ListResolverRulesRequest listResolverRulesRequest) {
        return new ListResolverRulesPublisher(this, applyPaginatorUserAgent(listResolverRulesRequest));
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specifies the Resolver operations and resources that you want to allow another AWS account to be able to use.
     * </p>
     *
     * @param putResolverRulePolicyRequest
     * @return A Java Future containing the result of the PutResolverRulePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPolicyDocumentException The specified resolver rule policy is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>UnknownResourceException The specified resource doesn't exist.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.PutResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/PutResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResolverRulePolicyResponse> putResolverRulePolicy(
            PutResolverRulePolicyRequest putResolverRulePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResolverRulePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResolverRulePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResolverRulePolicyRequest, PutResolverRulePolicyResponse>()
                            .withOperationName("PutResolverRulePolicy")
                            .withMarshaller(new PutResolverRulePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putResolverRulePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidTagException The specified tag is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name of an inbound or an outbound resolver endpoint.
     * </p>
     *
     * @param updateResolverEndpointRequest
     * @return A Java Future containing the result of the UpdateResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UpdateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResolverEndpointResponse> updateResolverEndpoint(
            UpdateResolverEndpointRequest updateResolverEndpointRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverEndpointRequest, UpdateResolverEndpointResponse>()
                            .withOperationName("UpdateResolverEndpoint")
                            .withMarshaller(new UpdateResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateResolverEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings for a specified resolver rule. <code>ResolverRuleId</code> is required, and all other parameters
     * are optional. If you don't specify a parameter, it retains its current value.
     * </p>
     *
     * @param updateResolverRuleRequest
     * @return A Java Future containing the result of the UpdateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceUnavailableException The specified resource isn't available.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UpdateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResolverRuleResponse> updateResolverRule(UpdateResolverRuleRequest updateResolverRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverRuleRequest, UpdateResolverRuleResponse>()
                            .withOperationName("UpdateResolverRule")
                            .withMarshaller(new UpdateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateResolverRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53ResolverException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyDocument")
                                .exceptionBuilderSupplier(InvalidPolicyDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private <T extends Route53ResolverRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
