/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * For queries that originate in your VPC, detailed information about a resolver rule, which specifies how to route DNS
 * queries out of the VPC. The <code>ResolverRule</code> parameter appears in the response to a
 * <a>CreateResolverRule</a>, <a>DeleteResolverRule</a>, <a>GetResolverRule</a>, <a>ListResolverRules</a>, or
 * <a>UpdateResolverRule</a> request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResolverRule implements SdkPojo, Serializable, ToCopyableBuilder<ResolverRule.Builder, ResolverRule> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> CREATOR_REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::creatorRequestId)).setter(setter(Builder::creatorRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatorRequestId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> RULE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::ruleTypeAsString)).setter(setter(Builder::ruleType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleType").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<TargetAddress>> TARGET_IPS_FIELD = SdkField
            .<List<TargetAddress>> builder(MarshallingType.LIST)
            .getter(getter(ResolverRule::targetIps))
            .setter(setter(Builder::targetIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetAddress> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetAddress::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOLVER_ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::resolverEndpointId)).setter(setter(Builder::resolverEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolverEndpointId").build())
            .build();

    private static final SdkField<String> OWNER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::ownerId)).setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId").build()).build();

    private static final SdkField<String> SHARE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ResolverRule::shareStatusAsString)).setter(setter(Builder::shareStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShareStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            CREATOR_REQUEST_ID_FIELD, ARN_FIELD, DOMAIN_NAME_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD, RULE_TYPE_FIELD,
            NAME_FIELD, TARGET_IPS_FIELD, RESOLVER_ENDPOINT_ID_FIELD, OWNER_ID_FIELD, SHARE_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String creatorRequestId;

    private final String arn;

    private final String domainName;

    private final String status;

    private final String statusMessage;

    private final String ruleType;

    private final String name;

    private final List<TargetAddress> targetIps;

    private final String resolverEndpointId;

    private final String ownerId;

    private final String shareStatus;

    private ResolverRule(BuilderImpl builder) {
        this.id = builder.id;
        this.creatorRequestId = builder.creatorRequestId;
        this.arn = builder.arn;
        this.domainName = builder.domainName;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
        this.ruleType = builder.ruleType;
        this.name = builder.name;
        this.targetIps = builder.targetIps;
        this.resolverEndpointId = builder.resolverEndpointId;
        this.ownerId = builder.ownerId;
        this.shareStatus = builder.shareStatus;
    }

    /**
     * <p>
     * The ID that Resolver assigned to the resolver rule when you created it.
     * </p>
     * 
     * @return The ID that Resolver assigned to the resolver rule when you created it.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * A unique string that you specified when you created the resolver rule. <code>CreatorRequestId</code>identifies
     * the request and allows failed requests to be retried without the risk of executing the operation twice.
     * </p>
     * 
     * @return A unique string that you specified when you created the resolver rule. <code>CreatorRequestId</code>
     *         identifies the request and allows failed requests to be retried without the risk of executing the
     *         operation twice.
     */
    public String creatorRequestId() {
        return creatorRequestId;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) for the resolver rule specified by <code>Id</code>.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) for the resolver rule specified by <code>Id</code>.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * DNS queries for this domain name are forwarded to the IP addresses that are specified in <code>TargetIps</code>.
     * If a query matches multiple resolver rules (example.com and www.example.com), the query is routed using the
     * resolver rule that contains the most specific domain name (www.example.com).
     * </p>
     * 
     * @return DNS queries for this domain name are forwarded to the IP addresses that are specified in
     *         <code>TargetIps</code>. If a query matches multiple resolver rules (example.com and www.example.com), the
     *         query is routed using the resolver rule that contains the most specific domain name (www.example.com).
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * A code that specifies the current status of the resolver rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResolverRuleStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return A code that specifies the current status of the resolver rule.
     * @see ResolverRuleStatus
     */
    public ResolverRuleStatus status() {
        return ResolverRuleStatus.fromValue(status);
    }

    /**
     * <p>
     * A code that specifies the current status of the resolver rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResolverRuleStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return A code that specifies the current status of the resolver rule.
     * @see ResolverRuleStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A detailed description of the status of a resolver rule.
     * </p>
     * 
     * @return A detailed description of the status of a resolver rule.
     */
    public String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleType} will
     * return {@link RuleTypeOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ruleTypeAsString}.
     * </p>
     * 
     * @return This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
     * @see RuleTypeOption
     */
    public RuleTypeOption ruleType() {
        return RuleTypeOption.fromValue(ruleType);
    }

    /**
     * <p>
     * This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleType} will
     * return {@link RuleTypeOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ruleTypeAsString}.
     * </p>
     * 
     * @return This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
     * @see RuleTypeOption
     */
    public String ruleTypeAsString() {
        return ruleType;
    }

    /**
     * <p>
     * The name for the resolver rule, which you specified when you created the resolver rule.
     * </p>
     * 
     * @return The name for the resolver rule, which you specified when you created the resolver rule.
     */
    public String name() {
        return name;
    }

    /**
     * Returns true if the TargetIps property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTargetIps() {
        return targetIps != null && !(targetIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array that contains the IP addresses and ports that you want to forward
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetIps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array that contains the IP addresses and ports that you want to forward
     */
    public List<TargetAddress> targetIps() {
        return targetIps;
    }

    /**
     * <p>
     * The ID of the endpoint that the rule is associated with.
     * </p>
     * 
     * @return The ID of the endpoint that the rule is associated with.
     */
    public String resolverEndpointId() {
        return resolverEndpointId;
    }

    /**
     * <p>
     * When a rule is shared with another AWS account, the account ID of the account that the rule is shared with.
     * </p>
     * 
     * @return When a rule is shared with another AWS account, the account ID of the account that the rule is shared
     *         with.
     */
    public String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * Whether the rules is shared and, if so, whether the current account is sharing the rule with another account, or
     * another account is sharing the rule with the current account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shareStatus} will
     * return {@link ShareStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shareStatusAsString}.
     * </p>
     * 
     * @return Whether the rules is shared and, if so, whether the current account is sharing the rule with another
     *         account, or another account is sharing the rule with the current account.
     * @see ShareStatus
     */
    public ShareStatus shareStatus() {
        return ShareStatus.fromValue(shareStatus);
    }

    /**
     * <p>
     * Whether the rules is shared and, if so, whether the current account is sharing the rule with another account, or
     * another account is sharing the rule with the current account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shareStatus} will
     * return {@link ShareStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shareStatusAsString}.
     * </p>
     * 
     * @return Whether the rules is shared and, if so, whether the current account is sharing the rule with another
     *         account, or another account is sharing the rule with the current account.
     * @see ShareStatus
     */
    public String shareStatusAsString() {
        return shareStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(creatorRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(ruleTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(targetIps());
        hashCode = 31 * hashCode + Objects.hashCode(resolverEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(shareStatusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResolverRule)) {
            return false;
        }
        ResolverRule other = (ResolverRule) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(creatorRequestId(), other.creatorRequestId())
                && Objects.equals(arn(), other.arn()) && Objects.equals(domainName(), other.domainName())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(ruleTypeAsString(), other.ruleTypeAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(targetIps(), other.targetIps())
                && Objects.equals(resolverEndpointId(), other.resolverEndpointId()) && Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(shareStatusAsString(), other.shareStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ResolverRule").add("Id", id()).add("CreatorRequestId", creatorRequestId()).add("Arn", arn())
                .add("DomainName", domainName()).add("Status", statusAsString()).add("StatusMessage", statusMessage())
                .add("RuleType", ruleTypeAsString()).add("Name", name()).add("TargetIps", targetIps())
                .add("ResolverEndpointId", resolverEndpointId()).add("OwnerId", ownerId())
                .add("ShareStatus", shareStatusAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "CreatorRequestId":
            return Optional.ofNullable(clazz.cast(creatorRequestId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "RuleType":
            return Optional.ofNullable(clazz.cast(ruleTypeAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "TargetIps":
            return Optional.ofNullable(clazz.cast(targetIps()));
        case "ResolverEndpointId":
            return Optional.ofNullable(clazz.cast(resolverEndpointId()));
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "ShareStatus":
            return Optional.ofNullable(clazz.cast(shareStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResolverRule, T> g) {
        return obj -> g.apply((ResolverRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResolverRule> {
        /**
         * <p>
         * The ID that Resolver assigned to the resolver rule when you created it.
         * </p>
         * 
         * @param id
         *        The ID that Resolver assigned to the resolver rule when you created it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A unique string that you specified when you created the resolver rule. <code>CreatorRequestId</code>
         * identifies the request and allows failed requests to be retried without the risk of executing the operation
         * twice.
         * </p>
         * 
         * @param creatorRequestId
         *        A unique string that you specified when you created the resolver rule. <code>CreatorRequestId</code>
         *        identifies the request and allows failed requests to be retried without the risk of executing the
         *        operation twice.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorRequestId(String creatorRequestId);

        /**
         * <p>
         * The ARN (Amazon Resource Name) for the resolver rule specified by <code>Id</code>.
         * </p>
         * 
         * @param arn
         *        The ARN (Amazon Resource Name) for the resolver rule specified by <code>Id</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * DNS queries for this domain name are forwarded to the IP addresses that are specified in
         * <code>TargetIps</code>. If a query matches multiple resolver rules (example.com and www.example.com), the
         * query is routed using the resolver rule that contains the most specific domain name (www.example.com).
         * </p>
         * 
         * @param domainName
         *        DNS queries for this domain name are forwarded to the IP addresses that are specified in
         *        <code>TargetIps</code>. If a query matches multiple resolver rules (example.com and www.example.com),
         *        the query is routed using the resolver rule that contains the most specific domain name
         *        (www.example.com).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * A code that specifies the current status of the resolver rule.
         * </p>
         * 
         * @param status
         *        A code that specifies the current status of the resolver rule.
         * @see ResolverRuleStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverRuleStatus
         */
        Builder status(String status);

        /**
         * <p>
         * A code that specifies the current status of the resolver rule.
         * </p>
         * 
         * @param status
         *        A code that specifies the current status of the resolver rule.
         * @see ResolverRuleStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResolverRuleStatus
         */
        Builder status(ResolverRuleStatus status);

        /**
         * <p>
         * A detailed description of the status of a resolver rule.
         * </p>
         * 
         * @param statusMessage
         *        A detailed description of the status of a resolver rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
         * </p>
         * 
         * @param ruleType
         *        This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
         * @see RuleTypeOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleTypeOption
         */
        Builder ruleType(String ruleType);

        /**
         * <p>
         * This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
         * </p>
         * 
         * @param ruleType
         *        This value is always <code>FORWARD</code>. Other resolver rule types aren't supported.
         * @see RuleTypeOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleTypeOption
         */
        Builder ruleType(RuleTypeOption ruleType);

        /**
         * <p>
         * The name for the resolver rule, which you specified when you created the resolver rule.
         * </p>
         * 
         * @param name
         *        The name for the resolver rule, which you specified when you created the resolver rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An array that contains the IP addresses and ports that you want to forward
         * </p>
         * 
         * @param targetIps
         *        An array that contains the IP addresses and ports that you want to forward
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetIps(Collection<TargetAddress> targetIps);

        /**
         * <p>
         * An array that contains the IP addresses and ports that you want to forward
         * </p>
         * 
         * @param targetIps
         *        An array that contains the IP addresses and ports that you want to forward
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetIps(TargetAddress... targetIps);

        /**
         * <p>
         * An array that contains the IP addresses and ports that you want to forward
         * </p>
         * This is a convenience that creates an instance of the {@link List<TargetAddress>.Builder} avoiding the need
         * to create one manually via {@link List<TargetAddress>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TargetAddress>.Builder#build()} is called immediately and
         * its result is passed to {@link #targetIps(List<TargetAddress>)}.
         * 
         * @param targetIps
         *        a consumer that will call methods on {@link List<TargetAddress>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetIps(List<TargetAddress>)
         */
        Builder targetIps(Consumer<TargetAddress.Builder>... targetIps);

        /**
         * <p>
         * The ID of the endpoint that the rule is associated with.
         * </p>
         * 
         * @param resolverEndpointId
         *        The ID of the endpoint that the rule is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolverEndpointId(String resolverEndpointId);

        /**
         * <p>
         * When a rule is shared with another AWS account, the account ID of the account that the rule is shared with.
         * </p>
         * 
         * @param ownerId
         *        When a rule is shared with another AWS account, the account ID of the account that the rule is shared
         *        with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * Whether the rules is shared and, if so, whether the current account is sharing the rule with another account,
         * or another account is sharing the rule with the current account.
         * </p>
         * 
         * @param shareStatus
         *        Whether the rules is shared and, if so, whether the current account is sharing the rule with another
         *        account, or another account is sharing the rule with the current account.
         * @see ShareStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShareStatus
         */
        Builder shareStatus(String shareStatus);

        /**
         * <p>
         * Whether the rules is shared and, if so, whether the current account is sharing the rule with another account,
         * or another account is sharing the rule with the current account.
         * </p>
         * 
         * @param shareStatus
         *        Whether the rules is shared and, if so, whether the current account is sharing the rule with another
         *        account, or another account is sharing the rule with the current account.
         * @see ShareStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShareStatus
         */
        Builder shareStatus(ShareStatus shareStatus);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String creatorRequestId;

        private String arn;

        private String domainName;

        private String status;

        private String statusMessage;

        private String ruleType;

        private String name;

        private List<TargetAddress> targetIps = DefaultSdkAutoConstructList.getInstance();

        private String resolverEndpointId;

        private String ownerId;

        private String shareStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(ResolverRule model) {
            id(model.id);
            creatorRequestId(model.creatorRequestId);
            arn(model.arn);
            domainName(model.domainName);
            status(model.status);
            statusMessage(model.statusMessage);
            ruleType(model.ruleType);
            name(model.name);
            targetIps(model.targetIps);
            resolverEndpointId(model.resolverEndpointId);
            ownerId(model.ownerId);
            shareStatus(model.shareStatus);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getCreatorRequestId() {
            return creatorRequestId;
        }

        @Override
        public final Builder creatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
            return this;
        }

        public final void setCreatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResolverRuleStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        public final String getRuleTypeAsString() {
            return ruleType;
        }

        @Override
        public final Builder ruleType(String ruleType) {
            this.ruleType = ruleType;
            return this;
        }

        @Override
        public final Builder ruleType(RuleTypeOption ruleType) {
            this.ruleType(ruleType == null ? null : ruleType.toString());
            return this;
        }

        public final void setRuleType(String ruleType) {
            this.ruleType = ruleType;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<TargetAddress.Builder> getTargetIps() {
            return targetIps != null ? targetIps.stream().map(TargetAddress::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targetIps(Collection<TargetAddress> targetIps) {
            this.targetIps = TargetListCopier.copy(targetIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetIps(TargetAddress... targetIps) {
            targetIps(Arrays.asList(targetIps));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetIps(Consumer<TargetAddress.Builder>... targetIps) {
            targetIps(Stream.of(targetIps).map(c -> TargetAddress.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTargetIps(Collection<TargetAddress.BuilderImpl> targetIps) {
            this.targetIps = TargetListCopier.copyFromBuilder(targetIps);
        }

        public final String getResolverEndpointId() {
            return resolverEndpointId;
        }

        @Override
        public final Builder resolverEndpointId(String resolverEndpointId) {
            this.resolverEndpointId = resolverEndpointId;
            return this;
        }

        public final void setResolverEndpointId(String resolverEndpointId) {
            this.resolverEndpointId = resolverEndpointId;
        }

        public final String getOwnerId() {
            return ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        public final String getShareStatusAsString() {
            return shareStatus;
        }

        @Override
        public final Builder shareStatus(String shareStatus) {
            this.shareStatus = shareStatus;
            return this;
        }

        @Override
        public final Builder shareStatus(ShareStatus shareStatus) {
            this.shareStatus(shareStatus == null ? null : shareStatus.toString());
            return this;
        }

        public final void setShareStatus(String shareStatus) {
            this.shareStatus = shareStatus;
        }

        @Override
        public ResolverRule build() {
            return new ResolverRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
