/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.retries.internal;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.retries.api.AcquireInitialTokenRequest;
import software.amazon.awssdk.retries.api.AcquireInitialTokenResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.retries.api.RecordSuccessRequest;
import software.amazon.awssdk.retries.api.RecordSuccessResponse;
import software.amazon.awssdk.retries.api.RefreshRetryTokenRequest;
import software.amazon.awssdk.retries.api.RefreshRetryTokenResponse;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.retries.api.RetryToken;
import software.amazon.awssdk.retries.api.TokenAcquisitionFailedException;
import software.amazon.awssdk.retries.api.internal.RefreshRetryTokenResponseImpl;
import software.amazon.awssdk.retries.internal.DefaultRetryToken;
import software.amazon.awssdk.retries.internal.circuitbreaker.AcquireResponse;
import software.amazon.awssdk.retries.internal.circuitbreaker.ReleaseResponse;
import software.amazon.awssdk.retries.internal.circuitbreaker.TokenBucket;
import software.amazon.awssdk.retries.internal.circuitbreaker.TokenBucketStore;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public abstract class BaseRetryStrategy
implements RetryStrategy {
    protected final Logger log;
    protected final List<Predicate<Throwable>> retryPredicates;
    protected final int maxAttempts;
    protected final boolean circuitBreakerEnabled;
    protected final BackoffStrategy backoffStrategy;
    protected final BackoffStrategy throttlingBackoffStrategy;
    protected final Predicate<Throwable> treatAsThrottling;
    protected final int exceptionCost;
    protected final TokenBucketStore tokenBucketStore;

    BaseRetryStrategy(Logger log, Builder builder) {
        this.log = log;
        this.retryPredicates = Collections.unmodifiableList((List)Validate.paramNotNull((Object)builder.retryPredicates, (String)"retryPredicates"));
        this.maxAttempts = Validate.isPositive((int)builder.maxAttempts, (String)"maxAttempts");
        this.circuitBreakerEnabled = builder.circuitBreakerEnabled == null || builder.circuitBreakerEnabled != false;
        this.backoffStrategy = (BackoffStrategy)Validate.paramNotNull((Object)builder.backoffStrategy, (String)"backoffStrategy");
        this.throttlingBackoffStrategy = (BackoffStrategy)Validate.paramNotNull((Object)builder.throttlingBackoffStrategy, (String)"throttlingBackoffStrategy");
        this.treatAsThrottling = (Predicate)Validate.paramNotNull((Object)builder.treatAsThrottling, (String)"treatAsThrottling");
        this.exceptionCost = (Integer)Validate.paramNotNull((Object)builder.exceptionCost, (String)"exceptionCost");
        this.tokenBucketStore = (TokenBucketStore)Validate.paramNotNull((Object)builder.tokenBucketStore, (String)"tokenBucketStore");
    }

    public final AcquireInitialTokenResponse acquireInitialToken(AcquireInitialTokenRequest request) {
        this.logAcquireInitialToken(request);
        DefaultRetryToken token = DefaultRetryToken.builder().scope(request.scope()).build();
        return AcquireInitialTokenResponse.create((RetryToken)token, (Duration)this.computeInitialBackoff(request));
    }

    public final RefreshRetryTokenResponse refreshRetryToken(RefreshRetryTokenRequest request) {
        DefaultRetryToken token = BaseRetryStrategy.asDefaultRetryToken(request.token());
        this.throwOnNonRetryableException(request);
        this.throwOnMaxAttemptsReached(request);
        AcquireResponse acquireResponse = this.requestAcquireCapacity(request, token);
        this.throwOnAcquisitionFailure(request, acquireResponse);
        this.updateStateForRetry(request);
        DefaultRetryToken refreshedToken = this.refreshToken(request, acquireResponse);
        Duration backoff = this.computeBackoff(request, refreshedToken);
        this.logRefreshTokenSuccess(refreshedToken, acquireResponse, backoff);
        return RefreshRetryTokenResponseImpl.create((RetryToken)refreshedToken, (Duration)backoff);
    }

    public final RecordSuccessResponse recordSuccess(RecordSuccessRequest request) {
        DefaultRetryToken token = BaseRetryStrategy.asDefaultRetryToken(request.token());
        ReleaseResponse releaseResponse = this.releaseTokenBucketCapacity(token);
        DefaultRetryToken refreshedToken = this.refreshRetryTokenAfterSuccess(token, releaseResponse);
        this.updateStateForSuccess(token);
        this.logRecordSuccess(token, releaseResponse);
        return RecordSuccessResponse.create((RetryToken)refreshedToken);
    }

    public int maxAttempts() {
        return this.maxAttempts;
    }

    protected Duration computeInitialBackoff(AcquireInitialTokenRequest request) {
        return Duration.ZERO;
    }

    protected Duration computeBackoff(RefreshRetryTokenRequest request, DefaultRetryToken token) {
        Duration backoff = this.treatAsThrottling.test(request.failure()) ? this.throttlingBackoffStrategy.computeDelay(token.attempt()) : this.backoffStrategy.computeDelay(token.attempt());
        Duration suggested = request.suggestedDelay().orElse(Duration.ZERO);
        return BaseRetryStrategy.maxOf(suggested, backoff);
    }

    protected void updateStateForSuccess(DefaultRetryToken token) {
    }

    protected void updateStateForRetry(RefreshRetryTokenRequest request) {
    }

    protected int exceptionCost(RefreshRetryTokenRequest request) {
        if (this.circuitBreakerEnabled) {
            return this.exceptionCost;
        }
        return 0;
    }

    public final boolean hasRetryPredicates() {
        return !this.retryPredicates.isEmpty();
    }

    private DefaultRetryToken refreshToken(RefreshRetryTokenRequest request, AcquireResponse acquireResponse) {
        DefaultRetryToken token = BaseRetryStrategy.asDefaultRetryToken(request.token());
        return token.toBuilder().increaseAttempt().state(DefaultRetryToken.TokenState.IN_PROGRESS).capacityAcquired(acquireResponse.capacityAcquired()).capacityRemaining(acquireResponse.capacityRemaining()).addFailure(request.failure()).build();
    }

    private AcquireResponse requestAcquireCapacity(RefreshRetryTokenRequest request, DefaultRetryToken token) {
        TokenBucket tokenBucket = this.tokenBucketStore.tokenBucketForScope(token.scope());
        return tokenBucket.tryAcquire(this.exceptionCost(request));
    }

    private ReleaseResponse releaseTokenBucketCapacity(DefaultRetryToken token) {
        TokenBucket bucket = this.tokenBucketStore.tokenBucketForScope(token.scope());
        int capacityReleased = Math.max(token.capacityAcquired(), 1);
        return bucket.release(capacityReleased);
    }

    private DefaultRetryToken refreshRetryTokenAfterSuccess(DefaultRetryToken token, ReleaseResponse releaseResponse) {
        return token.toBuilder().capacityRemaining(releaseResponse.currentCapacity()).state(DefaultRetryToken.TokenState.SUCCEEDED).build();
    }

    private void throwOnMaxAttemptsReached(RefreshRetryTokenRequest request) {
        DefaultRetryToken token = BaseRetryStrategy.asDefaultRetryToken(request.token());
        if (this.maxAttemptsReached(token)) {
            Throwable failure = request.failure();
            TokenBucket tokenBucket = this.tokenBucketStore.tokenBucketForScope(token.scope());
            DefaultRetryToken refreshedToken = token.toBuilder().capacityRemaining(tokenBucket.currentCapacity()).state(DefaultRetryToken.TokenState.MAX_RETRIES_REACHED).addFailure(failure).build();
            String message = this.maxAttemptsReachedMessage(refreshedToken);
            this.log.debug(() -> message, failure);
            throw new TokenAcquisitionFailedException(message, (RetryToken)refreshedToken, failure);
        }
    }

    private void throwOnNonRetryableException(RefreshRetryTokenRequest request) {
        DefaultRetryToken token = BaseRetryStrategy.asDefaultRetryToken(request.token());
        Throwable failure = request.failure();
        if (this.isNonRetryableException(request)) {
            String message = this.nonRetryableExceptionMessage(token);
            this.log.debug(() -> message, failure);
            TokenBucket tokenBucket = this.tokenBucketStore.tokenBucketForScope(token.scope());
            DefaultRetryToken refreshedToken = token.toBuilder().capacityRemaining(tokenBucket.currentCapacity()).state(DefaultRetryToken.TokenState.NON_RETRYABLE_EXCEPTION).addFailure(failure).build();
            throw new TokenAcquisitionFailedException(message, (RetryToken)refreshedToken, failure);
        }
        int attempt = token.attempt();
        this.log.debug(() -> String.format("Request attempt %d encountered retryable failure.", attempt), failure);
    }

    private void throwOnAcquisitionFailure(RefreshRetryTokenRequest request, AcquireResponse acquireResponse) {
        DefaultRetryToken token = BaseRetryStrategy.asDefaultRetryToken(request.token());
        if (acquireResponse.acquisitionFailed()) {
            Throwable failure = request.failure();
            DefaultRetryToken refreshedToken = token.toBuilder().capacityRemaining(acquireResponse.capacityRemaining()).capacityAcquired(acquireResponse.capacityAcquired()).state(DefaultRetryToken.TokenState.TOKEN_ACQUISITION_FAILED).addFailure(failure).build();
            String message = this.acquisitionFailedMessage(acquireResponse);
            this.log.debug(() -> message, failure);
            throw new TokenAcquisitionFailedException(message, (RetryToken)refreshedToken, failure);
        }
    }

    private String nonRetryableExceptionMessage(DefaultRetryToken token) {
        return String.format("Request attempt %d encountered non-retryable failure", token.attempt());
    }

    private String maxAttemptsReachedMessage(DefaultRetryToken token) {
        return String.format("Request will not be retried. Retries have been exhausted (cost: 0, capacity: %d/%d)", token.capacityAcquired(), token.capacityRemaining());
    }

    private String acquisitionFailedMessage(AcquireResponse response) {
        return String.format("Request will not be retried to protect the caller and downstream service. The cost of retrying (%d) exceeds the available retry capacity (%d/%d).", response.capacityRequested(), response.capacityRemaining(), response.maxCapacity());
    }

    private void logAcquireInitialToken(AcquireInitialTokenRequest request) {
        TokenBucket tokenBucket = this.tokenBucketStore.tokenBucketForScope(request.scope());
        this.log.debug(() -> String.format("Request attempt 1 token acquired (backoff: 0ms, cost: 0, capacity: %d/%d)", tokenBucket.currentCapacity(), tokenBucket.maxCapacity()));
    }

    private void logRefreshTokenSuccess(DefaultRetryToken token, AcquireResponse acquireResponse, Duration delay) {
        this.log.debug(() -> String.format("Request attempt %d token acquired (backoff: %dms, cost: %d, capacity: %d/%d)", token.attempt(), delay.toMillis(), acquireResponse.capacityAcquired(), acquireResponse.capacityRemaining(), acquireResponse.maxCapacity()));
    }

    private void logRecordSuccess(DefaultRetryToken token, ReleaseResponse release) {
        this.log.debug(() -> String.format("Request attempt %d succeeded (cost: -%d, capacity: %d/%d)", token.attempt(), release.capacityReleased(), release.currentCapacity(), release.maxCapacity()));
    }

    private boolean maxAttemptsReached(DefaultRetryToken token) {
        return token.attempt() >= this.maxAttempts;
    }

    private boolean isNonRetryableException(RefreshRetryTokenRequest request) {
        Throwable failure = request.failure();
        for (Predicate<Throwable> retryPredicate : this.retryPredicates) {
            if (!retryPredicate.test(failure)) continue;
            return false;
        }
        return true;
    }

    static Duration maxOf(Duration left, Duration right) {
        if (left.compareTo(right) >= 0) {
            return left;
        }
        return right;
    }

    static DefaultRetryToken asDefaultRetryToken(RetryToken token) {
        return (DefaultRetryToken)Validate.isInstanceOf(DefaultRetryToken.class, (Object)token, (String)"RetryToken is of unexpected class (%s), This token was not created by this retry strategy.", (Object[])new Object[]{token.getClass().getName()});
    }

    static class Builder {
        private List<Predicate<Throwable>> retryPredicates;
        private int maxAttempts;
        private Boolean circuitBreakerEnabled;
        private Integer exceptionCost;
        private BackoffStrategy backoffStrategy;
        private BackoffStrategy throttlingBackoffStrategy;
        private Predicate<Throwable> treatAsThrottling = throwable -> false;
        private TokenBucketStore tokenBucketStore;

        Builder() {
            this.retryPredicates = new ArrayList<Predicate<Throwable>>();
        }

        Builder(BaseRetryStrategy strategy) {
            this.retryPredicates = new ArrayList<Predicate<Throwable>>(strategy.retryPredicates);
            this.maxAttempts = strategy.maxAttempts;
            this.circuitBreakerEnabled = strategy.circuitBreakerEnabled;
            this.exceptionCost = strategy.exceptionCost;
            this.backoffStrategy = strategy.backoffStrategy;
            this.throttlingBackoffStrategy = strategy.throttlingBackoffStrategy;
            this.treatAsThrottling = strategy.treatAsThrottling;
            this.tokenBucketStore = strategy.tokenBucketStore;
        }

        void setRetryOnException(Predicate<Throwable> shouldRetry) {
            this.retryPredicates.add(shouldRetry);
        }

        void setMaxAttempts(int maxAttempts) {
            this.maxAttempts = maxAttempts;
        }

        void setTokenBucketStore(TokenBucketStore tokenBucketStore) {
            this.tokenBucketStore = tokenBucketStore;
        }

        void setCircuitBreakerEnabled(Boolean enabled) {
            this.circuitBreakerEnabled = enabled;
        }

        void setBackoffStrategy(BackoffStrategy backoffStrategy) {
            this.backoffStrategy = backoffStrategy;
        }

        void setThrottlingBackoffStrategy(BackoffStrategy throttlingBackoffStrategy) {
            this.throttlingBackoffStrategy = throttlingBackoffStrategy;
        }

        void setTreatAsThrottling(Predicate<Throwable> treatAsThrottling) {
            this.treatAsThrottling = treatAsThrottling;
        }

        void setTokenBucketExceptionCost(int exceptionCost) {
            this.exceptionCost = exceptionCost;
        }
    }
}

