/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.resourceexplorer2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A resource in Amazon Web Services that Amazon Web Services Resource Explorer has discovered, and for which it has
 * stored information in the index of the Amazon Web Services Region that contains the resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Resource implements SdkPojo, Serializable, ToCopyableBuilder<Resource.Builder, Resource> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Resource::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> OWNING_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwningAccountId").getter(getter(Resource::owningAccountId)).setter(setter(Builder::owningAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwningAccountId").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(Resource::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(Resource::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Service")
            .getter(getter(Resource::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Service").build()).build();

    private static final SdkField<Instant> LAST_REPORTED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastReportedAt")
            .getter(getter(Resource::lastReportedAt))
            .setter(setter(Builder::lastReportedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastReportedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<ResourceProperty>> PROPERTIES_FIELD = SdkField
            .<List<ResourceProperty>> builder(MarshallingType.LIST)
            .memberName("Properties")
            .getter(getter(Resource::properties))
            .setter(setter(Builder::properties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Properties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            OWNING_ACCOUNT_ID_FIELD, REGION_FIELD, RESOURCE_TYPE_FIELD, SERVICE_FIELD, LAST_REPORTED_AT_FIELD, PROPERTIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String owningAccountId;

    private final String region;

    private final String resourceType;

    private final String service;

    private final Instant lastReportedAt;

    private final List<ResourceProperty> properties;

    private Resource(BuilderImpl builder) {
        this.arn = builder.arn;
        this.owningAccountId = builder.owningAccountId;
        this.region = builder.region;
        this.resourceType = builder.resourceType;
        this.service = builder.service;
        this.lastReportedAt = builder.lastReportedAt;
        this.properties = builder.properties;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon resource name
     * (ARN)</a> of the resource.
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon resource
     *         name (ARN)</a> of the resource.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Amazon Web Services account that owns the resource.
     * </p>
     * 
     * @return The Amazon Web Services account that owns the resource.
     */
    public final String owningAccountId() {
        return owningAccountId;
    }

    /**
     * <p>
     * The Amazon Web Services Region in which the resource was created and exists.
     * </p>
     * 
     * @return The Amazon Web Services Region in which the resource was created and exists.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The type of the resource.
     * </p>
     * 
     * @return The type of the resource.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The Amazon Web Services service that owns the resource and is responsible for creating and updating it.
     * </p>
     * 
     * @return The Amazon Web Services service that owns the resource and is responsible for creating and updating it.
     */
    public final String service() {
        return service;
    }

    /**
     * <p>
     * The date and time that Resource Explorer last queried this resource and updated the index with the latest
     * information about the resource.
     * </p>
     * 
     * @return The date and time that Resource Explorer last queried this resource and updated the index with the latest
     *         information about the resource.
     */
    public final Instant lastReportedAt() {
        return lastReportedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the Properties property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProperties() {
        return properties != null && !(properties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A structure with additional type-specific details about the resource. These properties can be added by turning on
     * integration between Resource Explorer and other Amazon Web Services services.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProperties} method.
     * </p>
     * 
     * @return A structure with additional type-specific details about the resource. These properties can be added by
     *         turning on integration between Resource Explorer and other Amazon Web Services services.
     */
    public final List<ResourceProperty> properties() {
        return properties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(owningAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(lastReportedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasProperties() ? properties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Resource)) {
            return false;
        }
        Resource other = (Resource) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(owningAccountId(), other.owningAccountId())
                && Objects.equals(region(), other.region()) && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(service(), other.service()) && Objects.equals(lastReportedAt(), other.lastReportedAt())
                && hasProperties() == other.hasProperties() && Objects.equals(properties(), other.properties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Resource").add("Arn", arn()).add("OwningAccountId", owningAccountId()).add("Region", region())
                .add("ResourceType", resourceType()).add("Service", service()).add("LastReportedAt", lastReportedAt())
                .add("Properties", hasProperties() ? properties() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "OwningAccountId":
            return Optional.ofNullable(clazz.cast(owningAccountId()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "Service":
            return Optional.ofNullable(clazz.cast(service()));
        case "LastReportedAt":
            return Optional.ofNullable(clazz.cast(lastReportedAt()));
        case "Properties":
            return Optional.ofNullable(clazz.cast(properties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Arn", ARN_FIELD);
        map.put("OwningAccountId", OWNING_ACCOUNT_ID_FIELD);
        map.put("Region", REGION_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("Service", SERVICE_FIELD);
        map.put("LastReportedAt", LAST_REPORTED_AT_FIELD);
        map.put("Properties", PROPERTIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Resource, T> g) {
        return obj -> g.apply((Resource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Resource> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon resource name
         * (ARN)</a> of the resource.
         * </p>
         * 
         * @param arn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         *        resource name (ARN)</a> of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Amazon Web Services account that owns the resource.
         * </p>
         * 
         * @param owningAccountId
         *        The Amazon Web Services account that owns the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owningAccountId(String owningAccountId);

        /**
         * <p>
         * The Amazon Web Services Region in which the resource was created and exists.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region in which the resource was created and exists.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The type of the resource.
         * </p>
         * 
         * @param resourceType
         *        The type of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Amazon Web Services service that owns the resource and is responsible for creating and updating it.
         * </p>
         * 
         * @param service
         *        The Amazon Web Services service that owns the resource and is responsible for creating and updating
         *        it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The date and time that Resource Explorer last queried this resource and updated the index with the latest
         * information about the resource.
         * </p>
         * 
         * @param lastReportedAt
         *        The date and time that Resource Explorer last queried this resource and updated the index with the
         *        latest information about the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastReportedAt(Instant lastReportedAt);

        /**
         * <p>
         * A structure with additional type-specific details about the resource. These properties can be added by
         * turning on integration between Resource Explorer and other Amazon Web Services services.
         * </p>
         * 
         * @param properties
         *        A structure with additional type-specific details about the resource. These properties can be added by
         *        turning on integration between Resource Explorer and other Amazon Web Services services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder properties(Collection<ResourceProperty> properties);

        /**
         * <p>
         * A structure with additional type-specific details about the resource. These properties can be added by
         * turning on integration between Resource Explorer and other Amazon Web Services services.
         * </p>
         * 
         * @param properties
         *        A structure with additional type-specific details about the resource. These properties can be added by
         *        turning on integration between Resource Explorer and other Amazon Web Services services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder properties(ResourceProperty... properties);

        /**
         * <p>
         * A structure with additional type-specific details about the resource. These properties can be added by
         * turning on integration between Resource Explorer and other Amazon Web Services services.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.resourceexplorer2.model.ResourceProperty.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.resourceexplorer2.model.ResourceProperty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.resourceexplorer2.model.ResourceProperty.Builder#build()} is called
         * immediately and its result is passed to {@link #properties(List<ResourceProperty>)}.
         * 
         * @param properties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.resourceexplorer2.model.ResourceProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #properties(java.util.Collection<ResourceProperty>)
         */
        Builder properties(Consumer<ResourceProperty.Builder>... properties);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String owningAccountId;

        private String region;

        private String resourceType;

        private String service;

        private Instant lastReportedAt;

        private List<ResourceProperty> properties = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Resource model) {
            arn(model.arn);
            owningAccountId(model.owningAccountId);
            region(model.region);
            resourceType(model.resourceType);
            service(model.service);
            lastReportedAt(model.lastReportedAt);
            properties(model.properties);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getOwningAccountId() {
            return owningAccountId;
        }

        public final void setOwningAccountId(String owningAccountId) {
            this.owningAccountId = owningAccountId;
        }

        @Override
        public final Builder owningAccountId(String owningAccountId) {
            this.owningAccountId = owningAccountId;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final String getService() {
            return service;
        }

        public final void setService(String service) {
            this.service = service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final Instant getLastReportedAt() {
            return lastReportedAt;
        }

        public final void setLastReportedAt(Instant lastReportedAt) {
            this.lastReportedAt = lastReportedAt;
        }

        @Override
        public final Builder lastReportedAt(Instant lastReportedAt) {
            this.lastReportedAt = lastReportedAt;
            return this;
        }

        public final List<ResourceProperty.Builder> getProperties() {
            List<ResourceProperty.Builder> result = ResourcePropertyListCopier.copyToBuilder(this.properties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProperties(Collection<ResourceProperty.BuilderImpl> properties) {
            this.properties = ResourcePropertyListCopier.copyFromBuilder(properties);
        }

        @Override
        public final Builder properties(Collection<ResourceProperty> properties) {
            this.properties = ResourcePropertyListCopier.copy(properties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder properties(ResourceProperty... properties) {
            properties(Arrays.asList(properties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder properties(Consumer<ResourceProperty.Builder>... properties) {
            properties(Stream.of(properties).map(c -> ResourceProperty.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Resource build() {
            return new Resource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
