/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information about an HSM configuration, which is an object that describes to Amazon Redshift clusters the
 * information they require to connect to an HSM where they can store database encryption keys.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HsmConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<HsmConfiguration.Builder, HsmConfiguration> {
    private static final SdkField<String> HSM_CONFIGURATION_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HsmConfigurationIdentifier")
            .getter(getter(HsmConfiguration::hsmConfigurationIdentifier))
            .setter(setter(Builder::hsmConfigurationIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HsmConfigurationIdentifier").build())
            .build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(HsmConfiguration::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> HSM_IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HsmIpAddress").getter(getter(HsmConfiguration::hsmIpAddress)).setter(setter(Builder::hsmIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HsmIpAddress").build()).build();

    private static final SdkField<String> HSM_PARTITION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HsmPartitionName").getter(getter(HsmConfiguration::hsmPartitionName))
            .setter(setter(Builder::hsmPartitionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HsmPartitionName").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(HsmConfiguration::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            HSM_CONFIGURATION_IDENTIFIER_FIELD, DESCRIPTION_FIELD, HSM_IP_ADDRESS_FIELD, HSM_PARTITION_NAME_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String hsmConfigurationIdentifier;

    private final String description;

    private final String hsmIpAddress;

    private final String hsmPartitionName;

    private final List<Tag> tags;

    private HsmConfiguration(BuilderImpl builder) {
        this.hsmConfigurationIdentifier = builder.hsmConfigurationIdentifier;
        this.description = builder.description;
        this.hsmIpAddress = builder.hsmIpAddress;
        this.hsmPartitionName = builder.hsmPartitionName;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the Amazon Redshift HSM configuration.
     * </p>
     * 
     * @return The name of the Amazon Redshift HSM configuration.
     */
    public final String hsmConfigurationIdentifier() {
        return hsmConfigurationIdentifier;
    }

    /**
     * <p>
     * A text description of the HSM configuration.
     * </p>
     * 
     * @return A text description of the HSM configuration.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The IP address that the Amazon Redshift cluster must use to access the HSM.
     * </p>
     * 
     * @return The IP address that the Amazon Redshift cluster must use to access the HSM.
     */
    public final String hsmIpAddress() {
        return hsmIpAddress;
    }

    /**
     * <p>
     * The name of the partition in the HSM where the Amazon Redshift clusters will store their database encryption
     * keys.
     * </p>
     * 
     * @return The name of the partition in the HSM where the Amazon Redshift clusters will store their database
     *         encryption keys.
     */
    public final String hsmPartitionName() {
        return hsmPartitionName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of tags for the HSM configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tags for the HSM configuration.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hsmConfigurationIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hsmIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(hsmPartitionName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HsmConfiguration)) {
            return false;
        }
        HsmConfiguration other = (HsmConfiguration) obj;
        return Objects.equals(hsmConfigurationIdentifier(), other.hsmConfigurationIdentifier())
                && Objects.equals(description(), other.description()) && Objects.equals(hsmIpAddress(), other.hsmIpAddress())
                && Objects.equals(hsmPartitionName(), other.hsmPartitionName()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HsmConfiguration").add("HsmConfigurationIdentifier", hsmConfigurationIdentifier())
                .add("Description", description()).add("HsmIpAddress", hsmIpAddress())
                .add("HsmPartitionName", hsmPartitionName()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HsmConfigurationIdentifier":
            return Optional.ofNullable(clazz.cast(hsmConfigurationIdentifier()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "HsmIpAddress":
            return Optional.ofNullable(clazz.cast(hsmIpAddress()));
        case "HsmPartitionName":
            return Optional.ofNullable(clazz.cast(hsmPartitionName()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HsmConfiguration, T> g) {
        return obj -> g.apply((HsmConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HsmConfiguration> {
        /**
         * <p>
         * The name of the Amazon Redshift HSM configuration.
         * </p>
         * 
         * @param hsmConfigurationIdentifier
         *        The name of the Amazon Redshift HSM configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hsmConfigurationIdentifier(String hsmConfigurationIdentifier);

        /**
         * <p>
         * A text description of the HSM configuration.
         * </p>
         * 
         * @param description
         *        A text description of the HSM configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IP address that the Amazon Redshift cluster must use to access the HSM.
         * </p>
         * 
         * @param hsmIpAddress
         *        The IP address that the Amazon Redshift cluster must use to access the HSM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hsmIpAddress(String hsmIpAddress);

        /**
         * <p>
         * The name of the partition in the HSM where the Amazon Redshift clusters will store their database encryption
         * keys.
         * </p>
         * 
         * @param hsmPartitionName
         *        The name of the partition in the HSM where the Amazon Redshift clusters will store their database
         *        encryption keys.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hsmPartitionName(String hsmPartitionName);

        /**
         * <p>
         * The list of tags for the HSM configuration.
         * </p>
         * 
         * @param tags
         *        The list of tags for the HSM configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The list of tags for the HSM configuration.
         * </p>
         * 
         * @param tags
         *        The list of tags for the HSM configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The list of tags for the HSM configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.redshift.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.redshift.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.redshift.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.redshift.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String hsmConfigurationIdentifier;

        private String description;

        private String hsmIpAddress;

        private String hsmPartitionName;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(HsmConfiguration model) {
            hsmConfigurationIdentifier(model.hsmConfigurationIdentifier);
            description(model.description);
            hsmIpAddress(model.hsmIpAddress);
            hsmPartitionName(model.hsmPartitionName);
            tags(model.tags);
        }

        public final String getHsmConfigurationIdentifier() {
            return hsmConfigurationIdentifier;
        }

        public final void setHsmConfigurationIdentifier(String hsmConfigurationIdentifier) {
            this.hsmConfigurationIdentifier = hsmConfigurationIdentifier;
        }

        @Override
        public final Builder hsmConfigurationIdentifier(String hsmConfigurationIdentifier) {
            this.hsmConfigurationIdentifier = hsmConfigurationIdentifier;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getHsmIpAddress() {
            return hsmIpAddress;
        }

        public final void setHsmIpAddress(String hsmIpAddress) {
            this.hsmIpAddress = hsmIpAddress;
        }

        @Override
        public final Builder hsmIpAddress(String hsmIpAddress) {
            this.hsmIpAddress = hsmIpAddress;
            return this;
        }

        public final String getHsmPartitionName() {
            return hsmPartitionName;
        }

        public final void setHsmPartitionName(String hsmPartitionName) {
            this.hsmPartitionName = hsmPartitionName;
        }

        @Override
        public final Builder hsmPartitionName(String hsmPartitionName) {
            this.hsmPartitionName = hsmPartitionName;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public HsmConfiguration build() {
            return new HsmConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
