/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <i>sheet</i>, which is an object that contains a set of visuals that are viewed together on one page in Amazon
 * QuickSight. Every analysis and dashboard contains at least one sheet. Each sheet contains at least one visualization
 * widget, for example a chart, pivot table, or narrative insight. Sheets can be associated with other components, such
 * as controls, filters, and so on.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Sheet implements SdkPojo, Serializable, ToCopyableBuilder<Sheet.Builder, Sheet> {
    private static final SdkField<String> SHEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SheetId").getter(getter(Sheet::sheetId)).setter(setter(Builder::sheetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SheetId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Sheet::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<SheetImage>> IMAGES_FIELD = SdkField
            .<List<SheetImage>> builder(MarshallingType.LIST)
            .memberName("Images")
            .getter(getter(Sheet::images))
            .setter(setter(Builder::images))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Images").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SheetImage> builder(MarshallingType.SDK_POJO)
                                            .constructor(SheetImage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SHEET_ID_FIELD, NAME_FIELD,
            IMAGES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sheetId;

    private final String name;

    private final List<SheetImage> images;

    private Sheet(BuilderImpl builder) {
        this.sheetId = builder.sheetId;
        this.name = builder.name;
        this.images = builder.images;
    }

    /**
     * <p>
     * The unique identifier associated with a sheet.
     * </p>
     * 
     * @return The unique identifier associated with a sheet.
     */
    public final String sheetId() {
        return sheetId;
    }

    /**
     * <p>
     * The name of a sheet. This name is displayed on the sheet's tab in the Amazon QuickSight console.
     * </p>
     * 
     * @return The name of a sheet. This name is displayed on the sheet's tab in the Amazon QuickSight console.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Images property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasImages() {
        return images != null && !(images instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of images on a sheet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImages} method.
     * </p>
     * 
     * @return A list of images on a sheet.
     */
    public final List<SheetImage> images() {
        return images;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sheetId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasImages() ? images() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Sheet)) {
            return false;
        }
        Sheet other = (Sheet) obj;
        return Objects.equals(sheetId(), other.sheetId()) && Objects.equals(name(), other.name())
                && hasImages() == other.hasImages() && Objects.equals(images(), other.images());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Sheet").add("SheetId", sheetId()).add("Name", name())
                .add("Images", hasImages() ? images() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SheetId":
            return Optional.ofNullable(clazz.cast(sheetId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Images":
            return Optional.ofNullable(clazz.cast(images()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SheetId", SHEET_ID_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Images", IMAGES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Sheet, T> g) {
        return obj -> g.apply((Sheet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Sheet> {
        /**
         * <p>
         * The unique identifier associated with a sheet.
         * </p>
         * 
         * @param sheetId
         *        The unique identifier associated with a sheet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sheetId(String sheetId);

        /**
         * <p>
         * The name of a sheet. This name is displayed on the sheet's tab in the Amazon QuickSight console.
         * </p>
         * 
         * @param name
         *        The name of a sheet. This name is displayed on the sheet's tab in the Amazon QuickSight console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A list of images on a sheet.
         * </p>
         * 
         * @param images
         *        A list of images on a sheet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder images(Collection<SheetImage> images);

        /**
         * <p>
         * A list of images on a sheet.
         * </p>
         * 
         * @param images
         *        A list of images on a sheet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder images(SheetImage... images);

        /**
         * <p>
         * A list of images on a sheet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.quicksight.model.SheetImage.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.quicksight.model.SheetImage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.quicksight.model.SheetImage.Builder#build()} is called immediately and
         * its result is passed to {@link #images(List<SheetImage>)}.
         * 
         * @param images
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.quicksight.model.SheetImage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #images(java.util.Collection<SheetImage>)
         */
        Builder images(Consumer<SheetImage.Builder>... images);
    }

    static final class BuilderImpl implements Builder {
        private String sheetId;

        private String name;

        private List<SheetImage> images = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Sheet model) {
            sheetId(model.sheetId);
            name(model.name);
            images(model.images);
        }

        public final String getSheetId() {
            return sheetId;
        }

        public final void setSheetId(String sheetId) {
            this.sheetId = sheetId;
        }

        @Override
        public final Builder sheetId(String sheetId) {
            this.sheetId = sheetId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final List<SheetImage.Builder> getImages() {
            List<SheetImage.Builder> result = SheetImageListCopier.copyToBuilder(this.images);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setImages(Collection<SheetImage.BuilderImpl> images) {
            this.images = SheetImageListCopier.copyFromBuilder(images);
        }

        @Override
        public final Builder images(Collection<SheetImage> images) {
            this.images = SheetImageListCopier.copy(images);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder images(SheetImage... images) {
            images(Arrays.asList(images));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder images(Consumer<SheetImage.Builder>... images) {
            images(Stream.of(images).map(c -> SheetImage.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Sheet build() {
            return new Sheet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
