/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qldb.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.qldb.QldbClient;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest;
import software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.qldb.QldbClient#listJournalS3ExportsForLedgerPaginator(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable responses = client.listJournalS3ExportsForLedgerPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable responses = client
 *             .listJournalS3ExportsForLedgerPaginator(request);
 *     for (software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.qldb.paginators.ListJournalS3ExportsForLedgerIterable responses = client.listJournalS3ExportsForLedgerPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listJournalS3ExportsForLedger(software.amazon.awssdk.services.qldb.model.ListJournalS3ExportsForLedgerRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListJournalS3ExportsForLedgerIterable implements SdkIterable<ListJournalS3ExportsForLedgerResponse> {
    private final QldbClient client;

    private final ListJournalS3ExportsForLedgerRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListJournalS3ExportsForLedgerIterable(QldbClient client, ListJournalS3ExportsForLedgerRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListJournalS3ExportsForLedgerResponseFetcher();
    }

    @Override
    public Iterator<ListJournalS3ExportsForLedgerResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListJournalS3ExportsForLedgerResponseFetcher implements SyncPageFetcher<ListJournalS3ExportsForLedgerResponse> {
        @Override
        public boolean hasNextPage(ListJournalS3ExportsForLedgerResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListJournalS3ExportsForLedgerResponse nextPage(ListJournalS3ExportsForLedgerResponse previousPage) {
            if (previousPage == null) {
                return client.listJournalS3ExportsForLedger(firstRequest);
            }
            return client.listJournalS3ExportsForLedger(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
