/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.proton.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.proton.ProtonAsyncClient;
import software.amazon.awssdk.services.proton.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.proton.model.GetComponentRequest;
import software.amazon.awssdk.services.proton.model.GetComponentResponse;
import software.amazon.awssdk.services.proton.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.proton.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.proton.model.GetEnvironmentTemplateVersionRequest;
import software.amazon.awssdk.services.proton.model.GetEnvironmentTemplateVersionResponse;
import software.amazon.awssdk.services.proton.model.GetServiceInstanceRequest;
import software.amazon.awssdk.services.proton.model.GetServiceInstanceResponse;
import software.amazon.awssdk.services.proton.model.GetServiceRequest;
import software.amazon.awssdk.services.proton.model.GetServiceResponse;
import software.amazon.awssdk.services.proton.model.GetServiceTemplateVersionRequest;
import software.amazon.awssdk.services.proton.model.GetServiceTemplateVersionResponse;
import software.amazon.awssdk.services.proton.model.ProtonRequest;
import software.amazon.awssdk.services.proton.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultProtonAsyncWaiter implements ProtonAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final ProtonAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetComponentResponse> componentDeletedWaiter;

    private final AsyncWaiter<GetComponentResponse> componentDeployedWaiter;

    private final AsyncWaiter<GetEnvironmentResponse> environmentDeployedWaiter;

    private final AsyncWaiter<GetEnvironmentTemplateVersionResponse> environmentTemplateVersionRegisteredWaiter;

    private final AsyncWaiter<GetServiceResponse> serviceCreatedWaiter;

    private final AsyncWaiter<GetServiceResponse> serviceDeletedWaiter;

    private final AsyncWaiter<GetServiceInstanceResponse> serviceInstanceDeployedWaiter;

    private final AsyncWaiter<GetServiceResponse> servicePipelineDeployedWaiter;

    private final AsyncWaiter<GetServiceTemplateVersionResponse> serviceTemplateVersionRegisteredWaiter;

    private final AsyncWaiter<GetServiceResponse> serviceUpdatedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultProtonAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ProtonAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.componentDeletedWaiter = AsyncWaiter.builder(GetComponentResponse.class)
                .acceptors(componentDeletedWaiterAcceptors())
                .overrideConfiguration(componentDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.componentDeployedWaiter = AsyncWaiter.builder(GetComponentResponse.class)
                .acceptors(componentDeployedWaiterAcceptors())
                .overrideConfiguration(componentDeployedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.environmentDeployedWaiter = AsyncWaiter.builder(GetEnvironmentResponse.class)
                .acceptors(environmentDeployedWaiterAcceptors())
                .overrideConfiguration(environmentDeployedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.environmentTemplateVersionRegisteredWaiter = AsyncWaiter.builder(GetEnvironmentTemplateVersionResponse.class)
                .acceptors(environmentTemplateVersionRegisteredWaiterAcceptors())
                .overrideConfiguration(environmentTemplateVersionRegisteredWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.serviceCreatedWaiter = AsyncWaiter.builder(GetServiceResponse.class).acceptors(serviceCreatedWaiterAcceptors())
                .overrideConfiguration(serviceCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.serviceDeletedWaiter = AsyncWaiter.builder(GetServiceResponse.class).acceptors(serviceDeletedWaiterAcceptors())
                .overrideConfiguration(serviceDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.serviceInstanceDeployedWaiter = AsyncWaiter.builder(GetServiceInstanceResponse.class)
                .acceptors(serviceInstanceDeployedWaiterAcceptors())
                .overrideConfiguration(serviceInstanceDeployedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.servicePipelineDeployedWaiter = AsyncWaiter.builder(GetServiceResponse.class)
                .acceptors(servicePipelineDeployedWaiterAcceptors())
                .overrideConfiguration(servicePipelineDeployedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.serviceTemplateVersionRegisteredWaiter = AsyncWaiter.builder(GetServiceTemplateVersionResponse.class)
                .acceptors(serviceTemplateVersionRegisteredWaiterAcceptors())
                .overrideConfiguration(serviceTemplateVersionRegisteredWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.serviceUpdatedWaiter = AsyncWaiter.builder(GetServiceResponse.class).acceptors(serviceUpdatedWaiterAcceptors())
                .overrideConfiguration(serviceUpdatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetComponentResponse>> waitUntilComponentDeleted(
            GetComponentRequest getComponentRequest) {
        return componentDeletedWaiter.runAsync(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetComponentResponse>> waitUntilComponentDeleted(
            GetComponentRequest getComponentRequest, WaiterOverrideConfiguration overrideConfig) {
        return componentDeletedWaiter.runAsync(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)),
                componentDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetComponentResponse>> waitUntilComponentDeployed(
            GetComponentRequest getComponentRequest) {
        return componentDeployedWaiter.runAsync(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetComponentResponse>> waitUntilComponentDeployed(
            GetComponentRequest getComponentRequest, WaiterOverrideConfiguration overrideConfig) {
        return componentDeployedWaiter.runAsync(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)),
                componentDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetEnvironmentResponse>> waitUntilEnvironmentDeployed(
            GetEnvironmentRequest getEnvironmentRequest) {
        return environmentDeployedWaiter.runAsync(() -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetEnvironmentResponse>> waitUntilEnvironmentDeployed(
            GetEnvironmentRequest getEnvironmentRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentDeployedWaiter.runAsync(() -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)),
                environmentDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetEnvironmentTemplateVersionResponse>> waitUntilEnvironmentTemplateVersionRegistered(
            GetEnvironmentTemplateVersionRequest getEnvironmentTemplateVersionRequest) {
        return environmentTemplateVersionRegisteredWaiter.runAsync(() -> client
                .getEnvironmentTemplateVersion(applyWaitersUserAgent(getEnvironmentTemplateVersionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetEnvironmentTemplateVersionResponse>> waitUntilEnvironmentTemplateVersionRegistered(
            GetEnvironmentTemplateVersionRequest getEnvironmentTemplateVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentTemplateVersionRegisteredWaiter.runAsync(
                () -> client.getEnvironmentTemplateVersion(applyWaitersUserAgent(getEnvironmentTemplateVersionRequest)),
                environmentTemplateVersionRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServiceCreated(GetServiceRequest getServiceRequest) {
        return serviceCreatedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServiceCreated(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return serviceCreatedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                serviceCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServiceDeleted(GetServiceRequest getServiceRequest) {
        return serviceDeletedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServiceDeleted(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return serviceDeletedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                serviceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceInstanceResponse>> waitUntilServiceInstanceDeployed(
            GetServiceInstanceRequest getServiceInstanceRequest) {
        return serviceInstanceDeployedWaiter.runAsync(() -> client
                .getServiceInstance(applyWaitersUserAgent(getServiceInstanceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceInstanceResponse>> waitUntilServiceInstanceDeployed(
            GetServiceInstanceRequest getServiceInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return serviceInstanceDeployedWaiter.runAsync(
                () -> client.getServiceInstance(applyWaitersUserAgent(getServiceInstanceRequest)),
                serviceInstanceDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServicePipelineDeployed(
            GetServiceRequest getServiceRequest) {
        return servicePipelineDeployedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServicePipelineDeployed(
            GetServiceRequest getServiceRequest, WaiterOverrideConfiguration overrideConfig) {
        return servicePipelineDeployedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                servicePipelineDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceTemplateVersionResponse>> waitUntilServiceTemplateVersionRegistered(
            GetServiceTemplateVersionRequest getServiceTemplateVersionRequest) {
        return serviceTemplateVersionRegisteredWaiter.runAsync(() -> client
                .getServiceTemplateVersion(applyWaitersUserAgent(getServiceTemplateVersionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceTemplateVersionResponse>> waitUntilServiceTemplateVersionRegistered(
            GetServiceTemplateVersionRequest getServiceTemplateVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return serviceTemplateVersionRegisteredWaiter.runAsync(
                () -> client.getServiceTemplateVersion(applyWaitersUserAgent(getServiceTemplateVersionRequest)),
                serviceTemplateVersionRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServiceUpdated(GetServiceRequest getServiceRequest) {
        return serviceUpdatedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetServiceResponse>> waitUntilServiceUpdated(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return serviceUpdatedWaiter.runAsync(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                serviceUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetComponentResponse>> componentDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetComponentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("component").field("deploymentStatus").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (component.deploymentStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetComponentResponse>> componentDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetComponentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("component").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("component").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (component.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetEnvironmentResponse>> environmentDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetEnvironmentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("environment").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("environment").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (environment.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetEnvironmentTemplateVersionResponse>> environmentTemplateVersionRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super GetEnvironmentTemplateVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("environmentTemplateVersion").field("status").value(), "DRAFT");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("environmentTemplateVersion").field("status").value(), "PUBLISHED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("environmentTemplateVersion").field("status").value(),
                            "REGISTRATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (environmentTemplateVersion.status=REGISTRATION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> serviceCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("service").field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "CREATE_FAILED_CLEANUP_COMPLETE");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=CREATE_FAILED_CLEANUP_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "CREATE_FAILED_CLEANUP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=CREATE_FAILED_CLEANUP_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> serviceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceInstanceResponse>> serviceInstanceDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("serviceInstance").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("serviceInstance").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (serviceInstance.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> servicePipelineDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("service").field("pipeline").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("pipeline").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.pipeline.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceTemplateVersionResponse>> serviceTemplateVersionRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceTemplateVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("serviceTemplateVersion").field("status").value(), "DRAFT");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("serviceTemplateVersion").field("status").value(), "PUBLISHED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("serviceTemplateVersion").field("status").value(), "REGISTRATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (serviceTemplateVersion.status=REGISTRATION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> serviceUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("service").field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_FAILED_CLEANUP_COMPLETE");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_FAILED_CLEANUP_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_FAILED_CLEANUP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_FAILED_CLEANUP_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_COMPLETE_CLEANUP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_COMPLETE_CLEANUP_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration componentDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration componentDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentTemplateVersionRegisteredWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(150);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceInstanceDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration servicePipelineDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(360);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceTemplateVersionRegisteredWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(150);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ProtonAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ProtonRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ProtonAsyncWaiter.Builder {
        private ProtonAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public ProtonAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public ProtonAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ProtonAsyncWaiter.Builder client(ProtonAsyncClient client) {
            this.client = client;
            return this;
        }

        public ProtonAsyncWaiter build() {
            return new DefaultProtonAsyncWaiter(this);
        }
    }
}
