/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pricing;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.pricing.model.DescribeServicesRequest;
import software.amazon.awssdk.services.pricing.model.DescribeServicesResponse;
import software.amazon.awssdk.services.pricing.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest;
import software.amazon.awssdk.services.pricing.model.GetAttributeValuesResponse;
import software.amazon.awssdk.services.pricing.model.GetProductsRequest;
import software.amazon.awssdk.services.pricing.model.GetProductsResponse;
import software.amazon.awssdk.services.pricing.model.InternalErrorException;
import software.amazon.awssdk.services.pricing.model.InvalidNextTokenException;
import software.amazon.awssdk.services.pricing.model.InvalidParameterException;
import software.amazon.awssdk.services.pricing.model.NotFoundException;
import software.amazon.awssdk.services.pricing.model.PricingException;
import software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable;
import software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable;
import software.amazon.awssdk.services.pricing.paginators.GetProductsIterable;

/**
 * Service client for accessing AWS Pricing. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * AWS Price List Service API (AWS Price List Service) is a centralized and convenient way to programmatically query
 * Amazon Web Services for services, products, and pricing information. The AWS Price List Service uses standardized
 * product attributes such as <code>Location</code>, <code>Storage Class</code>, and <code>Operating System</code>, and
 * provides prices at the SKU level. You can use the AWS Price List Service to build cost control and scenario planning
 * tools, reconcile billing data, forecast future spend for budgeting purposes, and provide cost benefit analysis that
 * compare your internal workloads with AWS.
 * </p>
 * <p>
 * Use <code>GetServices</code> without a service code to retrieve the service codes for all AWS services, then
 * <code>GetServices</code> with a service code to retreive the attribute names for that service. After you have the
 * service code and attribute names, you can use <code>GetAttributeValues</code> to see what values are available for an
 * attribute. With the service code and an attribute name and value, you can use <code>GetProducts</code> to find
 * specific products that you're interested in, such as an <code>AmazonEC2</code> instance, with a
 * <code>Provisioned IOPS</code> <code>volumeType</code>.
 * </p>
 * <p>
 * Service Endpoint
 * </p>
 * <p>
 * AWS Price List Service API provides the following two endpoints:
 * </p>
 * <ul>
 * <li>
 * <p>
 * https://api.pricing.us-east-1.amazonaws.com
 * </p>
 * </li>
 * <li>
 * <p>
 * https://api.pricing.ap-south-1.amazonaws.com
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public interface PricingClient extends SdkClient {
    String SERVICE_NAME = "pricing";

    /**
     * Create a {@link PricingClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static PricingClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link PricingClient}.
     */
    static PricingClientBuilder builder() {
        return new DefaultPricingClientBuilder();
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     *
     * @return Result of the DescribeServices operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see #describeServices(DescribeServicesRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServicesResponse describeServices() throws InternalErrorException, InvalidParameterException,
            NotFoundException, InvalidNextTokenException, ExpiredNextTokenException, AwsServiceException, SdkClientException,
            PricingException {
        return describeServices(DescribeServicesRequest.builder().build());
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     *
     * @param describeServicesRequest
     * @return Result of the DescribeServices operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServicesResponse describeServices(DescribeServicesRequest describeServicesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeServicesRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeServicesRequest#builder()}
     * </p>
     *
     * @param describeServicesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeServicesRequest.Builder} to create a request.
     * @return Result of the DescribeServices operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServicesResponse describeServices(Consumer<DescribeServicesRequest.Builder> describeServicesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return describeServices(DescribeServicesRequest.builder().applyMutation(describeServicesRequest).build());
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client
     *             .describeServicesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.DescribeServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see #describeServicesPaginator(DescribeServicesRequest)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServicesIterable describeServicesPaginator() throws InternalErrorException, InvalidParameterException,
            NotFoundException, InvalidNextTokenException, ExpiredNextTokenException, AwsServiceException, SdkClientException,
            PricingException {
        return describeServicesPaginator(DescribeServicesRequest.builder().build());
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client
     *             .describeServicesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.DescribeServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation.</b>
     * </p>
     *
     * @param describeServicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServicesIterable describeServicesPaginator(DescribeServicesRequest describeServicesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client
     *             .describeServicesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.DescribeServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeServicesRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeServicesRequest#builder()}
     * </p>
     *
     * @param describeServicesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeServicesRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeServicesIterable describeServicesPaginator(Consumer<DescribeServicesRequest.Builder> describeServicesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return describeServicesPaginator(DescribeServicesRequest.builder().applyMutation(describeServicesRequest).build());
    }

    /**
     * <p>
     * Returns a list of attribute values. Attibutes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">AWS Billing and Cost
     * Management User Guide</a>.
     * </p>
     *
     * @param getAttributeValuesRequest
     * @return Result of the GetAttributeValues operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetAttributeValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAttributeValuesResponse getAttributeValues(GetAttributeValuesRequest getAttributeValuesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of attribute values. Attibutes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">AWS Billing and Cost
     * Management User Guide</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAttributeValuesRequest.Builder} avoiding the
     * need to create one manually via {@link GetAttributeValuesRequest#builder()}
     * </p>
     *
     * @param getAttributeValuesRequest
     *        A {@link Consumer} that will call methods on {@link GetAttributeValuesRequest.Builder} to create a
     *        request.
     * @return Result of the GetAttributeValues operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetAttributeValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAttributeValuesResponse getAttributeValues(Consumer<GetAttributeValuesRequest.Builder> getAttributeValuesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return getAttributeValues(GetAttributeValuesRequest.builder().applyMutation(getAttributeValuesRequest).build());
    }

    /**
     * <p>
     * Returns a list of attribute values. Attibutes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">AWS Billing and Cost
     * Management User Guide</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAttributeValues(software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client.getAttributeValuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client
     *             .getAttributeValuesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.GetAttributeValuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client.getAttributeValuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAttributeValues(software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getAttributeValuesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetAttributeValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAttributeValuesIterable getAttributeValuesPaginator(GetAttributeValuesRequest getAttributeValuesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of attribute values. Attibutes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">AWS Billing and Cost
     * Management User Guide</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAttributeValues(software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client.getAttributeValuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client
     *             .getAttributeValuesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.GetAttributeValuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client.getAttributeValuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAttributeValues(software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAttributeValuesRequest.Builder} avoiding the
     * need to create one manually via {@link GetAttributeValuesRequest#builder()}
     * </p>
     *
     * @param getAttributeValuesRequest
     *        A {@link Consumer} that will call methods on {@link GetAttributeValuesRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetAttributeValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAttributeValuesIterable getAttributeValuesPaginator(
            Consumer<GetAttributeValuesRequest.Builder> getAttributeValuesRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, PricingException {
        return getAttributeValuesPaginator(GetAttributeValuesRequest.builder().applyMutation(getAttributeValuesRequest).build());
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     *
     * @param getProductsRequest
     * @return Result of the GetProducts operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    default GetProductsResponse getProducts(GetProductsRequest getProductsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, PricingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetProductsRequest.Builder} avoiding the need to
     * create one manually via {@link GetProductsRequest#builder()}
     * </p>
     *
     * @param getProductsRequest
     *        A {@link Consumer} that will call methods on {@link GetProductsRequest.Builder} to create a request.
     * @return Result of the GetProducts operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    default GetProductsResponse getProducts(Consumer<GetProductsRequest.Builder> getProductsRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return getProducts(GetProductsRequest.builder().applyMutation(getProductsRequest).build());
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getProducts(software.amazon.awssdk.services.pricing.model.GetProductsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.GetProductsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getProducts(software.amazon.awssdk.services.pricing.model.GetProductsRequest)} operation.</b>
     * </p>
     *
     * @param getProductsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    default GetProductsIterable getProductsPaginator(GetProductsRequest getProductsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, PricingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getProducts(software.amazon.awssdk.services.pricing.model.GetProductsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.GetProductsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getProducts(software.amazon.awssdk.services.pricing.model.GetProductsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link GetProductsRequest.Builder} avoiding the need to
     * create one manually via {@link GetProductsRequest#builder()}
     * </p>
     *
     * @param getProductsRequest
     *        A {@link Consumer} that will call methods on {@link GetProductsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    default GetProductsIterable getProductsPaginator(Consumer<GetProductsRequest.Builder> getProductsRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return getProductsPaginator(GetProductsRequest.builder().applyMutation(getProductsRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("api.pricing");
    }
}
