/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pricing;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pricing.model.DescribeServicesRequest;
import software.amazon.awssdk.services.pricing.model.DescribeServicesResponse;
import software.amazon.awssdk.services.pricing.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest;
import software.amazon.awssdk.services.pricing.model.GetAttributeValuesResponse;
import software.amazon.awssdk.services.pricing.model.GetProductsRequest;
import software.amazon.awssdk.services.pricing.model.GetProductsResponse;
import software.amazon.awssdk.services.pricing.model.InternalErrorException;
import software.amazon.awssdk.services.pricing.model.InvalidNextTokenException;
import software.amazon.awssdk.services.pricing.model.InvalidParameterException;
import software.amazon.awssdk.services.pricing.model.NotFoundException;
import software.amazon.awssdk.services.pricing.model.PricingException;
import software.amazon.awssdk.services.pricing.model.PricingRequest;
import software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable;
import software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable;
import software.amazon.awssdk.services.pricing.paginators.GetProductsIterable;
import software.amazon.awssdk.services.pricing.transform.DescribeServicesRequestMarshaller;
import software.amazon.awssdk.services.pricing.transform.GetAttributeValuesRequestMarshaller;
import software.amazon.awssdk.services.pricing.transform.GetProductsRequestMarshaller;

/**
 * Internal implementation of {@link PricingClient}.
 *
 * @see PricingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPricingClient implements PricingClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPricingClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     *
     * @param describeServicesRequest
     * @return Result of the DescribeServices operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServicesResponse describeServices(DescribeServicesRequest describeServicesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeServicesRequest, DescribeServicesResponse>()
                .withOperationName("DescribeServices").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeServicesRequest)
                .withMarshaller(new DescribeServicesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client
     *             .describeServicesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.DescribeServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.DescribeServicesIterable responses = client.describeServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeServices(software.amazon.awssdk.services.pricing.model.DescribeServicesRequest)} operation.</b>
     * </p>
     *
     * @param describeServicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.DescribeServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServicesIterable describeServicesPaginator(DescribeServicesRequest describeServicesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return new DescribeServicesIterable(this, applyPaginatorUserAgent(describeServicesRequest));
    }

    /**
     * <p>
     * Returns a list of attribute values. Attibutes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">AWS Billing and Cost
     * Management User Guide</a>.
     * </p>
     *
     * @param getAttributeValuesRequest
     * @return Result of the GetAttributeValues operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetAttributeValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAttributeValuesResponse getAttributeValues(GetAttributeValuesRequest getAttributeValuesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAttributeValuesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAttributeValuesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetAttributeValuesRequest, GetAttributeValuesResponse>()
                .withOperationName("GetAttributeValues").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getAttributeValuesRequest)
                .withMarshaller(new GetAttributeValuesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of attribute values. Attibutes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">AWS Billing and Cost
     * Management User Guide</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAttributeValues(software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client.getAttributeValuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client
     *             .getAttributeValuesPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.GetAttributeValuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetAttributeValuesIterable responses = client.getAttributeValuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAttributeValues(software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getAttributeValuesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetAttributeValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAttributeValuesIterable getAttributeValuesPaginator(GetAttributeValuesRequest getAttributeValuesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, PricingException {
        return new GetAttributeValuesIterable(this, applyPaginatorUserAgent(getAttributeValuesRequest));
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     *
     * @param getProductsRequest
     * @return Result of the GetProducts operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProductsResponse getProducts(GetProductsRequest getProductsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, PricingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProductsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProductsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetProductsRequest, GetProductsResponse>()
                .withOperationName("GetProducts").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getProductsRequest)
                .withMarshaller(new GetProductsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getProducts(software.amazon.awssdk.services.pricing.model.GetProductsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     *     for (software.amazon.awssdk.services.pricing.model.GetProductsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pricing.paginators.GetProductsIterable responses = client.getProductsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getProducts(software.amazon.awssdk.services.pricing.model.GetProductsRequest)} operation.</b>
     * </p>
     *
     * @param getProductsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         One or more parameters had an invalid value.
     * @throws NotFoundException
     *         The requested resource can't be found.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws ExpiredNextTokenException
     *         The pagination token expired. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PricingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PricingClient.GetProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProductsIterable getProductsPaginator(GetProductsRequest getProductsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, PricingException {
        return new GetProductsIterable(this, applyPaginatorUserAgent(getProductsRequest));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PricingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredNextTokenException")
                                .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends PricingRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
