/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.polly;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.polly.model.DeleteLexiconRequest;
import software.amazon.awssdk.services.polly.model.DeleteLexiconResponse;
import software.amazon.awssdk.services.polly.model.DescribeVoicesRequest;
import software.amazon.awssdk.services.polly.model.DescribeVoicesResponse;
import software.amazon.awssdk.services.polly.model.EngineNotSupportedException;
import software.amazon.awssdk.services.polly.model.GetLexiconRequest;
import software.amazon.awssdk.services.polly.model.GetLexiconResponse;
import software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskRequest;
import software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskResponse;
import software.amazon.awssdk.services.polly.model.InvalidLexiconException;
import software.amazon.awssdk.services.polly.model.InvalidNextTokenException;
import software.amazon.awssdk.services.polly.model.InvalidS3BucketException;
import software.amazon.awssdk.services.polly.model.InvalidS3KeyException;
import software.amazon.awssdk.services.polly.model.InvalidSampleRateException;
import software.amazon.awssdk.services.polly.model.InvalidSnsTopicArnException;
import software.amazon.awssdk.services.polly.model.InvalidSsmlException;
import software.amazon.awssdk.services.polly.model.InvalidTaskIdException;
import software.amazon.awssdk.services.polly.model.LanguageNotSupportedException;
import software.amazon.awssdk.services.polly.model.LexiconNotFoundException;
import software.amazon.awssdk.services.polly.model.LexiconSizeExceededException;
import software.amazon.awssdk.services.polly.model.ListLexiconsRequest;
import software.amazon.awssdk.services.polly.model.ListLexiconsResponse;
import software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest;
import software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse;
import software.amazon.awssdk.services.polly.model.MarksNotSupportedForFormatException;
import software.amazon.awssdk.services.polly.model.MaxLexemeLengthExceededException;
import software.amazon.awssdk.services.polly.model.MaxLexiconsNumberExceededException;
import software.amazon.awssdk.services.polly.model.PollyException;
import software.amazon.awssdk.services.polly.model.PollyRequest;
import software.amazon.awssdk.services.polly.model.PutLexiconRequest;
import software.amazon.awssdk.services.polly.model.PutLexiconResponse;
import software.amazon.awssdk.services.polly.model.ServiceFailureException;
import software.amazon.awssdk.services.polly.model.SsmlMarksNotSupportedForTextTypeException;
import software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskRequest;
import software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskResponse;
import software.amazon.awssdk.services.polly.model.SynthesisTaskNotFoundException;
import software.amazon.awssdk.services.polly.model.SynthesizeSpeechRequest;
import software.amazon.awssdk.services.polly.model.SynthesizeSpeechResponse;
import software.amazon.awssdk.services.polly.model.TextLengthExceededException;
import software.amazon.awssdk.services.polly.model.UnsupportedPlsAlphabetException;
import software.amazon.awssdk.services.polly.model.UnsupportedPlsLanguageException;
import software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher;
import software.amazon.awssdk.services.polly.transform.DeleteLexiconRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.DescribeVoicesRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.GetLexiconRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.GetSpeechSynthesisTaskRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.ListLexiconsRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.ListSpeechSynthesisTasksRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.PutLexiconRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.StartSpeechSynthesisTaskRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.SynthesizeSpeechRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PollyAsyncClient}.
 *
 * @see PollyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPollyAsyncClient implements PollyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPollyAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPollyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified pronunciation lexicon stored in an AWS Region. A lexicon which has been deleted is not
     * available for speech synthesis, nor is it possible to retrieve it using either the <code>GetLexicon</code> or
     * <code>ListLexicon</code> APIs.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param deleteLexiconRequest
     * @return A Java Future containing the result of the DeleteLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DeleteLexicon
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DeleteLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLexiconResponse> deleteLexicon(DeleteLexiconRequest deleteLexiconRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLexiconResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLexiconResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLexiconResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLexiconRequest, DeleteLexiconResponse>()
                            .withOperationName("DeleteLexicon")
                            .withMarshaller(new DeleteLexiconRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteLexiconRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of voices that are available for use when requesting speech synthesis. Each voice speaks a
     * specified language, is either male or female, and is identified by an ID, which is the ASCII version of the voice
     * name.
     * </p>
     * <p>
     * When synthesizing speech ( <code>SynthesizeSpeech</code> ), you provide the voice ID for the voice you want from
     * the list of voices returned by <code>DescribeVoices</code>.
     * </p>
     * <p>
     * For example, you want your news reader application to read news in a specific language, but giving a user the
     * option to choose the voice. Using the <code>DescribeVoices</code> operation you can provide the user with a list
     * of available voices to select from.
     * </p>
     * <p>
     * You can optionally specify a language code to filter the available voices. For example, if you specify
     * <code>en-US</code>, the operation returns a list of all available US English voices.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>polly:DescribeVoices</code> action.
     * </p>
     *
     * @param describeVoicesRequest
     * @return A Java Future containing the result of the DescribeVoices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DescribeVoices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DescribeVoices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVoicesResponse> describeVoices(DescribeVoicesRequest describeVoicesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVoicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVoicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVoicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVoicesRequest, DescribeVoicesResponse>()
                            .withOperationName("DescribeVoices")
                            .withMarshaller(new DescribeVoicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeVoicesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the content of the specified pronunciation lexicon stored in an AWS Region. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing Lexicons</a>.
     * </p>
     *
     * @param getLexiconRequest
     * @return A Java Future containing the result of the GetLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetLexicon
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLexiconResponse> getLexicon(GetLexiconRequest getLexiconRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLexiconResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLexiconResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLexiconResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLexiconRequest, GetLexiconResponse>().withOperationName("GetLexicon")
                            .withMarshaller(new GetLexiconRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getLexiconRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specific SpeechSynthesisTask object based on its TaskID. This object contains information about the
     * given speech synthesis task, including the status of the task, and a link to the S3 bucket containing the output
     * of the task.
     * </p>
     *
     * @param getSpeechSynthesisTaskRequest
     * @return A Java Future containing the result of the GetSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTaskIdException The provided Task ID is not valid. Please provide a valid Task ID and try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SynthesisTaskNotFoundException The Speech Synthesis task with requested Task ID cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetSpeechSynthesisTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetSpeechSynthesisTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSpeechSynthesisTaskResponse> getSpeechSynthesisTask(
            GetSpeechSynthesisTaskRequest getSpeechSynthesisTaskRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSpeechSynthesisTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSpeechSynthesisTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSpeechSynthesisTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSpeechSynthesisTaskRequest, GetSpeechSynthesisTaskResponse>()
                            .withOperationName("GetSpeechSynthesisTask")
                            .withMarshaller(new GetSpeechSynthesisTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSpeechSynthesisTaskRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of pronunciation lexicons stored in an AWS Region. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing Lexicons</a>.
     * </p>
     *
     * @param listLexiconsRequest
     * @return A Java Future containing the result of the ListLexicons operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListLexicons
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListLexicons" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLexiconsResponse> listLexicons(ListLexiconsRequest listLexiconsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLexiconsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLexiconsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLexiconsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLexiconsRequest, ListLexiconsResponse>()
                            .withOperationName("ListLexicons").withMarshaller(new ListLexiconsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listLexiconsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of SpeechSynthesisTask objects ordered by their creation date. This operation can filter the tasks
     * by their status, for example, allowing users to list only tasks that are completed.
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     * @return A Java Future containing the result of the ListSpeechSynthesisTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpeechSynthesisTasksResponse> listSpeechSynthesisTasks(
            ListSpeechSynthesisTasksRequest listSpeechSynthesisTasksRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpeechSynthesisTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSpeechSynthesisTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpeechSynthesisTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpeechSynthesisTasksRequest, ListSpeechSynthesisTasksResponse>()
                            .withOperationName("ListSpeechSynthesisTasks")
                            .withMarshaller(new ListSpeechSynthesisTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSpeechSynthesisTasksRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of SpeechSynthesisTask objects ordered by their creation date. This operation can filter the tasks
     * by their status, for example, allowing users to list only tasks that are completed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSpeechSynthesisTasksPublisher listSpeechSynthesisTasksPaginator(
            ListSpeechSynthesisTasksRequest listSpeechSynthesisTasksRequest) {
        return new ListSpeechSynthesisTasksPublisher(this, applyPaginatorUserAgent(listSpeechSynthesisTasksRequest));
    }

    /**
     * <p>
     * Stores a pronunciation lexicon in an AWS Region. If a lexicon with the same name already exists in the region, it
     * is overwritten by the new lexicon. Lexicon operations have eventual consistency, therefore, it might take some
     * time before the lexicon is available to the SynthesizeSpeech operation.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param putLexiconRequest
     * @return A Java Future containing the result of the PutLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidLexiconException Amazon Polly can't find the specified lexicon. Verify that the lexicon's name
     *         is spelled correctly, and then try again.</li>
     *         <li>UnsupportedPlsAlphabetException The alphabet specified by the lexicon is not a supported alphabet.
     *         Valid values are <code>x-sampa</code> and <code>ipa</code>.</li>
     *         <li>UnsupportedPlsLanguageException The language specified in the lexicon is unsupported. For a list of
     *         supported languages, see <a
     *         href="https://docs.aws.amazon.com/polly/latest/dg/API_LexiconAttributes.html">Lexicon Attributes</a>.</li>
     *         <li>LexiconSizeExceededException The maximum size of the specified lexicon would be exceeded by this
     *         operation.</li>
     *         <li>MaxLexemeLengthExceededException The maximum size of the lexeme would be exceeded by this operation.</li>
     *         <li>MaxLexiconsNumberExceededException The maximum number of lexicons would be exceeded by this
     *         operation.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.PutLexicon
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/PutLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutLexiconResponse> putLexicon(PutLexiconRequest putLexiconRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLexiconResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutLexiconResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLexiconResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLexiconRequest, PutLexiconResponse>().withOperationName("PutLexicon")
                            .withMarshaller(new PutLexiconRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putLexiconRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows the creation of an asynchronous synthesis task, by starting a new <code>SpeechSynthesisTask</code>. This
     * operation requires all the standard information needed for speech synthesis, plus the name of an Amazon S3 bucket
     * for the service to store the output of the synthesis task and two optional parameters (OutputS3KeyPrefix and
     * SnsTopicArn). Once the synthesis task is created, this operation will return a SpeechSynthesisTask object, which
     * will include an identifier of this task as well as the current status.
     * </p>
     *
     * @param startSpeechSynthesisTaskRequest
     * @return A Java Future containing the result of the StartSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidS3BucketException The provided Amazon S3 bucket name is invalid. Please check your input with
     *         S3 bucket naming requirements and try again.</li>
     *         <li>InvalidS3KeyException The provided Amazon S3 key prefix is invalid. Please provide a valid S3 object
     *         key name.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSnsTopicArnException The provided SNS topic ARN is invalid. Please provide a valid SNS topic
     *         ARN and try again.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.StartSpeechSynthesisTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/StartSpeechSynthesisTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSpeechSynthesisTaskResponse> startSpeechSynthesisTask(
            StartSpeechSynthesisTaskRequest startSpeechSynthesisTaskRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSpeechSynthesisTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSpeechSynthesisTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSpeechSynthesisTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSpeechSynthesisTaskRequest, StartSpeechSynthesisTaskResponse>()
                            .withOperationName("StartSpeechSynthesisTask")
                            .withMarshaller(new StartSpeechSynthesisTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startSpeechSynthesisTaskRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes. SSML input must be valid, well-formed SSML.
     * Some alphabets might not be available with all the voices (for example, Cyrillic might not be read at all by
     * English voices) unless phoneme mapping is used. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/how-text-to-speech-works.html">How it Works</a>.
     * </p>
     *
     * @param synthesizeSpeechRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        Stream containing the synthesized speech.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.SynthesizeSpeech
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/SynthesizeSpeech" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> synthesizeSpeech(SynthesizeSpeechRequest synthesizeSpeechRequest,
            AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT> asyncResponseTransformer) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<SynthesizeSpeechResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SynthesizeSpeechResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<SynthesizeSpeechRequest, SynthesizeSpeechResponse>()
                            .withOperationName("SynthesizeSpeech")
                            .withMarshaller(new SynthesizeSpeechRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(synthesizeSpeechRequest), asyncResponseTransformer);
            executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> asyncResponseTransformer.exceptionOccurred(e));
                }
            });
            return executeFuture;
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseTransformer.exceptionOccurred(t));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PollyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3BucketException")
                                .exceptionBuilderSupplier(InvalidS3BucketException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSampleRateException")
                                .exceptionBuilderSupplier(InvalidSampleRateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SynthesisTaskNotFoundException")
                                .exceptionBuilderSupplier(SynthesisTaskNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaxLexiconsNumberExceededException")
                                .exceptionBuilderSupplier(MaxLexiconsNumberExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaxLexemeLengthExceededException")
                                .exceptionBuilderSupplier(MaxLexemeLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedPlsAlphabetException")
                                .exceptionBuilderSupplier(UnsupportedPlsAlphabetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LanguageNotSupportedException")
                                .exceptionBuilderSupplier(LanguageNotSupportedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextLengthExceededException")
                                .exceptionBuilderSupplier(TextLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLexiconException")
                                .exceptionBuilderSupplier(InvalidLexiconException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SsmlMarksNotSupportedForTextTypeException")
                                .exceptionBuilderSupplier(SsmlMarksNotSupportedForTextTypeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3KeyException")
                                .exceptionBuilderSupplier(InvalidS3KeyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSnsTopicArnException")
                                .exceptionBuilderSupplier(InvalidSnsTopicArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MarksNotSupportedForFormatException")
                                .exceptionBuilderSupplier(MarksNotSupportedForFormatException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTaskIdException")
                                .exceptionBuilderSupplier(InvalidTaskIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedPlsLanguageException")
                                .exceptionBuilderSupplier(UnsupportedPlsLanguageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LexiconNotFoundException")
                                .exceptionBuilderSupplier(LexiconNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LexiconSizeExceededException")
                                .exceptionBuilderSupplier(LexiconSizeExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSsmlException")
                                .exceptionBuilderSupplier(InvalidSsmlException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EngineNotSupportedException")
                                .exceptionBuilderSupplier(EngineNotSupportedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build());
    }

    private <T extends PollyRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
