/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the delivery configuration settings for sending a campaign or campaign treatment through a custom channel.
 * This object is required if you use the CampaignCustomMessage object to define the message to send for the campaign or
 * campaign treatment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomDeliveryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomDeliveryConfiguration.Builder, CustomDeliveryConfiguration> {
    private static final SdkField<String> DELIVERY_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomDeliveryConfiguration::deliveryUri)).setter(setter(Builder::deliveryUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryUri").build()).build();

    private static final SdkField<List<String>> ENDPOINT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CustomDeliveryConfiguration::endpointTypesAsStrings))
            .setter(setter(Builder::endpointTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DELIVERY_URI_FIELD,
            ENDPOINT_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deliveryUri;

    private final List<String> endpointTypes;

    private CustomDeliveryConfiguration(BuilderImpl builder) {
        this.deliveryUri = builder.deliveryUri;
        this.endpointTypes = builder.endpointTypes;
    }

    /**
     * <p>
     * The destination to send the campaign or treatment to. This value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the campaign or
     * treatment.
     * </p>
     * </li>
     * <li>
     * <p>
     * The URL for a web application or service that supports HTTPS and can receive the message. The URL has to be a
     * full URL, including the HTTPS protocol.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The destination to send the campaign or treatment to. This value can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the
     *         campaign or treatment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The URL for a web application or service that supports HTTPS and can receive the message. The URL has to
     *         be a full URL, including the HTTPS protocol.
     *         </p>
     *         </li>
     */
    public String deliveryUri() {
        return deliveryUri;
    }

    /**
     * <p>
     * The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel that you
     * can associate with an endpoint by using the ChannelType property of an endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEndpointTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel
     *         that you can associate with an endpoint by using the ChannelType property of an endpoint.
     */
    public List<EndpointTypesElement> endpointTypes() {
        return ListOf__EndpointTypesElementCopier.copyStringToEnum(endpointTypes);
    }

    /**
     * Returns true if the EndpointTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEndpointTypes() {
        return endpointTypes != null && !(endpointTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel that you
     * can associate with an endpoint by using the ChannelType property of an endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEndpointTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel
     *         that you can associate with an endpoint by using the ChannelType property of an endpoint.
     */
    public List<String> endpointTypesAsStrings() {
        return endpointTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deliveryUri());
        hashCode = 31 * hashCode + Objects.hashCode(endpointTypesAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomDeliveryConfiguration)) {
            return false;
        }
        CustomDeliveryConfiguration other = (CustomDeliveryConfiguration) obj;
        return Objects.equals(deliveryUri(), other.deliveryUri())
                && Objects.equals(endpointTypesAsStrings(), other.endpointTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CustomDeliveryConfiguration").add("DeliveryUri", deliveryUri())
                .add("EndpointTypes", endpointTypesAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeliveryUri":
            return Optional.ofNullable(clazz.cast(deliveryUri()));
        case "EndpointTypes":
            return Optional.ofNullable(clazz.cast(endpointTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomDeliveryConfiguration, T> g) {
        return obj -> g.apply((CustomDeliveryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomDeliveryConfiguration> {
        /**
         * <p>
         * The destination to send the campaign or treatment to. This value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the campaign
         * or treatment.
         * </p>
         * </li>
         * <li>
         * <p>
         * The URL for a web application or service that supports HTTPS and can receive the message. The URL has to be a
         * full URL, including the HTTPS protocol.
         * </p>
         * </li>
         * </ul>
         * 
         * @param deliveryUri
         *        The destination to send the campaign or treatment to. This value can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the
         *        campaign or treatment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The URL for a web application or service that supports HTTPS and can receive the message. The URL has
         *        to be a full URL, including the HTTPS protocol.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryUri(String deliveryUri);

        /**
         * <p>
         * The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel that
         * you can associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of
         *        channel that you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypesWithStrings(Collection<String> endpointTypes);

        /**
         * <p>
         * The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel that
         * you can associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of
         *        channel that you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypesWithStrings(String... endpointTypes);

        /**
         * <p>
         * The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel that
         * you can associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of
         *        channel that you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypes(Collection<EndpointTypesElement> endpointTypes);

        /**
         * <p>
         * The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of channel that
         * you can associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the campaign or treatment to. Each valid value maps to a type of
         *        channel that you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypes(EndpointTypesElement... endpointTypes);
    }

    static final class BuilderImpl implements Builder {
        private String deliveryUri;

        private List<String> endpointTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CustomDeliveryConfiguration model) {
            deliveryUri(model.deliveryUri);
            endpointTypesWithStrings(model.endpointTypes);
        }

        public final String getDeliveryUri() {
            return deliveryUri;
        }

        @Override
        public final Builder deliveryUri(String deliveryUri) {
            this.deliveryUri = deliveryUri;
            return this;
        }

        public final void setDeliveryUri(String deliveryUri) {
            this.deliveryUri = deliveryUri;
        }

        public final Collection<String> getEndpointTypes() {
            return endpointTypes;
        }

        @Override
        public final Builder endpointTypesWithStrings(Collection<String> endpointTypes) {
            this.endpointTypes = ListOf__EndpointTypesElementCopier.copy(endpointTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpointTypesWithStrings(String... endpointTypes) {
            endpointTypesWithStrings(Arrays.asList(endpointTypes));
            return this;
        }

        @Override
        public final Builder endpointTypes(Collection<EndpointTypesElement> endpointTypes) {
            this.endpointTypes = ListOf__EndpointTypesElementCopier.copyEnumToString(endpointTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpointTypes(EndpointTypesElement... endpointTypes) {
            endpointTypes(Arrays.asList(endpointTypes));
            return this;
        }

        public final void setEndpointTypes(Collection<String> endpointTypes) {
            this.endpointTypes = ListOf__EndpointTypesElementCopier.copy(endpointTypes);
        }

        @Override
        public CustomDeliveryConfiguration build() {
            return new CustomDeliveryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
