/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.paymentcryptography.internal.PaymentCryptographyServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.paymentcryptography.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.paymentcryptography.model.AccessDeniedException;
import software.amazon.awssdk.services.paymentcryptography.model.AddKeyReplicationRegionsRequest;
import software.amazon.awssdk.services.paymentcryptography.model.AddKeyReplicationRegionsResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ConflictException;
import software.amazon.awssdk.services.paymentcryptography.model.CreateAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.CreateAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.CreateKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.CreateKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DisableDefaultKeyReplicationRegionsRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DisableDefaultKeyReplicationRegionsResponse;
import software.amazon.awssdk.services.paymentcryptography.model.EnableDefaultKeyReplicationRegionsRequest;
import software.amazon.awssdk.services.paymentcryptography.model.EnableDefaultKeyReplicationRegionsResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ExportKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ExportKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetCertificateSigningRequestRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetCertificateSigningRequestResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetDefaultKeyReplicationRegionsRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetDefaultKeyReplicationRegionsResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForExportRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForExportResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForImportRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForImportResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetPublicKeyCertificateRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetPublicKeyCertificateResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ImportKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ImportKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.InternalServerException;
import software.amazon.awssdk.services.paymentcryptography.model.ListAliasesRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListAliasesResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ListKeysRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListKeysResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.PaymentCryptographyException;
import software.amazon.awssdk.services.paymentcryptography.model.RemoveKeyReplicationRegionsRequest;
import software.amazon.awssdk.services.paymentcryptography.model.RemoveKeyReplicationRegionsResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ResourceNotFoundException;
import software.amazon.awssdk.services.paymentcryptography.model.RestoreKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.RestoreKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.paymentcryptography.model.ServiceUnavailableException;
import software.amazon.awssdk.services.paymentcryptography.model.StartKeyUsageRequest;
import software.amazon.awssdk.services.paymentcryptography.model.StartKeyUsageResponse;
import software.amazon.awssdk.services.paymentcryptography.model.StopKeyUsageRequest;
import software.amazon.awssdk.services.paymentcryptography.model.StopKeyUsageResponse;
import software.amazon.awssdk.services.paymentcryptography.model.TagResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.TagResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ThrottlingException;
import software.amazon.awssdk.services.paymentcryptography.model.UntagResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.UntagResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.UpdateAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.UpdateAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ValidationException;
import software.amazon.awssdk.services.paymentcryptography.transform.AddKeyReplicationRegionsRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.CreateKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DeleteAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DeleteKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DisableDefaultKeyReplicationRegionsRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.EnableDefaultKeyReplicationRegionsRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ExportKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetCertificateSigningRequestRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetDefaultKeyReplicationRegionsRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetParametersForExportRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetParametersForImportRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetPublicKeyCertificateRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ImportKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListAliasesRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListKeysRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.RemoveKeyReplicationRegionsRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.RestoreKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.StartKeyUsageRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.StopKeyUsageRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.UpdateAliasRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PaymentCryptographyAsyncClient}.
 *
 * @see PaymentCryptographyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPaymentCryptographyAsyncClient implements PaymentCryptographyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPaymentCryptographyAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPaymentCryptographyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Payment_Cryptography" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds replication Amazon Web Services Regions to an existing Amazon Web Services Payment Cryptography key,
     * enabling the key to be used for cryptographic operations in additional Amazon Web Services Regions.
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html">
     * Multi-Region key replication</a> allow you to use the same key material across multiple Amazon Web Services
     * Regions, providing lower latency for applications distributed across regions. When you add Replication Regions,
     * Amazon Web Services Payment Cryptography securely replicates the key material to the specified Amazon Web
     * Services Regions.
     * </p>
     * <p>
     * The key must be in an active state to add Replication Regions. You can add multiple regions in a single
     * operation, and the key will be available for use in those regions once replication is complete.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_RemoveKeyReplicationRegions.html"
     * >RemoveKeyReplicationRegions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_EnableDefaultKeyReplicationRegions.html"
     * >EnableDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetDefaultKeyReplicationRegions.html"
     * >GetDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param addKeyReplicationRegionsRequest
     *        Input parameters for adding replication regions to a specific key.
     * @return A Java Future containing the result of the AddKeyReplicationRegions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.AddKeyReplicationRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/AddKeyReplicationRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddKeyReplicationRegionsResponse> addKeyReplicationRegions(
            AddKeyReplicationRegionsRequest addKeyReplicationRegionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addKeyReplicationRegionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addKeyReplicationRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddKeyReplicationRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddKeyReplicationRegionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddKeyReplicationRegionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AddKeyReplicationRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddKeyReplicationRegionsRequest, AddKeyReplicationRegionsResponse>()
                            .withOperationName("AddKeyReplicationRegions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AddKeyReplicationRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addKeyReplicationRegionsRequest));
            CompletableFuture<AddKeyReplicationRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an <i>alias</i>, or a friendly name, for an Amazon Web Services Payment Cryptography key. You can use an
     * alias to identify a key in the console and when you call cryptographic operations such as <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_EncryptData.html"
     * >EncryptData</a> or <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_DecryptData.html"
     * >DecryptData</a>.
     * </p>
     * <p>
     * You can associate the alias with any key in the same Amazon Web Services Region. Each alias is associated with
     * only one key at a time, but a key can have multiple aliases. You can't create an alias without a key. The alias
     * must be unique in the account and Amazon Web Services Region, but you can create another alias with the same name
     * in a different Amazon Web Services Region.
     * </p>
     * <p>
     * To change the key that's associated with the alias, call <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>.
     * To delete the alias, call <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html">DeleteAlias</a>.
     * These operations don't affect the underlying key. To get the alias that you created, call <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListAliases.html">ListAliases</a>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html">DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetAlias.html">GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListAliases.html">ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAliasRequest
     * @return A Java Future containing the result of the CreateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.CreateAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/CreateAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAliasResponse> createAlias(CreateAliasRequest createAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                            .withOperationName("CreateAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAliasRequest));
            CompletableFuture<CreateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Web Services Payment Cryptography key, a logical representation of a cryptographic key, that is
     * unique in your account and Amazon Web Services Region. You use keys for cryptographic functions such as
     * encryption and decryption.
     * </p>
     * <p>
     * In addition to the key material used in cryptographic operations, an Amazon Web Services Payment Cryptography key
     * includes metadata such as the key ARN, key usage, key origin, creation date, description, and key state.
     * </p>
     * <p>
     * When you create a key, you specify both immutable and mutable data about the key. The immutable data contains key
     * attributes that define the scope and cryptographic operations that you can perform using the key, for example key
     * class (example: <code>SYMMETRIC_KEY</code>), key algorithm (example: <code>TDES_2KEY</code>), key usage (example:
     * <code>TR31_P0_PIN_ENCRYPTION_KEY</code>) and key modes of use (example: <code>Encrypt</code>). Amazon Web
     * Services Payment Cryptography binds key attributes to keys using key blocks when you store or export them. Amazon
     * Web Services Payment Cryptography stores the key contents wrapped and never stores or transmits them in the
     * clear.
     * </p>
     * <p>
     * For information about valid combinations of key attributes, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>. The mutable data contained
     * within a key includes usage timestamp and key deletion timestamp and can be modified after creation.
     * </p>
     * <p>
     * You can use the <code>CreateKey</code> operation to generate an ECC (Elliptic Curve Cryptography) key pair used
     * for establishing an ECDH (Elliptic Curve Diffie-Hellman) key agreement between two parties. In the ECDH key
     * agreement process, both parties generate their own ECC key pair with key usage K3 and exchange the public keys.
     * Each party then use their private key, the received public key from the other party, and the key derivation
     * parameters including key derivation function, hash algorithm, derivation data, and key algorithm to derive a
     * shared key.
     * </p>
     * <p>
     * To maintain the single-use principle of cryptographic keys in payments, ECDH derived keys should not be used for
     * multiple purposes, such as a <code>TR31_P0_PIN_ENCRYPTION_KEY</code> and
     * <code>TR31_K1_KEY_BLOCK_PROTECTION_KEY</code>. When creating ECC key pairs in Amazon Web Services Payment
     * Cryptography you can optionally set the <code>DeriveKeyUsage</code> parameter, which defines the key usage bound
     * to the symmetric key that will be derived using the ECC key pair.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html">DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetKey.html">GetKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListKeys.html">ListKeys</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createKeyRequest
     * @return A Java Future containing the result of the CreateKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.CreateKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/CreateKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKeyResponse> createKey(CreateKeyRequest createKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKeyRequest, CreateKeyResponse>().withOperationName("CreateKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createKeyRequest));
            CompletableFuture<CreateKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the alias, but doesn't affect the underlying key.
     * </p>
     * <p>
     * Each key can have multiple aliases. To get the aliases of all keys, use the <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>
     * operation. To change the alias of a key, first use <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html">DeleteAlias</a>
     * to delete the current alias and then use <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateAlias.html">CreateAlias</a>
     * to create a new alias. To associate an existing alias with a different key, call <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateAlias.html">CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetAlias.html">GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListAliases.html">ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAliasRequest
     * @return A Java Future containing the result of the DeleteAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.DeleteAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DeleteAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAliasResponse> deleteAlias(DeleteAliasRequest deleteAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAliasRequest, DeleteAliasResponse>()
                            .withOperationName("DeleteAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAliasRequest));
            CompletableFuture<DeleteAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the key material and metadata associated with Amazon Web Services Payment Cryptography key.
     * </p>
     * <p>
     * Key deletion is irreversible. After a key is deleted, you can't perform cryptographic operations using the key.
     * For example, you can't decrypt data that was encrypted by a deleted Amazon Web Services Payment Cryptography key,
     * and the data may become unrecoverable. Because key deletion is destructive, Amazon Web Services Payment
     * Cryptography has a safety mechanism to prevent accidental deletion of a key. When you call this operation, Amazon
     * Web Services Payment Cryptography disables the specified key but doesn't delete it until after a waiting period
     * set using <code>DeleteKeyInDays</code>. The default waiting period is 7 days. During the waiting period, the
     * <code>KeyState</code> is <code>DELETE_PENDING</code>. After the key is deleted, the <code>KeyState</code> is
     * <code>DELETE_COMPLETE</code>.
     * </p>
     * <p>
     * You should delete a key only when you are sure that you don't need to use it anymore and no other parties are
     * utilizing this key. If you aren't sure, consider deactivating it instead by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StopKeyUsage.html"
     * >StopKeyUsage</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_RestoreKey.html">RestoreKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StartKeyUsage.html">
     * StartKeyUsage</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StopKeyUsage.html">StopKeyUsage<
     * /a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteKeyRequest
     * @return A Java Future containing the result of the DeleteKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.DeleteKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DeleteKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKeyResponse> deleteKey(DeleteKeyRequest deleteKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKeyRequest, DeleteKeyResponse>().withOperationName("DeleteKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteKeyRequest));
            CompletableFuture<DeleteKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a> settings for the specified Amazon Web Services Regions in your Amazon Web
     * Services account, preventing new keys from being automatically replicated to those regions.
     * </p>
     * <p>
     * After disabling Multi-Region key replication for specific regions, new keys created in your account will not be
     * automatically replicated to those regions. You can still manually add replication to those regions for individual
     * keys using the <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_AddKeyReplicationRegions.html"
     * >AddKeyReplicationRegions</a> operation.
     * </p>
     * <p>
     * This operation does not affect existing keys or their current replication configuration.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_EnableDefaultKeyReplicationRegions.html"
     * >EnableDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetDefaultKeyReplicationRegions.html"
     * >GetDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param disableDefaultKeyReplicationRegionsRequest
     *        Input parameters for disabling default key replication regions for the account.
     * @return A Java Future containing the result of the DisableDefaultKeyReplicationRegions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.DisableDefaultKeyReplicationRegions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DisableDefaultKeyReplicationRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableDefaultKeyReplicationRegionsResponse> disableDefaultKeyReplicationRegions(
            DisableDefaultKeyReplicationRegionsRequest disableDefaultKeyReplicationRegionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableDefaultKeyReplicationRegionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableDefaultKeyReplicationRegionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableDefaultKeyReplicationRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableDefaultKeyReplicationRegionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisableDefaultKeyReplicationRegionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisableDefaultKeyReplicationRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableDefaultKeyReplicationRegionsRequest, DisableDefaultKeyReplicationRegionsResponse>()
                            .withOperationName("DisableDefaultKeyReplicationRegions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableDefaultKeyReplicationRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableDefaultKeyReplicationRegionsRequest));
            CompletableFuture<DisableDefaultKeyReplicationRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a> settings for your Amazon Web Services account, causing new keys to be
     * automatically replicated to the specified Amazon Web Services Regions when created.
     * </p>
     * <p>
     * When Multi-Region key replication are enabled, any new keys created in your account will automatically be
     * replicated to these regions unless you explicitly override this behavior during key creation. This simplifies key
     * management for applications that operate across multiple regions.
     * </p>
     * <p>
     * Existing keys are not affected by this operation - only keys created after enabling default replication will be
     * automatically replicated.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DisableDefaultKeyReplicationRegions.html"
     * >DisableDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetDefaultKeyReplicationRegions.html"
     * >GetDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param enableDefaultKeyReplicationRegionsRequest
     *        Input parameters for enabling default key replication regions for the account.
     * @return A Java Future containing the result of the EnableDefaultKeyReplicationRegions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.EnableDefaultKeyReplicationRegions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/EnableDefaultKeyReplicationRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableDefaultKeyReplicationRegionsResponse> enableDefaultKeyReplicationRegions(
            EnableDefaultKeyReplicationRegionsRequest enableDefaultKeyReplicationRegionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableDefaultKeyReplicationRegionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableDefaultKeyReplicationRegionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableDefaultKeyReplicationRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableDefaultKeyReplicationRegionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EnableDefaultKeyReplicationRegionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<EnableDefaultKeyReplicationRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableDefaultKeyReplicationRegionsRequest, EnableDefaultKeyReplicationRegionsResponse>()
                            .withOperationName("EnableDefaultKeyReplicationRegions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableDefaultKeyReplicationRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableDefaultKeyReplicationRegionsRequest));
            CompletableFuture<EnableDefaultKeyReplicationRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports a key from Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography simplifies key exchange by replacing the existing paper-based approach
     * with a modern electronic approach. With <code>ExportKey</code> you can export symmetric keys using either
     * symmetric and asymmetric key exchange mechanisms. Using this operation, you can share your Amazon Web Services
     * Payment Cryptography generated keys with other service partners to perform cryptographic operations outside of
     * Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * For symmetric key exchange, Amazon Web Services Payment Cryptography uses the ANSI X9 TR-31 norm in accordance
     * with PCI PIN guidelines. And for asymmetric key exchange, Amazon Web Services Payment Cryptography supports ANSI
     * X9 TR-34 norm, RSA unwrap, and ECDH (Elliptic Curve Diffie-Hellman) key exchange mechanisms. Asymmetric key
     * exchange methods are typically used to establish bi-directional trust between the two parties exhanging keys and
     * are used for initial key exchange such as Key Encryption Key (KEK). After which you can export working keys using
     * symmetric method to perform various cryptographic operations within Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * PCI requires specific minimum key strength of wrapping keys used to protect the keys being exchanged
     * electronically. These requirements can change when PCI standards are revised. The rules specify that wrapping
     * keys used for transport must be at least as strong as the key being protected. For more information on
     * recommended key strength of wrapping keys and key exchange mechanism, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-importexport.html">Importing and
     * exporting keys</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * You can also use <code>ExportKey</code> functionality to generate and export an IPEK (Initial Pin Encryption Key)
     * from Amazon Web Services Payment Cryptography using either TR-31 or TR-34 export key exchange. IPEK is generated
     * from BDK (Base Derivation Key) and <code>ExportDukptInitialKey</code> attribute KSN (<code>KeySerialNumber</code>
     * ). The generated IPEK does not persist within Amazon Web Services Payment Cryptography and has to be re-generated
     * each time during export.
     * </p>
     * <p>
     * For key exchange using TR-31 or TR-34 key blocks, you can also export optional blocks within the key block header
     * which contain additional attribute information about the key. The <code>KeyVersion</code> within
     * <code>KeyBlockHeaders</code> indicates the version of the key within the key block. Furthermore,
     * <code>KeyExportability</code> within <code>KeyBlockHeaders</code> can be used to further restrict exportability
     * of the key after export from Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * The <code>OptionalBlocks</code> contain the additional data related to the key. For information on data type that
     * can be included within optional blocks, refer to <a
     * href="https://webstore.ansi.org/standards/ascx9/ansix91432022">ASC X9.143-2022</a>.
     * </p>
     * <note>
     * <p>
     * Data included in key block headers is signed but transmitted in clear text. Sensitive or confidential information
     * should not be included in optional blocks. Refer to ASC X9.143-2022 standard for information on allowed data
     * type.
     * </p>
     * </note>
     * <p>
     * <b>To export initial keys (KEK) or IPEK using TR-34</b>
     * </p>
     * <p>
     * Using this operation, you can export initial key using TR-34 asymmetric key exchange. You can only export KEK
     * generated within Amazon Web Services Payment Cryptography. In TR-34 terminology, the sending party of the key is
     * called Key Distribution Host (KDH) and the receiving party of the key is called Key Receiving Device (KRD).
     * During key export process, KDH is Amazon Web Services Payment Cryptography which initiates key export and KRD is
     * the user receiving the key.
     * </p>
     * <p>
     * To initiate TR-34 key export, the KRD must obtain an export token by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html"
     * >GetParametersForExport</a>. This operation also generates a key pair for the purpose of key export, signs the
     * key and returns back the signing public key certificate (also known as KDH signing certificate) and root
     * certificate chain. The KDH uses the private key to sign the the export payload and the signing public key
     * certificate is provided to KRD to verify the signature. The KRD can import the root certificate into its Hardware
     * Security Module (HSM), as required. The export token and the associated KDH signing certificate expires after 30
     * days.
     * </p>
     * <p>
     * Next the KRD generates a key pair for the the purpose of encrypting the KDH key and provides the public key
     * cerificate (also known as KRD wrapping certificate) back to KDH. The KRD will also import the root cerificate
     * chain into Amazon Web Services Payment Cryptography by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a> for
     * <code>RootCertificatePublicKey</code>. The KDH, Amazon Web Services Payment Cryptography, will use the KRD
     * wrapping cerificate to encrypt (wrap) the key under export and signs it with signing private key to generate a
     * TR-34 WrappedKeyBlock. For more information on TR-34 key export, see section <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-export.html">Exporting symmetric
     * keys</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ExportAttributes</code>: Specify export attributes in case of IPEK export. This parameter is optional for
     * KEK export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExportKeyIdentifier</code>: The <code>KeyARN</code> of the KEK or BDK (in case of IPEK) under export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr34KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>KeyARN</code> of the certificate chain that
     * signed the KRD wrapping key certificate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExportToken</code>: Obtained from KDH by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html"
     * >GetParametersForImport</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappingKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) of the KRD
     * wrapping key Amazon Web Services Payment Cryptography uses for encryption of the TR-34 export payload. This
     * certificate must be signed by the root certificate (CertificateAuthorityPublicKeyIdentifier) imported into Amazon
     * Web Services Payment Cryptography.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When this operation is successful, Amazon Web Services Payment Cryptography returns the KEK or IPEK as a TR-34
     * WrappedKeyBlock.
     * </p>
     * <p>
     * <b>To export initial keys (KEK) or IPEK using RSA Wrap and Unwrap</b>
     * </p>
     * <p>
     * Using this operation, you can export initial key using asymmetric RSA wrap and unwrap key exchange method. To
     * initiate export, generate an asymmetric key pair on the receiving HSM and obtain the public key certificate in
     * PEM format (base64 encoded) for the purpose of wrapping and the root certifiate chain. Import the root
     * certificate into Amazon Web Services Payment Cryptography by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a> for
     * <code>RootCertificatePublicKey</code>.
     * </p>
     * <p>
     * Next call <code>ExportKey</code> and set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>KeyARN</code> of the certificate chain that
     * signed wrapping key certificate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Set to <code>KeyCryptogram</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappingKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) obtained by the
     * receiving HSM and signed by the root certificate (CertificateAuthorityPublicKeyIdentifier) imported into Amazon
     * Web Services Payment Cryptography. The receiving HSM uses its private key component to unwrap the
     * WrappedKeyCryptogram.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When this operation is successful, Amazon Web Services Payment Cryptography returns the WrappedKeyCryptogram.
     * </p>
     * <p>
     * <b>To export working keys or IPEK using TR-31</b>
     * </p>
     * <p>
     * Using this operation, you can export working keys or IPEK using TR-31 symmetric key exchange. In TR-31, you must
     * use an initial key such as KEK to encrypt or wrap the key under export. To establish a KEK, you can use <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a> or
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ExportAttributes</code>: Specify export attributes in case of IPEK export. This parameter is optional for
     * KEK export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExportKeyIdentifier</code>: The <code>KeyARN</code> of the KEK or BDK (in case of IPEK) under export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr31KeyBlock</code> parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To export working keys using ECDH</b>
     * </p>
     * <p>
     * You can also use ECDH key agreement to export working keys in a TR-31 keyblock, where the wrapping key is an ECDH
     * derived key.
     * </p>
     * <p>
     * To initiate a TR-31 key export using ECDH, both sides must create an ECC key pair with key usage K3 and exchange
     * public key certificates. In Amazon Web Services Payment Cryptography, you can do this by calling
     * <code>CreateKey</code>. If you have not already done so, you must import the CA chain that issued the receiving
     * public key certificate by calling <code>ImportKey</code> with input <code>RootCertificatePublicKey</code> for
     * root CA or <code>TrustedPublicKey</code> for intermediate CA. You can then complete a TR-31 key export by
     * deriving a shared wrapping key using the service ECC key pair, public certificate of your ECC key pair outside of
     * Amazon Web Services Payment Cryptography, and the key derivation parameters including key derivation function,
     * hash algorithm, derivation data, key algorithm.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>DiffieHellmanTr31KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PrivateKeyIdentifier</code>: The <code>KeyArn</code> of the ECC key pair created within Amazon Web Services
     * Payment Cryptography to derive a shared KEK.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The public key certificate of the receiving ECC key pair in PEM format (base64
     * encoded) to derive a shared KEK.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>keyARN</code> of the CA that signed the public
     * key certificate of the receiving ECC key pair.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When this operation is successful, Amazon Web Services Payment Cryptography returns the working key as a TR-31
     * WrappedKeyBlock, where the wrapping key is the ECDH derived key.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html">
     * GetParametersForExport</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param exportKeyRequest
     * @return A Java Future containing the result of the ExportKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ExportKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ExportKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportKeyResponse> exportKey(ExportKeyRequest exportKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ExportKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportKeyRequest, ExportKeyResponse>().withOperationName("ExportKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportKeyRequest));
            CompletableFuture<ExportKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the Amazon Web Services Payment Cryptography key associated with the alias.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateAlias.html">CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html">DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListAliases.html">ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAliasRequest
     * @return A Java Future containing the result of the GetAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetAlias" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAliasResponse> getAlias(GetAliasRequest getAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAliasRequest, GetAliasResponse>().withOperationName("GetAlias")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAliasRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAliasRequest));
            CompletableFuture<GetAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a certificate signing request (CSR) from a key pair.
     * </p>
     *
     * @param getCertificateSigningRequestRequest
     * @return A Java Future containing the result of the GetCertificateSigningRequest operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetCertificateSigningRequest
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetCertificateSigningRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCertificateSigningRequestResponse> getCertificateSigningRequest(
            GetCertificateSigningRequestRequest getCertificateSigningRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCertificateSigningRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCertificateSigningRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCertificateSigningRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCertificateSigningRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCertificateSigningRequestResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetCertificateSigningRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCertificateSigningRequestRequest, GetCertificateSigningRequestResponse>()
                            .withOperationName("GetCertificateSigningRequest").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCertificateSigningRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCertificateSigningRequestRequest));
            CompletableFuture<GetCertificateSigningRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of Amazon Web Services Regions where <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a> is currently enabled for your Amazon Web Services account.
     * </p>
     * <p>
     * This operation returns the current Multi-Region key replication configuration. New keys created in your account
     * will be automatically replicated to these regions unless explicitly overridden during key creation.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_EnableDefaultKeyReplicationRegions.html"
     * >EnableDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DisableDefaultKeyReplicationRegions.html"
     * >DisableDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getDefaultKeyReplicationRegionsRequest
     *        Input parameters for retrieving the account's default key replication regions. This operation requires no
     *        input parameters.
     * @return A Java Future containing the result of the GetDefaultKeyReplicationRegions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetDefaultKeyReplicationRegions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetDefaultKeyReplicationRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDefaultKeyReplicationRegionsResponse> getDefaultKeyReplicationRegions(
            GetDefaultKeyReplicationRegionsRequest getDefaultKeyReplicationRegionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDefaultKeyReplicationRegionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDefaultKeyReplicationRegionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDefaultKeyReplicationRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDefaultKeyReplicationRegionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDefaultKeyReplicationRegionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDefaultKeyReplicationRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDefaultKeyReplicationRegionsRequest, GetDefaultKeyReplicationRegionsResponse>()
                            .withOperationName("GetDefaultKeyReplicationRegions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDefaultKeyReplicationRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDefaultKeyReplicationRegionsRequest));
            CompletableFuture<GetDefaultKeyReplicationRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the key metadata for an Amazon Web Services Payment Cryptography key, including the immutable and mutable
     * attributes specified when the key was created. Returns key metadata including attributes, state, and timestamps,
     * but does not return the actual cryptographic key material.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html">DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListKeys.html">ListKeys</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getKeyRequest
     * @return A Java Future containing the result of the GetKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetKeyResponse> getKey(GetKeyRequest getKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKeyRequest, GetKeyResponse>().withOperationName("GetKey")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKeyRequest));
            CompletableFuture<GetKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the export token and the signing key certificate to initiate a TR-34 key export from Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * The signing key certificate signs the wrapped key under export within the TR-34 key payload. The export token and
     * signing key certificate must be in place and operational before calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ExportKey.html">ExportKey</a>. The
     * export token expires in 30 days. You can use the same export token to export multiple keys from your service
     * account.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ExportKey.html">ExportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html">
     * GetParametersForImport</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getParametersForExportRequest
     * @return A Java Future containing the result of the GetParametersForExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetParametersForExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetParametersForExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetParametersForExportResponse> getParametersForExport(
            GetParametersForExportRequest getParametersForExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParametersForExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParametersForExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParametersForExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParametersForExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParametersForExportResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetParametersForExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParametersForExportRequest, GetParametersForExportResponse>()
                            .withOperationName("GetParametersForExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetParametersForExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getParametersForExportRequest));
            CompletableFuture<GetParametersForExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the import token and the wrapping key certificate in PEM format (base64 encoded) to initiate a TR-34
     * WrappedKeyBlock or a RSA WrappedKeyCryptogram import into Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * The wrapping key certificate wraps the key under import. The import token and wrapping key certificate must be in
     * place and operational before calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>. The
     * import token expires in 30 days. You can use the same import token to import multiple keys into your service
     * account.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html">
     * GetParametersForExport</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ImportKey.html">ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getParametersForImportRequest
     * @return A Java Future containing the result of the GetParametersForImport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetParametersForImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetParametersForImport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetParametersForImportResponse> getParametersForImport(
            GetParametersForImportRequest getParametersForImportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParametersForImportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParametersForImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParametersForImport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParametersForImportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParametersForImportResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetParametersForImportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParametersForImportRequest, GetParametersForImportResponse>()
                            .withOperationName("GetParametersForImport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetParametersForImportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getParametersForImportRequest));
            CompletableFuture<GetParametersForImportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the public key certificate of the asymmetric key pair that exists within Amazon Web Services Payment
     * Cryptography.
     * </p>
     * <p>
     * Unlike the private key of an asymmetric key, which never leaves Amazon Web Services Payment Cryptography
     * unencrypted, callers with <code>GetPublicKeyCertificate</code> permission can download the public key certificate
     * of the asymmetric key. You can share the public key certificate to allow others to encrypt messages and verify
     * signatures outside of Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     *
     * @param getPublicKeyCertificateRequest
     * @return A Java Future containing the result of the GetPublicKeyCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetPublicKeyCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetPublicKeyCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPublicKeyCertificateResponse> getPublicKeyCertificate(
            GetPublicKeyCertificateRequest getPublicKeyCertificateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPublicKeyCertificateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPublicKeyCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPublicKeyCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPublicKeyCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPublicKeyCertificateResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPublicKeyCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPublicKeyCertificateRequest, GetPublicKeyCertificateResponse>()
                            .withOperationName("GetPublicKeyCertificate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPublicKeyCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPublicKeyCertificateRequest));
            CompletableFuture<GetPublicKeyCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports symmetric keys and public key certificates in PEM format (base64 encoded) into Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography simplifies key exchange by replacing the existing paper-based approach
     * with a modern electronic approach. With <code>ImportKey</code> you can import symmetric keys using either
     * symmetric and asymmetric key exchange mechanisms.
     * </p>
     * <p>
     * For symmetric key exchange, Amazon Web Services Payment Cryptography uses the ANSI X9 TR-31 norm in accordance
     * with PCI PIN guidelines. And for asymmetric key exchange, Amazon Web Services Payment Cryptography supports ANSI
     * X9 TR-34 norm, RSA unwrap, and ECDH (Elliptic Curve Diffie-Hellman) key exchange mechanisms. Asymmetric key
     * exchange methods are typically used to establish bi-directional trust between the two parties exhanging keys and
     * are used for initial key exchange such as Key Encryption Key (KEK) or Zone Master Key (ZMK). After which you can
     * import working keys using symmetric method to perform various cryptographic operations within Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * PCI requires specific minimum key strength of wrapping keys used to protect the keys being exchanged
     * electronically. These requirements can change when PCI standards are revised. The rules specify that wrapping
     * keys used for transport must be at least as strong as the key being protected. For more information on
     * recommended key strength of wrapping keys and key exchange mechanism, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-importexport.html">Importing and
     * exporting keys</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * You can also import a <i>root public key certificate</i>, used to sign other public key certificates, or a
     * <i>trusted public key certificate</i> under an already established root public key certificate.
     * </p>
     * <p>
     * <b>To import a public root key certificate</b>
     * </p>
     * <p>
     * Using this operation, you can import the public component (in PEM cerificate format) of your private root key.
     * You can use the imported public root key certificate for digital signatures, for example signing wrapping key or
     * signing key in TR-34, within your Amazon Web Services Payment Cryptography account.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: <code>RootCertificatePublicKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyClass</code>: <code>PUBLIC_KEY</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyModesOfUse</code>: <code>Verify</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyUsage</code>: <code>TR31_S0_ASYMMETRIC_KEY_FOR_DIGITAL_SIGNATURE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) of the private root
     * key under import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import a trusted public key certificate</b>
     * </p>
     * <p>
     * The root public key certificate must be in place and operational before you import a trusted public key
     * certificate. Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: <code>TrustedCertificatePublicKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: <code>KeyArn</code> of the
     * <code>RootCertificatePublicKey</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyModesOfUse</code> and <code>KeyUsage</code>: Corresponding to the cryptographic operations such as wrap,
     * sign, or encrypt that you will allow the trusted public key certificate to perform.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The trusted public key certificate in PEM format (base64 encoded) under
     * import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import initial keys (KEK or ZMK or similar) using TR-34</b>
     * </p>
     * <p>
     * Using this operation, you can import initial key using TR-34 asymmetric key exchange. In TR-34 terminology, the
     * sending party of the key is called Key Distribution Host (KDH) and the receiving party of the key is called Key
     * Receiving Device (KRD). During the key import process, KDH is the user who initiates the key import and KRD is
     * Amazon Web Services Payment Cryptography who receives the key.
     * </p>
     * <p>
     * To initiate TR-34 key import, the KDH must obtain an import token by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html"
     * >GetParametersForImport</a>. This operation generates an encryption keypair for the purpose of key import, signs
     * the key and returns back the wrapping key certificate (also known as KRD wrapping certificate) and the root
     * certificate chain. The KDH must trust and install the KRD wrapping certificate on its HSM and use it to encrypt
     * (wrap) the KDH key during TR-34 WrappedKeyBlock generation. The import token and associated KRD wrapping
     * certificate expires after 30 days.
     * </p>
     * <p>
     * Next the KDH generates a key pair for the purpose of signing the encrypted KDH key and provides the public
     * certificate of the signing key to Amazon Web Services Payment Cryptography. The KDH will also need to import the
     * root certificate chain of the KDH signing certificate by calling <code>ImportKey</code> for
     * <code>RootCertificatePublicKey</code>. For more information on TR-34 key import, see section <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-import.html">Importing symmetric
     * keys</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr34KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>KeyARN</code> of the certificate chain that
     * signed the KDH signing key certificate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ImportToken</code>: Obtained from KRD by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html"
     * >GetParametersForImport</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappedKeyBlock</code>: The TR-34 wrapped key material from KDH. It contains the KDH key under import,
     * wrapped with KRD wrapping certificate and signed by KDH signing private key. This TR-34 key block is typically
     * generated by the KDH Hardware Security Module (HSM) outside of Amazon Web Services Payment Cryptography.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SigningKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) of the KDH signing
     * key generated under the root certificate (CertificateAuthorityPublicKeyIdentifier) imported in Amazon Web
     * Services Payment Cryptography.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import initial keys (KEK or ZMK or similar) using RSA Wrap and Unwrap</b>
     * </p>
     * <p>
     * Using this operation, you can import initial key using asymmetric RSA wrap and unwrap key exchange method. To
     * initiate import, call <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html"
     * >GetParametersForImport</a> with <code>KeyMaterial</code> set to <code>KEY_CRYPTOGRAM</code> to generate an
     * import token. This operation also generates an encryption keypair for the purpose of key import, signs the key
     * and returns back the wrapping key certificate in PEM format (base64 encoded) and its root certificate chain. The
     * import token and associated KRD wrapping certificate expires after 30 days.
     * </p>
     * <p>
     * You must trust and install the wrapping certificate and its certificate chain on the sending HSM and use it to
     * wrap the key under export for WrappedKeyCryptogram generation. Next call <code>ImportKey</code> with
     * <code>KeyMaterial</code> set to <code>KEY_CRYPTOGRAM</code> and provide the <code>ImportToken</code> and
     * <code>KeyAttributes</code> for the key under import.
     * </p>
     * <p>
     * <b>To import working keys using TR-31</b>
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography uses TR-31 symmetric key exchange norm to import working keys. A KEK
     * must be established within Amazon Web Services Payment Cryptography by using TR-34 key import or by using <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>. To
     * initiate a TR-31 key import, set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr31KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappedKeyBlock</code>: The TR-31 wrapped key material. It contains the key under import, encrypted using
     * KEK. The TR-31 key block is typically generated by a HSM outside of Amazon Web Services Payment Cryptography.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappingKeyIdentifier</code>: The <code>KeyArn</code> of the KEK that Amazon Web Services Payment
     * Cryptography uses to decrypt or unwrap the key under import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import working keys using ECDH</b>
     * </p>
     * <p>
     * You can also use ECDH key agreement to import working keys as a TR-31 keyblock, where the wrapping key is an ECDH
     * derived key.
     * </p>
     * <p>
     * To initiate a TR-31 key import using ECDH, both sides must create an ECC key pair with key usage K3 and exchange
     * public key certificates. In Amazon Web Services Payment Cryptography, you can do this by calling
     * <code>CreateKey</code> and then <code>GetPublicKeyCertificate</code> to retrieve its public key certificate.
     * Next, you can then generate a TR-31 WrappedKeyBlock using your own ECC key pair, the public certificate of the
     * service's ECC key pair, and the key derivation parameters including key derivation function, hash algorithm,
     * derivation data, and key algorithm. If you have not already done so, you must import the CA chain that issued the
     * receiving public key certificate by calling <code>ImportKey</code> with input
     * <code>RootCertificatePublicKey</code> for root CA or <code>TrustedPublicKey</code> for intermediate CA. To
     * complete the TR-31 key import, you can use the following parameters. It is important that the ECDH key derivation
     * parameters you use should match those used during import to derive the same shared wrapping key within Amazon Web
     * Services Payment Cryptography.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>DiffieHellmanTr31KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PrivateKeyIdentifier</code>: The <code>KeyArn</code> of the ECC key pair created within Amazon Web Services
     * Payment Cryptography to derive a shared KEK.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The public key certificate of the receiving ECC key pair in PEM format (base64
     * encoded) to derive a shared KEK.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>keyARN</code> of the CA that signed the public
     * key certificate of the receiving ECC key pair.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ExportKey.html">ExportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForImport.html">
     * GetParametersForImport</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param importKeyRequest
     * @return A Java Future containing the result of the ImportKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ImportKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ImportKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportKeyResponse> importKey(ImportKeyRequest importKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ImportKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ImportKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportKeyRequest, ImportKeyResponse>().withOperationName("ImportKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(importKeyRequest));
            CompletableFuture<ImportKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the aliases for all keys in the caller's Amazon Web Services account and Amazon Web Services Region. You
     * can filter the aliases by <code>keyARN</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-managealias.html">Using aliases</a>
     * in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the aliases.
     * When the response contains only a subset of aliases, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>ListAliases</code> request to get more aliases. When you receive a response with no
     * NextToken (or an empty or null value), that means there are no more aliases to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateAlias.html">CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html">DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetAlias.html">GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UpdateAlias.html">UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listAliasesRequest
     * @return A Java Future containing the result of the ListAliases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ListAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAliasesResponse> listAliases(ListAliasesRequest listAliasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAliasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAliasesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAliasesRequest, ListAliasesResponse>()
                            .withOperationName("ListAliases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAliasesRequest));
            CompletableFuture<ListAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the keys in the caller's Amazon Web Services account and Amazon Web Services Region. You can filter the
     * list of keys.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the keys. When
     * the response contains only a subset of keys, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListKeys</code> request to get more keys. When you receive a response with no NextToken (or an
     * empty or null value), that means there are no more keys to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html">DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetKey.html">GetKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listKeysRequest
     * @return A Java Future containing the result of the ListKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ListKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListKeys" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListKeysResponse> listKeys(ListKeysRequest listKeysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKeysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListKeysResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKeysRequest, ListKeysResponse>().withOperationName("ListKeys")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKeysRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listKeysRequest));
            CompletableFuture<ListKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for an Amazon Web Services resource.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTagsForResource</code> request to get more tags. When you receive a response with no
     * NextToken (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html">TagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UntagResource.html">
     * UntagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes Replication Regions from an existing Amazon Web Services Payment Cryptography key, disabling the key's
     * availability for cryptographic operations in the specified Amazon Web Services Regions.
     * </p>
     * <p>
     * When you remove Replication Regions, the key material is securely deleted from those regions and can no longer be
     * used for cryptographic operations there. This operation is irreversible for the specified Amazon Web Services
     * Regions. For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a>.
     * </p>
     * <important>
     * <p>
     * Ensure that no active cryptographic operations or applications depend on the key in the regions you're removing
     * before performing this operation.
     * </p>
     * </important>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_AddKeyReplicationRegions.html">
     * AddKeyReplicationRegions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DisableDefaultKeyReplicationRegions.html"
     * >DisableDefaultKeyReplicationRegions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param removeKeyReplicationRegionsRequest
     *        Input parameters for removing replication regions from a specific key.
     * @return A Java Future containing the result of the RemoveKeyReplicationRegions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.RemoveKeyReplicationRegions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/RemoveKeyReplicationRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveKeyReplicationRegionsResponse> removeKeyReplicationRegions(
            RemoveKeyReplicationRegionsRequest removeKeyReplicationRegionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeKeyReplicationRegionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeKeyReplicationRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveKeyReplicationRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveKeyReplicationRegionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveKeyReplicationRegionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RemoveKeyReplicationRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveKeyReplicationRegionsRequest, RemoveKeyReplicationRegionsResponse>()
                            .withOperationName("RemoveKeyReplicationRegions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RemoveKeyReplicationRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(removeKeyReplicationRegionsRequest));
            CompletableFuture<RemoveKeyReplicationRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a scheduled key deletion during the waiting period. Use this operation to restore a <code>Key</code> that
     * is scheduled for deletion.
     * </p>
     * <p>
     * During the waiting period, the <code>KeyState</code> is <code>DELETE_PENDING</code> and
     * <code>deletePendingTimestamp</code> contains the date and time after which the <code>Key</code> will be deleted.
     * After <code>Key</code> is restored, the <code>KeyState</code> is <code>CREATE_COMPLETE</code>, and the value for
     * <code>deletePendingTimestamp</code> is removed.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html">DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StartKeyUsage.html">
     * StartKeyUsage</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StopKeyUsage.html">StopKeyUsage<
     * /a>
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreKeyRequest
     * @return A Java Future containing the result of the RestoreKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.RestoreKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/RestoreKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreKeyResponse> restoreKey(RestoreKeyRequest restoreKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RestoreKeyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RestoreKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreKeyRequest, RestoreKeyResponse>().withOperationName("RestoreKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreKeyRequest));
            CompletableFuture<RestoreKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables an Amazon Web Services Payment Cryptography key, which makes it active for cryptographic operations
     * within Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StopKeyUsage.html">StopKeyUsage<
     * /a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startKeyUsageRequest
     * @return A Java Future containing the result of the StartKeyUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.StartKeyUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/StartKeyUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartKeyUsageResponse> startKeyUsage(StartKeyUsageRequest startKeyUsageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startKeyUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startKeyUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartKeyUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartKeyUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartKeyUsageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartKeyUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartKeyUsageRequest, StartKeyUsageResponse>()
                            .withOperationName("StartKeyUsage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartKeyUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startKeyUsageRequest));
            CompletableFuture<StartKeyUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables an Amazon Web Services Payment Cryptography key, which makes it inactive within Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * You can use this operation instead of <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html">DeleteKey</a> to
     * deactivate a key. You can enable the key in the future by calling <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StartKeyUsage.html"
     * >StartKeyUsage</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteKey.html">DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_StartKeyUsage.html">
     * StartKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopKeyUsageRequest
     * @return A Java Future containing the result of the StopKeyUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.StopKeyUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/StopKeyUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopKeyUsageResponse> stopKeyUsage(StopKeyUsageRequest stopKeyUsageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopKeyUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopKeyUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopKeyUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopKeyUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopKeyUsageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopKeyUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopKeyUsageRequest, StopKeyUsageResponse>()
                            .withOperationName("StopKeyUsage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopKeyUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopKeyUsageRequest));
            CompletableFuture<StopKeyUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or edits tags on an Amazon Web Services Payment Cryptography key.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * Each tag consists of a tag key and a tag value, both of which are case-sensitive strings. The tag value can be an
     * empty (null) string. To add a tag, specify a new tag key and a tag value. To edit a tag, specify an existing tag
     * key and a new tag value. You can also add tags to an Amazon Web Services Payment Cryptography key when you create
     * it with <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateKey.html">CreateKey</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListTagsForResource.html">
     * ListTagsForResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_UntagResource.html">
     * UntagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</p>
     *         <p>
     *         You have reached the maximum number of keys, aliases, or other resources allowed in your account. Review
     *         your current usage and consider deleting unused resources or requesting a quota increase.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a tag from an Amazon Web Services Payment Cryptography key.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListTagsForResource.html">
     * ListTagsForResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html">TagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates an existing Amazon Web Services Payment Cryptography alias with a different key. Each alias is
     * associated with only one Amazon Web Services Payment Cryptography key at a time, although a key can have multiple
     * aliases. The alias and the Amazon Web Services Payment Cryptography key must be in the same Amazon Web Services
     * account and Amazon Web Services Region
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_CreateAlias.html">CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_DeleteAlias.html">DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetAlias.html">GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_ListAliases.html">ListAliases</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateAliasRequest
     * @return A Java Future containing the result of the UpdateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</p>
     *         <p>
     *         The Amazon Web Services Payment Cryptography service is temporarily unavailable. This is typically a
     *         temporary condition - retry your request after a brief delay.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</p>
     *         <p>
     *         One or more parameters in your request are invalid. Check the parameter values, formats, and constraints
     *         specified in the API documentation.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</p>
     *         <p>
     *         The requested operation conflicts with the current state of the resource. For example, attempting to
     *         delete a key that is currently being used, or trying to create a resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         This exception is thrown when the caller lacks the necessary IAM permissions to perform the requested
     *         operation. Verify that your IAM policy includes the required permissions for the specific Amazon Web
     *         Services Payment Cryptography action you're attempting.</li>
     *         <li>ResourceNotFoundException The request was denied due to resource not found.</p>
     *         <p>
     *         The specified key, alias, or other resource does not exist in your account or region. Verify that the
     *         resource identifier is correct and that the resource exists in the expected region.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         You have exceeded the rate limits for Amazon Web Services Payment Cryptography API calls. Implement
     *         exponential backoff and retry logic in your application to handle throttling gracefully.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         This indicates a server-side error within the Amazon Web Services Payment Cryptography service. If this
     *         error persists, contact support for assistance.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.UpdateAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/UpdateAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAliasResponse> updateAlias(UpdateAliasRequest updateAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAliasResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAliasRequest, UpdateAliasResponse>()
                            .withOperationName("UpdateAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAliasRequest));
            CompletableFuture<UpdateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PaymentCryptographyServiceClientConfiguration serviceClientConfiguration() {
        return new PaymentCryptographyServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PaymentCryptographyException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        PaymentCryptographyServiceClientConfigurationBuilder serviceConfigBuilder = new PaymentCryptographyServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
