/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.partnercentralbenefits.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A summary view of a benefit application containing key information for list operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BenefitApplicationSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<BenefitApplicationSummary.Builder, BenefitApplicationSummary> {
    private static final SdkField<String> CATALOG_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Catalog")
            .getter(getter(BenefitApplicationSummary::catalog)).setter(setter(Builder::catalog))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Catalog").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(BenefitApplicationSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(BenefitApplicationSummary::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(BenefitApplicationSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> BENEFIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BenefitId").getter(getter(BenefitApplicationSummary::benefitId)).setter(setter(Builder::benefitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BenefitId").build()).build();

    private static final SdkField<List<String>> PROGRAMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Programs")
            .getter(getter(BenefitApplicationSummary::programs))
            .setter(setter(Builder::programs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Programs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> FULFILLMENT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FulfillmentTypes")
            .getter(getter(BenefitApplicationSummary::fulfillmentTypesAsStrings))
            .setter(setter(Builder::fulfillmentTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FulfillmentTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(BenefitApplicationSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Stage")
            .getter(getter(BenefitApplicationSummary::stage)).setter(setter(Builder::stage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Stage").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(BenefitApplicationSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt")
            .getter(getter(BenefitApplicationSummary::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, String>> BENEFIT_APPLICATION_DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("BenefitApplicationDetails")
            .getter(getter(BenefitApplicationSummary::benefitApplicationDetails))
            .setter(setter(Builder::benefitApplicationDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BenefitApplicationDetails").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> ASSOCIATED_RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AssociatedResources")
            .getter(getter(BenefitApplicationSummary::associatedResources))
            .setter(setter(Builder::associatedResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CATALOG_FIELD, NAME_FIELD,
            ID_FIELD, ARN_FIELD, BENEFIT_ID_FIELD, PROGRAMS_FIELD, FULFILLMENT_TYPES_FIELD, STATUS_FIELD, STAGE_FIELD,
            CREATED_AT_FIELD, UPDATED_AT_FIELD, BENEFIT_APPLICATION_DETAILS_FIELD, ASSOCIATED_RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String catalog;

    private final String name;

    private final String id;

    private final String arn;

    private final String benefitId;

    private final List<String> programs;

    private final List<String> fulfillmentTypes;

    private final String status;

    private final String stage;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final Map<String, String> benefitApplicationDetails;

    private final List<String> associatedResources;

    private BenefitApplicationSummary(BuilderImpl builder) {
        this.catalog = builder.catalog;
        this.name = builder.name;
        this.id = builder.id;
        this.arn = builder.arn;
        this.benefitId = builder.benefitId;
        this.programs = builder.programs;
        this.fulfillmentTypes = builder.fulfillmentTypes;
        this.status = builder.status;
        this.stage = builder.stage;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.benefitApplicationDetails = builder.benefitApplicationDetails;
        this.associatedResources = builder.associatedResources;
    }

    /**
     * <p>
     * The catalog identifier that the benefit application belongs to.
     * </p>
     * 
     * @return The catalog identifier that the benefit application belongs to.
     */
    public final String catalog() {
        return catalog;
    }

    /**
     * <p>
     * The human-readable name of the benefit application.
     * </p>
     * 
     * @return The human-readable name of the benefit application.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The unique identifier of the benefit application.
     * </p>
     * 
     * @return The unique identifier of the benefit application.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the benefit application.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the benefit application.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The identifier of the benefit being requested in this application.
     * </p>
     * 
     * @return The identifier of the benefit being requested in this application.
     */
    public final String benefitId() {
        return benefitId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Programs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPrograms() {
        return programs != null && !(programs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The AWS partner programs associated with this benefit application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrograms} method.
     * </p>
     * 
     * @return The AWS partner programs associated with this benefit application.
     */
    public final List<String> programs() {
        return programs;
    }

    /**
     * <p>
     * The fulfillment types requested for this benefit application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFulfillmentTypes} method.
     * </p>
     * 
     * @return The fulfillment types requested for this benefit application.
     */
    public final List<FulfillmentType> fulfillmentTypes() {
        return FulfillmentTypesCopier.copyStringToEnum(fulfillmentTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FulfillmentTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFulfillmentTypes() {
        return fulfillmentTypes != null && !(fulfillmentTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The fulfillment types requested for this benefit application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFulfillmentTypes} method.
     * </p>
     * 
     * @return The fulfillment types requested for this benefit application.
     */
    public final List<String> fulfillmentTypesAsStrings() {
        return fulfillmentTypes;
    }

    /**
     * <p>
     * The current processing status of the benefit application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BenefitApplicationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current processing status of the benefit application.
     * @see BenefitApplicationStatus
     */
    public final BenefitApplicationStatus status() {
        return BenefitApplicationStatus.fromValue(status);
    }

    /**
     * <p>
     * The current processing status of the benefit application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BenefitApplicationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current processing status of the benefit application.
     * @see BenefitApplicationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The current stage in the benefit application processing workflow..
     * </p>
     * 
     * @return The current stage in the benefit application processing workflow..
     */
    public final String stage() {
        return stage;
    }

    /**
     * <p>
     * The timestamp when the benefit application was created.
     * </p>
     * 
     * @return The timestamp when the benefit application was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the benefit application was last updated.
     * </p>
     * 
     * @return The timestamp when the benefit application was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the BenefitApplicationDetails property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasBenefitApplicationDetails() {
        return benefitApplicationDetails != null && !(benefitApplicationDetails instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional attributes and metadata associated with the benefit application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBenefitApplicationDetails} method.
     * </p>
     * 
     * @return Additional attributes and metadata associated with the benefit application.
     */
    public final Map<String, String> benefitApplicationDetails() {
        return benefitApplicationDetails;
    }

    /**
     * For responses, this returns true if the service returned a value for the AssociatedResources property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAssociatedResources() {
        return associatedResources != null && !(associatedResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * AWS resources that are associated with this benefit application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssociatedResources} method.
     * </p>
     * 
     * @return AWS resources that are associated with this benefit application.
     */
    public final List<String> associatedResources() {
        return associatedResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(catalog());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(benefitId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPrograms() ? programs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFulfillmentTypes() ? fulfillmentTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stage());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasBenefitApplicationDetails() ? benefitApplicationDetails() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAssociatedResources() ? associatedResources() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BenefitApplicationSummary)) {
            return false;
        }
        BenefitApplicationSummary other = (BenefitApplicationSummary) obj;
        return Objects.equals(catalog(), other.catalog()) && Objects.equals(name(), other.name())
                && Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(benefitId(), other.benefitId()) && hasPrograms() == other.hasPrograms()
                && Objects.equals(programs(), other.programs()) && hasFulfillmentTypes() == other.hasFulfillmentTypes()
                && Objects.equals(fulfillmentTypesAsStrings(), other.fulfillmentTypesAsStrings())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(stage(), other.stage())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && hasBenefitApplicationDetails() == other.hasBenefitApplicationDetails()
                && Objects.equals(benefitApplicationDetails(), other.benefitApplicationDetails())
                && hasAssociatedResources() == other.hasAssociatedResources()
                && Objects.equals(associatedResources(), other.associatedResources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BenefitApplicationSummary").add("Catalog", catalog()).add("Name", name()).add("Id", id())
                .add("Arn", arn()).add("BenefitId", benefitId()).add("Programs", hasPrograms() ? programs() : null)
                .add("FulfillmentTypes", hasFulfillmentTypes() ? fulfillmentTypesAsStrings() : null)
                .add("Status", statusAsString()).add("Stage", stage()).add("CreatedAt", createdAt())
                .add("UpdatedAt", updatedAt())
                .add("BenefitApplicationDetails", hasBenefitApplicationDetails() ? benefitApplicationDetails() : null)
                .add("AssociatedResources", hasAssociatedResources() ? associatedResources() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Catalog":
            return Optional.ofNullable(clazz.cast(catalog()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "BenefitId":
            return Optional.ofNullable(clazz.cast(benefitId()));
        case "Programs":
            return Optional.ofNullable(clazz.cast(programs()));
        case "FulfillmentTypes":
            return Optional.ofNullable(clazz.cast(fulfillmentTypesAsStrings()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Stage":
            return Optional.ofNullable(clazz.cast(stage()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "BenefitApplicationDetails":
            return Optional.ofNullable(clazz.cast(benefitApplicationDetails()));
        case "AssociatedResources":
            return Optional.ofNullable(clazz.cast(associatedResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Catalog", CATALOG_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Id", ID_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("BenefitId", BENEFIT_ID_FIELD);
        map.put("Programs", PROGRAMS_FIELD);
        map.put("FulfillmentTypes", FULFILLMENT_TYPES_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Stage", STAGE_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("UpdatedAt", UPDATED_AT_FIELD);
        map.put("BenefitApplicationDetails", BENEFIT_APPLICATION_DETAILS_FIELD);
        map.put("AssociatedResources", ASSOCIATED_RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BenefitApplicationSummary, T> g) {
        return obj -> g.apply((BenefitApplicationSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BenefitApplicationSummary> {
        /**
         * <p>
         * The catalog identifier that the benefit application belongs to.
         * </p>
         * 
         * @param catalog
         *        The catalog identifier that the benefit application belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalog(String catalog);

        /**
         * <p>
         * The human-readable name of the benefit application.
         * </p>
         * 
         * @param name
         *        The human-readable name of the benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique identifier of the benefit application.
         * </p>
         * 
         * @param id
         *        The unique identifier of the benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the benefit application.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The identifier of the benefit being requested in this application.
         * </p>
         * 
         * @param benefitId
         *        The identifier of the benefit being requested in this application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder benefitId(String benefitId);

        /**
         * <p>
         * The AWS partner programs associated with this benefit application.
         * </p>
         * 
         * @param programs
         *        The AWS partner programs associated with this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder programs(Collection<String> programs);

        /**
         * <p>
         * The AWS partner programs associated with this benefit application.
         * </p>
         * 
         * @param programs
         *        The AWS partner programs associated with this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder programs(String... programs);

        /**
         * <p>
         * The fulfillment types requested for this benefit application.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types requested for this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypesWithStrings(Collection<String> fulfillmentTypes);

        /**
         * <p>
         * The fulfillment types requested for this benefit application.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types requested for this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypesWithStrings(String... fulfillmentTypes);

        /**
         * <p>
         * The fulfillment types requested for this benefit application.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types requested for this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypes(Collection<FulfillmentType> fulfillmentTypes);

        /**
         * <p>
         * The fulfillment types requested for this benefit application.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types requested for this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypes(FulfillmentType... fulfillmentTypes);

        /**
         * <p>
         * The current processing status of the benefit application.
         * </p>
         * 
         * @param status
         *        The current processing status of the benefit application.
         * @see BenefitApplicationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BenefitApplicationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current processing status of the benefit application.
         * </p>
         * 
         * @param status
         *        The current processing status of the benefit application.
         * @see BenefitApplicationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BenefitApplicationStatus
         */
        Builder status(BenefitApplicationStatus status);

        /**
         * <p>
         * The current stage in the benefit application processing workflow..
         * </p>
         * 
         * @param stage
         *        The current stage in the benefit application processing workflow..
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stage(String stage);

        /**
         * <p>
         * The timestamp when the benefit application was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the benefit application was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the benefit application was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The timestamp when the benefit application was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * Additional attributes and metadata associated with the benefit application.
         * </p>
         * 
         * @param benefitApplicationDetails
         *        Additional attributes and metadata associated with the benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder benefitApplicationDetails(Map<String, String> benefitApplicationDetails);

        /**
         * <p>
         * AWS resources that are associated with this benefit application.
         * </p>
         * 
         * @param associatedResources
         *        AWS resources that are associated with this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedResources(Collection<String> associatedResources);

        /**
         * <p>
         * AWS resources that are associated with this benefit application.
         * </p>
         * 
         * @param associatedResources
         *        AWS resources that are associated with this benefit application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedResources(String... associatedResources);
    }

    static final class BuilderImpl implements Builder {
        private String catalog;

        private String name;

        private String id;

        private String arn;

        private String benefitId;

        private List<String> programs = DefaultSdkAutoConstructList.getInstance();

        private List<String> fulfillmentTypes = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String stage;

        private Instant createdAt;

        private Instant updatedAt;

        private Map<String, String> benefitApplicationDetails = DefaultSdkAutoConstructMap.getInstance();

        private List<String> associatedResources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BenefitApplicationSummary model) {
            catalog(model.catalog);
            name(model.name);
            id(model.id);
            arn(model.arn);
            benefitId(model.benefitId);
            programs(model.programs);
            fulfillmentTypesWithStrings(model.fulfillmentTypes);
            status(model.status);
            stage(model.stage);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            benefitApplicationDetails(model.benefitApplicationDetails);
            associatedResources(model.associatedResources);
        }

        public final String getCatalog() {
            return catalog;
        }

        public final void setCatalog(String catalog) {
            this.catalog = catalog;
        }

        @Override
        public final Builder catalog(String catalog) {
            this.catalog = catalog;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getBenefitId() {
            return benefitId;
        }

        public final void setBenefitId(String benefitId) {
            this.benefitId = benefitId;
        }

        @Override
        public final Builder benefitId(String benefitId) {
            this.benefitId = benefitId;
            return this;
        }

        public final Collection<String> getPrograms() {
            if (programs instanceof SdkAutoConstructList) {
                return null;
            }
            return programs;
        }

        public final void setPrograms(Collection<String> programs) {
            this.programs = ProgramsCopier.copy(programs);
        }

        @Override
        public final Builder programs(Collection<String> programs) {
            this.programs = ProgramsCopier.copy(programs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder programs(String... programs) {
            programs(Arrays.asList(programs));
            return this;
        }

        public final Collection<String> getFulfillmentTypes() {
            if (fulfillmentTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return fulfillmentTypes;
        }

        public final void setFulfillmentTypes(Collection<String> fulfillmentTypes) {
            this.fulfillmentTypes = FulfillmentTypesCopier.copy(fulfillmentTypes);
        }

        @Override
        public final Builder fulfillmentTypesWithStrings(Collection<String> fulfillmentTypes) {
            this.fulfillmentTypes = FulfillmentTypesCopier.copy(fulfillmentTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fulfillmentTypesWithStrings(String... fulfillmentTypes) {
            fulfillmentTypesWithStrings(Arrays.asList(fulfillmentTypes));
            return this;
        }

        @Override
        public final Builder fulfillmentTypes(Collection<FulfillmentType> fulfillmentTypes) {
            this.fulfillmentTypes = FulfillmentTypesCopier.copyEnumToString(fulfillmentTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fulfillmentTypes(FulfillmentType... fulfillmentTypes) {
            fulfillmentTypes(Arrays.asList(fulfillmentTypes));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(BenefitApplicationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStage() {
            return stage;
        }

        public final void setStage(String stage) {
            this.stage = stage;
        }

        @Override
        public final Builder stage(String stage) {
            this.stage = stage;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final Map<String, String> getBenefitApplicationDetails() {
            if (benefitApplicationDetails instanceof SdkAutoConstructMap) {
                return null;
            }
            return benefitApplicationDetails;
        }

        public final void setBenefitApplicationDetails(Map<String, String> benefitApplicationDetails) {
            this.benefitApplicationDetails = AttributesCopier.copy(benefitApplicationDetails);
        }

        @Override
        public final Builder benefitApplicationDetails(Map<String, String> benefitApplicationDetails) {
            this.benefitApplicationDetails = AttributesCopier.copy(benefitApplicationDetails);
            return this;
        }

        public final Collection<String> getAssociatedResources() {
            if (associatedResources instanceof SdkAutoConstructList) {
                return null;
            }
            return associatedResources;
        }

        public final void setAssociatedResources(Collection<String> associatedResources) {
            this.associatedResources = ArnsCopier.copy(associatedResources);
        }

        @Override
        public final Builder associatedResources(Collection<String> associatedResources) {
            this.associatedResources = ArnsCopier.copy(associatedResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedResources(String... associatedResources) {
            associatedResources(Arrays.asList(associatedResources));
            return this;
        }

        @Override
        public BenefitApplicationSummary build() {
            return new BenefitApplicationSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
