/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.partnercentralbenefits.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A summary view of a benefit allocation containing key information for list operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BenefitAllocationSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<BenefitAllocationSummary.Builder, BenefitAllocationSummary> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(BenefitAllocationSummary::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> CATALOG_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Catalog")
            .getter(getter(BenefitAllocationSummary::catalog)).setter(setter(Builder::catalog))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Catalog").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(BenefitAllocationSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(BenefitAllocationSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(BenefitAllocationSummary::statusReason))
            .setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(BenefitAllocationSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> BENEFIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BenefitId").getter(getter(BenefitAllocationSummary::benefitId)).setter(setter(Builder::benefitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BenefitId").build()).build();

    private static final SdkField<String> BENEFIT_APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BenefitApplicationId").getter(getter(BenefitAllocationSummary::benefitApplicationId))
            .setter(setter(Builder::benefitApplicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BenefitApplicationId").build())
            .build();

    private static final SdkField<List<String>> FULFILLMENT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FulfillmentTypes")
            .getter(getter(BenefitAllocationSummary::fulfillmentTypesAsStrings))
            .setter(setter(Builder::fulfillmentTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FulfillmentTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(BenefitAllocationSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> EXPIRES_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpiresAt")
            .getter(getter(BenefitAllocationSummary::expiresAt))
            .setter(setter(Builder::expiresAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpiresAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<String>> APPLICABLE_BENEFIT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ApplicableBenefitIds")
            .getter(getter(BenefitAllocationSummary::applicableBenefitIds))
            .setter(setter(Builder::applicableBenefitIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicableBenefitIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, CATALOG_FIELD,
            ARN_FIELD, STATUS_FIELD, STATUS_REASON_FIELD, NAME_FIELD, BENEFIT_ID_FIELD, BENEFIT_APPLICATION_ID_FIELD,
            FULFILLMENT_TYPES_FIELD, CREATED_AT_FIELD, EXPIRES_AT_FIELD, APPLICABLE_BENEFIT_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String catalog;

    private final String arn;

    private final String status;

    private final String statusReason;

    private final String name;

    private final String benefitId;

    private final String benefitApplicationId;

    private final List<String> fulfillmentTypes;

    private final Instant createdAt;

    private final Instant expiresAt;

    private final List<String> applicableBenefitIds;

    private BenefitAllocationSummary(BuilderImpl builder) {
        this.id = builder.id;
        this.catalog = builder.catalog;
        this.arn = builder.arn;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.name = builder.name;
        this.benefitId = builder.benefitId;
        this.benefitApplicationId = builder.benefitApplicationId;
        this.fulfillmentTypes = builder.fulfillmentTypes;
        this.createdAt = builder.createdAt;
        this.expiresAt = builder.expiresAt;
        this.applicableBenefitIds = builder.applicableBenefitIds;
    }

    /**
     * <p>
     * The unique identifier of the benefit allocation.
     * </p>
     * 
     * @return The unique identifier of the benefit allocation.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The catalog identifier that the benefit allocation belongs to.
     * </p>
     * 
     * @return The catalog identifier that the benefit allocation belongs to.
     */
    public final String catalog() {
        return catalog;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the benefit allocation.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the benefit allocation.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The current status of the benefit allocation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BenefitAllocationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the benefit allocation.
     * @see BenefitAllocationStatus
     */
    public final BenefitAllocationStatus status() {
        return BenefitAllocationStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the benefit allocation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BenefitAllocationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the benefit allocation.
     * @see BenefitAllocationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information explaining the current status of the benefit allocation.
     * </p>
     * 
     * @return Additional information explaining the current status of the benefit allocation.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The human-readable name of the benefit allocation.
     * </p>
     * 
     * @return The human-readable name of the benefit allocation.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The identifier of the benefit that this allocation is based on.
     * </p>
     * 
     * @return The identifier of the benefit that this allocation is based on.
     */
    public final String benefitId() {
        return benefitId;
    }

    /**
     * <p>
     * The identifier of the benefit application that resulted in this allocation.
     * </p>
     * 
     * @return The identifier of the benefit application that resulted in this allocation.
     */
    public final String benefitApplicationId() {
        return benefitApplicationId;
    }

    /**
     * <p>
     * The fulfillment types used for this benefit allocation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFulfillmentTypes} method.
     * </p>
     * 
     * @return The fulfillment types used for this benefit allocation.
     */
    public final List<FulfillmentType> fulfillmentTypes() {
        return FulfillmentTypesCopier.copyStringToEnum(fulfillmentTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FulfillmentTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFulfillmentTypes() {
        return fulfillmentTypes != null && !(fulfillmentTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The fulfillment types used for this benefit allocation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFulfillmentTypes} method.
     * </p>
     * 
     * @return The fulfillment types used for this benefit allocation.
     */
    public final List<String> fulfillmentTypesAsStrings() {
        return fulfillmentTypes;
    }

    /**
     * <p>
     * The timestamp when the benefit allocation was created.
     * </p>
     * 
     * @return The timestamp when the benefit allocation was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the benefit allocation expires.
     * </p>
     * 
     * @return The timestamp when the benefit allocation expires.
     */
    public final Instant expiresAt() {
        return expiresAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the ApplicableBenefitIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasApplicableBenefitIds() {
        return applicableBenefitIds != null && !(applicableBenefitIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of the benefits applicable for this allocation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApplicableBenefitIds} method.
     * </p>
     * 
     * @return The identifiers of the benefits applicable for this allocation.
     */
    public final List<String> applicableBenefitIds() {
        return applicableBenefitIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(catalog());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(benefitId());
        hashCode = 31 * hashCode + Objects.hashCode(benefitApplicationId());
        hashCode = 31 * hashCode + Objects.hashCode(hasFulfillmentTypes() ? fulfillmentTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(expiresAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasApplicableBenefitIds() ? applicableBenefitIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BenefitAllocationSummary)) {
            return false;
        }
        BenefitAllocationSummary other = (BenefitAllocationSummary) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(catalog(), other.catalog())
                && Objects.equals(arn(), other.arn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && Objects.equals(name(), other.name())
                && Objects.equals(benefitId(), other.benefitId())
                && Objects.equals(benefitApplicationId(), other.benefitApplicationId())
                && hasFulfillmentTypes() == other.hasFulfillmentTypes()
                && Objects.equals(fulfillmentTypesAsStrings(), other.fulfillmentTypesAsStrings())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(expiresAt(), other.expiresAt())
                && hasApplicableBenefitIds() == other.hasApplicableBenefitIds()
                && Objects.equals(applicableBenefitIds(), other.applicableBenefitIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BenefitAllocationSummary").add("Id", id()).add("Catalog", catalog()).add("Arn", arn())
                .add("Status", statusAsString()).add("StatusReason", statusReason()).add("Name", name())
                .add("BenefitId", benefitId()).add("BenefitApplicationId", benefitApplicationId())
                .add("FulfillmentTypes", hasFulfillmentTypes() ? fulfillmentTypesAsStrings() : null)
                .add("CreatedAt", createdAt()).add("ExpiresAt", expiresAt())
                .add("ApplicableBenefitIds", hasApplicableBenefitIds() ? applicableBenefitIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Catalog":
            return Optional.ofNullable(clazz.cast(catalog()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "BenefitId":
            return Optional.ofNullable(clazz.cast(benefitId()));
        case "BenefitApplicationId":
            return Optional.ofNullable(clazz.cast(benefitApplicationId()));
        case "FulfillmentTypes":
            return Optional.ofNullable(clazz.cast(fulfillmentTypesAsStrings()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "ExpiresAt":
            return Optional.ofNullable(clazz.cast(expiresAt()));
        case "ApplicableBenefitIds":
            return Optional.ofNullable(clazz.cast(applicableBenefitIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Catalog", CATALOG_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusReason", STATUS_REASON_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("BenefitId", BENEFIT_ID_FIELD);
        map.put("BenefitApplicationId", BENEFIT_APPLICATION_ID_FIELD);
        map.put("FulfillmentTypes", FULFILLMENT_TYPES_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("ExpiresAt", EXPIRES_AT_FIELD);
        map.put("ApplicableBenefitIds", APPLICABLE_BENEFIT_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BenefitAllocationSummary, T> g) {
        return obj -> g.apply((BenefitAllocationSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BenefitAllocationSummary> {
        /**
         * <p>
         * The unique identifier of the benefit allocation.
         * </p>
         * 
         * @param id
         *        The unique identifier of the benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The catalog identifier that the benefit allocation belongs to.
         * </p>
         * 
         * @param catalog
         *        The catalog identifier that the benefit allocation belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalog(String catalog);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the benefit allocation.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The current status of the benefit allocation.
         * </p>
         * 
         * @param status
         *        The current status of the benefit allocation.
         * @see BenefitAllocationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BenefitAllocationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the benefit allocation.
         * </p>
         * 
         * @param status
         *        The current status of the benefit allocation.
         * @see BenefitAllocationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BenefitAllocationStatus
         */
        Builder status(BenefitAllocationStatus status);

        /**
         * <p>
         * Additional information explaining the current status of the benefit allocation.
         * </p>
         * 
         * @param statusReason
         *        Additional information explaining the current status of the benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The human-readable name of the benefit allocation.
         * </p>
         * 
         * @param name
         *        The human-readable name of the benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The identifier of the benefit that this allocation is based on.
         * </p>
         * 
         * @param benefitId
         *        The identifier of the benefit that this allocation is based on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder benefitId(String benefitId);

        /**
         * <p>
         * The identifier of the benefit application that resulted in this allocation.
         * </p>
         * 
         * @param benefitApplicationId
         *        The identifier of the benefit application that resulted in this allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder benefitApplicationId(String benefitApplicationId);

        /**
         * <p>
         * The fulfillment types used for this benefit allocation.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types used for this benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypesWithStrings(Collection<String> fulfillmentTypes);

        /**
         * <p>
         * The fulfillment types used for this benefit allocation.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types used for this benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypesWithStrings(String... fulfillmentTypes);

        /**
         * <p>
         * The fulfillment types used for this benefit allocation.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types used for this benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypes(Collection<FulfillmentType> fulfillmentTypes);

        /**
         * <p>
         * The fulfillment types used for this benefit allocation.
         * </p>
         * 
         * @param fulfillmentTypes
         *        The fulfillment types used for this benefit allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fulfillmentTypes(FulfillmentType... fulfillmentTypes);

        /**
         * <p>
         * The timestamp when the benefit allocation was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the benefit allocation was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the benefit allocation expires.
         * </p>
         * 
         * @param expiresAt
         *        The timestamp when the benefit allocation expires.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiresAt(Instant expiresAt);

        /**
         * <p>
         * The identifiers of the benefits applicable for this allocation.
         * </p>
         * 
         * @param applicableBenefitIds
         *        The identifiers of the benefits applicable for this allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicableBenefitIds(Collection<String> applicableBenefitIds);

        /**
         * <p>
         * The identifiers of the benefits applicable for this allocation.
         * </p>
         * 
         * @param applicableBenefitIds
         *        The identifiers of the benefits applicable for this allocation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicableBenefitIds(String... applicableBenefitIds);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String catalog;

        private String arn;

        private String status;

        private String statusReason;

        private String name;

        private String benefitId;

        private String benefitApplicationId;

        private List<String> fulfillmentTypes = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private Instant expiresAt;

        private List<String> applicableBenefitIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BenefitAllocationSummary model) {
            id(model.id);
            catalog(model.catalog);
            arn(model.arn);
            status(model.status);
            statusReason(model.statusReason);
            name(model.name);
            benefitId(model.benefitId);
            benefitApplicationId(model.benefitApplicationId);
            fulfillmentTypesWithStrings(model.fulfillmentTypes);
            createdAt(model.createdAt);
            expiresAt(model.expiresAt);
            applicableBenefitIds(model.applicableBenefitIds);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getCatalog() {
            return catalog;
        }

        public final void setCatalog(String catalog) {
            this.catalog = catalog;
        }

        @Override
        public final Builder catalog(String catalog) {
            this.catalog = catalog;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(BenefitAllocationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getBenefitId() {
            return benefitId;
        }

        public final void setBenefitId(String benefitId) {
            this.benefitId = benefitId;
        }

        @Override
        public final Builder benefitId(String benefitId) {
            this.benefitId = benefitId;
            return this;
        }

        public final String getBenefitApplicationId() {
            return benefitApplicationId;
        }

        public final void setBenefitApplicationId(String benefitApplicationId) {
            this.benefitApplicationId = benefitApplicationId;
        }

        @Override
        public final Builder benefitApplicationId(String benefitApplicationId) {
            this.benefitApplicationId = benefitApplicationId;
            return this;
        }

        public final Collection<String> getFulfillmentTypes() {
            if (fulfillmentTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return fulfillmentTypes;
        }

        public final void setFulfillmentTypes(Collection<String> fulfillmentTypes) {
            this.fulfillmentTypes = FulfillmentTypesCopier.copy(fulfillmentTypes);
        }

        @Override
        public final Builder fulfillmentTypesWithStrings(Collection<String> fulfillmentTypes) {
            this.fulfillmentTypes = FulfillmentTypesCopier.copy(fulfillmentTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fulfillmentTypesWithStrings(String... fulfillmentTypes) {
            fulfillmentTypesWithStrings(Arrays.asList(fulfillmentTypes));
            return this;
        }

        @Override
        public final Builder fulfillmentTypes(Collection<FulfillmentType> fulfillmentTypes) {
            this.fulfillmentTypes = FulfillmentTypesCopier.copyEnumToString(fulfillmentTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fulfillmentTypes(FulfillmentType... fulfillmentTypes) {
            fulfillmentTypes(Arrays.asList(fulfillmentTypes));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getExpiresAt() {
            return expiresAt;
        }

        public final void setExpiresAt(Instant expiresAt) {
            this.expiresAt = expiresAt;
        }

        @Override
        public final Builder expiresAt(Instant expiresAt) {
            this.expiresAt = expiresAt;
            return this;
        }

        public final Collection<String> getApplicableBenefitIds() {
            if (applicableBenefitIds instanceof SdkAutoConstructList) {
                return null;
            }
            return applicableBenefitIds;
        }

        public final void setApplicableBenefitIds(Collection<String> applicableBenefitIds) {
            this.applicableBenefitIds = BenefitIdsCopier.copy(applicableBenefitIds);
        }

        @Override
        public final Builder applicableBenefitIds(Collection<String> applicableBenefitIds) {
            this.applicableBenefitIds = BenefitIdsCopier.copy(applicableBenefitIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder applicableBenefitIds(String... applicableBenefitIds) {
            applicableBenefitIds(Arrays.asList(applicableBenefitIds));
            return this;
        }

        @Override
        public BenefitAllocationSummary build() {
            return new BenefitAllocationSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
