/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a root. A root is a top-level parent node in the hierarchy of an organization that can contain
 * organizational units (OUs) and accounts. The root contains every AWS account in the organization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Root implements SdkPojo, Serializable, ToCopyableBuilder<Root.Builder, Root> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Root::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Root::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Root::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<PolicyTypeSummary>> POLICY_TYPES_FIELD = SdkField
            .<List<PolicyTypeSummary>> builder(MarshallingType.LIST)
            .memberName("PolicyTypes")
            .getter(getter(Root::policyTypes))
            .setter(setter(Builder::policyTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PolicyTypeSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(PolicyTypeSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            NAME_FIELD, POLICY_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final String name;

    private final List<PolicyTypeSummary> policyTypes;

    private Root(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.name = builder.name;
        this.policyTypes = builder.policyTypes;
    }

    /**
     * <p>
     * The unique identifier (ID) for the root.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a root ID string requires "r-" followed by
     * from 4 to 32 lowercase letters or digits.
     * </p>
     * 
     * @return The unique identifier (ID) for the root.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a root ID string requires "r-"
     *         followed by from 4 to 32 lowercase letters or digits.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the root.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
     * >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the root.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
     *         >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The friendly name of the root.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter is a
     * string of any of the characters in the ASCII character range.
     * </p>
     * 
     * @return The friendly name of the root.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter
     *         is a string of any of the characters in the ASCII character range.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns true if the PolicyTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasPolicyTypes() {
        return policyTypes != null && !(policyTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The types of policies that are currently enabled for the root and therefore can be attached to the root or to its
     * OUs or accounts.
     * </p>
     * <note>
     * <p>
     * Even if a policy type is shown as available in the organization, you can separately enable and disable them at
     * the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use <a>DescribeOrganization</a> to
     * see the availability of the policy types in that organization.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPolicyTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The types of policies that are currently enabled for the root and therefore can be attached to the root
     *         or to its OUs or accounts.</p> <note>
     *         <p>
     *         Even if a policy type is shown as available in the organization, you can separately enable and disable
     *         them at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
     *         <a>DescribeOrganization</a> to see the availability of the policy types in that organization.
     *         </p>
     */
    public final List<PolicyTypeSummary> policyTypes() {
        return policyTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasPolicyTypes() ? policyTypes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Root)) {
            return false;
        }
        Root other = (Root) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && hasPolicyTypes() == other.hasPolicyTypes() && Objects.equals(policyTypes(), other.policyTypes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Root").add("Id", id()).add("Arn", arn()).add("Name", name())
                .add("PolicyTypes", hasPolicyTypes() ? policyTypes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "PolicyTypes":
            return Optional.ofNullable(clazz.cast(policyTypes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Root, T> g) {
        return obj -> g.apply((Root) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Root> {
        /**
         * <p>
         * The unique identifier (ID) for the root.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a root ID string requires "r-" followed
         * by from 4 to 32 lowercase letters or digits.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) for the root.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a root ID string requires "r-"
         *        followed by from 4 to 32 lowercase letters or digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the root.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
         * >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the root.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
         *        >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The friendly name of the root.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter is a
         * string of any of the characters in the ASCII character range.
         * </p>
         * 
         * @param name
         *        The friendly name of the root.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this
         *        parameter is a string of any of the characters in the ASCII character range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The types of policies that are currently enabled for the root and therefore can be attached to the root or to
         * its OUs or accounts.
         * </p>
         * <note>
         * <p>
         * Even if a policy type is shown as available in the organization, you can separately enable and disable them
         * at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         * <a>DescribeOrganization</a> to see the availability of the policy types in that organization.
         * </p>
         * </note>
         * 
         * @param policyTypes
         *        The types of policies that are currently enabled for the root and therefore can be attached to the
         *        root or to its OUs or accounts.</p> <note>
         *        <p>
         *        Even if a policy type is shown as available in the organization, you can separately enable and disable
         *        them at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         *        <a>DescribeOrganization</a> to see the availability of the policy types in that organization.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyTypes(Collection<PolicyTypeSummary> policyTypes);

        /**
         * <p>
         * The types of policies that are currently enabled for the root and therefore can be attached to the root or to
         * its OUs or accounts.
         * </p>
         * <note>
         * <p>
         * Even if a policy type is shown as available in the organization, you can separately enable and disable them
         * at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         * <a>DescribeOrganization</a> to see the availability of the policy types in that organization.
         * </p>
         * </note>
         * 
         * @param policyTypes
         *        The types of policies that are currently enabled for the root and therefore can be attached to the
         *        root or to its OUs or accounts.</p> <note>
         *        <p>
         *        Even if a policy type is shown as available in the organization, you can separately enable and disable
         *        them at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         *        <a>DescribeOrganization</a> to see the availability of the policy types in that organization.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyTypes(PolicyTypeSummary... policyTypes);

        /**
         * <p>
         * The types of policies that are currently enabled for the root and therefore can be attached to the root or to
         * its OUs or accounts.
         * </p>
         * <note>
         * <p>
         * Even if a policy type is shown as available in the organization, you can separately enable and disable them
         * at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         * <a>DescribeOrganization</a> to see the availability of the policy types in that organization.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<PolicyTypeSummary>.Builder}
         * avoiding the need to create one manually via {@link List<PolicyTypeSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PolicyTypeSummary>.Builder#build()} is called immediately
         * and its result is passed to {@link #policyTypes(List<PolicyTypeSummary>)}.
         * 
         * @param policyTypes
         *        a consumer that will call methods on {@link List<PolicyTypeSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policyTypes(List<PolicyTypeSummary>)
         */
        Builder policyTypes(Consumer<PolicyTypeSummary.Builder>... policyTypes);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private String name;

        private List<PolicyTypeSummary> policyTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Root model) {
            id(model.id);
            arn(model.arn);
            name(model.name);
            policyTypes(model.policyTypes);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<PolicyTypeSummary.Builder> getPolicyTypes() {
            if (policyTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return policyTypes != null ? policyTypes.stream().map(PolicyTypeSummary::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder policyTypes(Collection<PolicyTypeSummary> policyTypes) {
            this.policyTypes = PolicyTypesCopier.copy(policyTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policyTypes(PolicyTypeSummary... policyTypes) {
            policyTypes(Arrays.asList(policyTypes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policyTypes(Consumer<PolicyTypeSummary.Builder>... policyTypes) {
            policyTypes(Stream.of(policyTypes).map(c -> PolicyTypeSummary.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPolicyTypes(Collection<PolicyTypeSummary.BuilderImpl> policyTypes) {
            this.policyTypes = PolicyTypesCopier.copyFromBuilder(policyTypes);
        }

        @Override
        public Root build() {
            return new Root(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
