/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an AWS account that is a member of an organization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Account implements SdkPojo, Serializable, ToCopyableBuilder<Account.Builder, Account> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Account::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Account::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> EMAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Account::email)).setter(setter(Builder::email))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Email").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Account::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Account::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> JOINED_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Account::joinedMethodAsString)).setter(setter(Builder::joinedMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JoinedMethod").build()).build();

    private static final SdkField<Instant> JOINED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Account::joinedTimestamp)).setter(setter(Builder::joinedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JoinedTimestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            EMAIL_FIELD, NAME_FIELD, STATUS_FIELD, JOINED_METHOD_FIELD, JOINED_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final String email;

    private final String name;

    private final String status;

    private final String joinedMethod;

    private final Instant joinedTimestamp;

    private Account(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.email = builder.email;
        this.name = builder.name;
        this.status = builder.status;
        this.joinedMethod = builder.joinedMethod;
        this.joinedTimestamp = builder.joinedTimestamp;
    }

    /**
     * <p>
     * The unique identifier (ID) of the account.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly 12
     * digits.
     * </p>
     * 
     * @return The unique identifier (ID) of the account.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly
     *         12 digits.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the account.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
     * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the account.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
     *         >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The email address associated with the AWS account.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for this parameter is a string of characters that
     * represents a standard internet email address.
     * </p>
     * 
     * @return The email address associated with the AWS account.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for this parameter is a string of
     *         characters that represents a standard internet email address.
     */
    public String email() {
        return email;
    }

    /**
     * <p>
     * The friendly name of the account.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter is a
     * string of any of the characters in the ASCII character range.
     * </p>
     * 
     * @return The friendly name of the account.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter
     *         is a string of any of the characters in the ASCII character range.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The status of the account in the organization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account in the organization.
     * @see AccountStatus
     */
    public AccountStatus status() {
        return AccountStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the account in the organization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account in the organization.
     * @see AccountStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The method by which the account joined the organization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #joinedMethod} will
     * return {@link AccountJoinedMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #joinedMethodAsString}.
     * </p>
     * 
     * @return The method by which the account joined the organization.
     * @see AccountJoinedMethod
     */
    public AccountJoinedMethod joinedMethod() {
        return AccountJoinedMethod.fromValue(joinedMethod);
    }

    /**
     * <p>
     * The method by which the account joined the organization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #joinedMethod} will
     * return {@link AccountJoinedMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #joinedMethodAsString}.
     * </p>
     * 
     * @return The method by which the account joined the organization.
     * @see AccountJoinedMethod
     */
    public String joinedMethodAsString() {
        return joinedMethod;
    }

    /**
     * <p>
     * The date the account became a part of the organization.
     * </p>
     * 
     * @return The date the account became a part of the organization.
     */
    public Instant joinedTimestamp() {
        return joinedTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(email());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(joinedMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(joinedTimestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Account)) {
            return false;
        }
        Account other = (Account) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn()) && Objects.equals(email(), other.email())
                && Objects.equals(name(), other.name()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(joinedMethodAsString(), other.joinedMethodAsString())
                && Objects.equals(joinedTimestamp(), other.joinedTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Account").add("Id", id()).add("Arn", arn())
                .add("Email", email() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***").add("Status", statusAsString())
                .add("JoinedMethod", joinedMethodAsString()).add("JoinedTimestamp", joinedTimestamp()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Email":
            return Optional.ofNullable(clazz.cast(email()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "JoinedMethod":
            return Optional.ofNullable(clazz.cast(joinedMethodAsString()));
        case "JoinedTimestamp":
            return Optional.ofNullable(clazz.cast(joinedTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Account, T> g) {
        return obj -> g.apply((Account) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Account> {
        /**
         * <p>
         * The unique identifier (ID) of the account.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly 12
         * digits.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) of the account.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires
         *        exactly 12 digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the account.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
         * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the account.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
         *        >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The email address associated with the AWS account.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for this parameter is a string of characters
         * that represents a standard internet email address.
         * </p>
         * 
         * @param email
         *        The email address associated with the AWS account.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for this parameter is a string of
         *        characters that represents a standard internet email address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder email(String email);

        /**
         * <p>
         * The friendly name of the account.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter is a
         * string of any of the characters in the ASCII character range.
         * </p>
         * 
         * @param name
         *        The friendly name of the account.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this
         *        parameter is a string of any of the characters in the ASCII character range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The status of the account in the organization.
         * </p>
         * 
         * @param status
         *        The status of the account in the organization.
         * @see AccountStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the account in the organization.
         * </p>
         * 
         * @param status
         *        The status of the account in the organization.
         * @see AccountStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountStatus
         */
        Builder status(AccountStatus status);

        /**
         * <p>
         * The method by which the account joined the organization.
         * </p>
         * 
         * @param joinedMethod
         *        The method by which the account joined the organization.
         * @see AccountJoinedMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountJoinedMethod
         */
        Builder joinedMethod(String joinedMethod);

        /**
         * <p>
         * The method by which the account joined the organization.
         * </p>
         * 
         * @param joinedMethod
         *        The method by which the account joined the organization.
         * @see AccountJoinedMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountJoinedMethod
         */
        Builder joinedMethod(AccountJoinedMethod joinedMethod);

        /**
         * <p>
         * The date the account became a part of the organization.
         * </p>
         * 
         * @param joinedTimestamp
         *        The date the account became a part of the organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder joinedTimestamp(Instant joinedTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private String email;

        private String name;

        private String status;

        private String joinedMethod;

        private Instant joinedTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(Account model) {
            id(model.id);
            arn(model.arn);
            email(model.email);
            name(model.name);
            status(model.status);
            joinedMethod(model.joinedMethod);
            joinedTimestamp(model.joinedTimestamp);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getEmail() {
            return email;
        }

        @Override
        public final Builder email(String email) {
            this.email = email;
            return this;
        }

        public final void setEmail(String email) {
            this.email = email;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AccountStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getJoinedMethod() {
            return joinedMethod;
        }

        @Override
        public final Builder joinedMethod(String joinedMethod) {
            this.joinedMethod = joinedMethod;
            return this;
        }

        @Override
        public final Builder joinedMethod(AccountJoinedMethod joinedMethod) {
            this.joinedMethod(joinedMethod == null ? null : joinedMethod.toString());
            return this;
        }

        public final void setJoinedMethod(String joinedMethod) {
            this.joinedMethod = joinedMethod;
        }

        public final Instant getJoinedTimestamp() {
            return joinedTimestamp;
        }

        @Override
        public final Builder joinedTimestamp(Instant joinedTimestamp) {
            this.joinedTimestamp = joinedTimestamp;
            return this;
        }

        public final void setJoinedTimestamp(Instant joinedTimestamp) {
            this.joinedTimestamp = joinedTimestamp;
        }

        @Override
        public Account build() {
            return new Account(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
