/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about an organization. An organization is a collection of accounts that are centrally managed
 * together using consolidated billing, organized hierarchically with organizational units (OUs), and controlled with
 * policies.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Organization implements SdkPojo, Serializable, ToCopyableBuilder<Organization.Builder, Organization> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Organization::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Organization::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> FEATURE_SET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Organization::featureSetAsString)).setter(setter(Builder::featureSet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureSet").build()).build();

    private static final SdkField<String> MASTER_ACCOUNT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Organization::masterAccountArn)).setter(setter(Builder::masterAccountArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterAccountArn").build()).build();

    private static final SdkField<String> MASTER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Organization::masterAccountId)).setter(setter(Builder::masterAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterAccountId").build()).build();

    private static final SdkField<String> MASTER_ACCOUNT_EMAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Organization::masterAccountEmail)).setter(setter(Builder::masterAccountEmail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterAccountEmail").build())
            .build();

    private static final SdkField<List<PolicyTypeSummary>> AVAILABLE_POLICY_TYPES_FIELD = SdkField
            .<List<PolicyTypeSummary>> builder(MarshallingType.LIST)
            .getter(getter(Organization::availablePolicyTypes))
            .setter(setter(Builder::availablePolicyTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailablePolicyTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PolicyTypeSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(PolicyTypeSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            FEATURE_SET_FIELD, MASTER_ACCOUNT_ARN_FIELD, MASTER_ACCOUNT_ID_FIELD, MASTER_ACCOUNT_EMAIL_FIELD,
            AVAILABLE_POLICY_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final String featureSet;

    private final String masterAccountArn;

    private final String masterAccountId;

    private final String masterAccountEmail;

    private final List<PolicyTypeSummary> availablePolicyTypes;

    private Organization(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.featureSet = builder.featureSet;
        this.masterAccountArn = builder.masterAccountArn;
        this.masterAccountId = builder.masterAccountId;
        this.masterAccountEmail = builder.masterAccountEmail;
        this.availablePolicyTypes = builder.availablePolicyTypes;
    }

    /**
     * <p>
     * The unique identifier (ID) of an organization.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an organization ID string requires "o-"
     * followed by from 10 to 32 lower-case letters or digits.
     * </p>
     * 
     * @return The unique identifier (ID) of an organization.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an organization ID string requires
     *         "o-" followed by from 10 to 32 lower-case letters or digits.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an organization.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
     * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an organization.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
     *         >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * Specifies the functionality that currently is available to the organization. If set to "ALL", then all features
     * are enabled and policies can be applied to accounts in the organization. If set to "CONSOLIDATED_BILLING", then
     * only consolidated billing functionality is available. For more information, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
     * >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureSet} will
     * return {@link OrganizationFeatureSet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureSetAsString}.
     * </p>
     * 
     * @return Specifies the functionality that currently is available to the organization. If set to "ALL", then all
     *         features are enabled and policies can be applied to accounts in the organization. If set to
     *         "CONSOLIDATED_BILLING", then only consolidated billing functionality is available. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
     *         >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
     * @see OrganizationFeatureSet
     */
    public OrganizationFeatureSet featureSet() {
        return OrganizationFeatureSet.fromValue(featureSet);
    }

    /**
     * <p>
     * Specifies the functionality that currently is available to the organization. If set to "ALL", then all features
     * are enabled and policies can be applied to accounts in the organization. If set to "CONSOLIDATED_BILLING", then
     * only consolidated billing functionality is available. For more information, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
     * >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureSet} will
     * return {@link OrganizationFeatureSet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureSetAsString}.
     * </p>
     * 
     * @return Specifies the functionality that currently is available to the organization. If set to "ALL", then all
     *         features are enabled and policies can be applied to accounts in the organization. If set to
     *         "CONSOLIDATED_BILLING", then only consolidated billing functionality is available. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
     *         >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
     * @see OrganizationFeatureSet
     */
    public String featureSetAsString() {
        return featureSet;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the account that is designated as the master account for the organization.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
     * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the account that is designated as the master account for the
     *         organization.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
     *         >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     */
    public String masterAccountArn() {
        return masterAccountArn;
    }

    /**
     * <p>
     * The unique identifier (ID) of the master account of an organization.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly 12
     * digits.
     * </p>
     * 
     * @return The unique identifier (ID) of the master account of an organization.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly
     *         12 digits.
     */
    public String masterAccountId() {
        return masterAccountId;
    }

    /**
     * <p>
     * The email address that is associated with the AWS account that is designated as the master account for the
     * organization.
     * </p>
     * 
     * @return The email address that is associated with the AWS account that is designated as the master account for
     *         the organization.
     */
    public String masterAccountEmail() {
        return masterAccountEmail;
    }

    /**
     * Returns true if the AvailablePolicyTypes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAvailablePolicyTypes() {
        return availablePolicyTypes != null && !(availablePolicyTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of policy types that are enabled for this organization. For example, if your organization has all features
     * enabled, then service control policies (SCPs) are included in the list.
     * </p>
     * <note>
     * <p>
     * Even if a policy type is shown as available in the organization, you can separately enable and disable them at
     * the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use <a>ListRoots</a> to see the
     * status of a policy type in that root.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAvailablePolicyTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of policy types that are enabled for this organization. For example, if your organization has all
     *         features enabled, then service control policies (SCPs) are included in the list.</p> <note>
     *         <p>
     *         Even if a policy type is shown as available in the organization, you can separately enable and disable
     *         them at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
     *         <a>ListRoots</a> to see the status of a policy type in that root.
     *         </p>
     */
    public List<PolicyTypeSummary> availablePolicyTypes() {
        return availablePolicyTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(featureSetAsString());
        hashCode = 31 * hashCode + Objects.hashCode(masterAccountArn());
        hashCode = 31 * hashCode + Objects.hashCode(masterAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(masterAccountEmail());
        hashCode = 31 * hashCode + Objects.hashCode(availablePolicyTypes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Organization)) {
            return false;
        }
        Organization other = (Organization) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(featureSetAsString(), other.featureSetAsString())
                && Objects.equals(masterAccountArn(), other.masterAccountArn())
                && Objects.equals(masterAccountId(), other.masterAccountId())
                && Objects.equals(masterAccountEmail(), other.masterAccountEmail())
                && Objects.equals(availablePolicyTypes(), other.availablePolicyTypes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Organization").add("Id", id()).add("Arn", arn()).add("FeatureSet", featureSetAsString())
                .add("MasterAccountArn", masterAccountArn()).add("MasterAccountId", masterAccountId())
                .add("MasterAccountEmail", masterAccountEmail() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AvailablePolicyTypes", availablePolicyTypes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "FeatureSet":
            return Optional.ofNullable(clazz.cast(featureSetAsString()));
        case "MasterAccountArn":
            return Optional.ofNullable(clazz.cast(masterAccountArn()));
        case "MasterAccountId":
            return Optional.ofNullable(clazz.cast(masterAccountId()));
        case "MasterAccountEmail":
            return Optional.ofNullable(clazz.cast(masterAccountEmail()));
        case "AvailablePolicyTypes":
            return Optional.ofNullable(clazz.cast(availablePolicyTypes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Organization, T> g) {
        return obj -> g.apply((Organization) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Organization> {
        /**
         * <p>
         * The unique identifier (ID) of an organization.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an organization ID string requires "o-"
         * followed by from 10 to 32 lower-case letters or digits.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) of an organization.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an organization ID string requires
         *        "o-" followed by from 10 to 32 lower-case letters or digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an organization.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
         * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of an organization.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
         *        >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Specifies the functionality that currently is available to the organization. If set to "ALL", then all
         * features are enabled and policies can be applied to accounts in the organization. If set to
         * "CONSOLIDATED_BILLING", then only consolidated billing functionality is available. For more information, see
         * <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
         * >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
         * </p>
         * 
         * @param featureSet
         *        Specifies the functionality that currently is available to the organization. If set to "ALL", then all
         *        features are enabled and policies can be applied to accounts in the organization. If set to
         *        "CONSOLIDATED_BILLING", then only consolidated billing functionality is available. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
         *        >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
         * @see OrganizationFeatureSet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrganizationFeatureSet
         */
        Builder featureSet(String featureSet);

        /**
         * <p>
         * Specifies the functionality that currently is available to the organization. If set to "ALL", then all
         * features are enabled and policies can be applied to accounts in the organization. If set to
         * "CONSOLIDATED_BILLING", then only consolidated billing functionality is available. For more information, see
         * <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
         * >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
         * </p>
         * 
         * @param featureSet
         *        Specifies the functionality that currently is available to the organization. If set to "ALL", then all
         *        features are enabled and policies can be applied to accounts in the organization. If set to
         *        "CONSOLIDATED_BILLING", then only consolidated billing functionality is available. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html"
         *        >Enabling All Features in Your Organization</a> in the <i>AWS Organizations User Guide</i>.
         * @see OrganizationFeatureSet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OrganizationFeatureSet
         */
        Builder featureSet(OrganizationFeatureSet featureSet);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the account that is designated as the master account for the organization.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
         * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * </p>
         * 
         * @param masterAccountArn
         *        The Amazon Resource Name (ARN) of the account that is designated as the master account for the
         *        organization.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
         *        >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterAccountArn(String masterAccountArn);

        /**
         * <p>
         * The unique identifier (ID) of the master account of an organization.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires exactly 12
         * digits.
         * </p>
         * 
         * @param masterAccountId
         *        The unique identifier (ID) of the master account of an organization.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for an account ID string requires
         *        exactly 12 digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterAccountId(String masterAccountId);

        /**
         * <p>
         * The email address that is associated with the AWS account that is designated as the master account for the
         * organization.
         * </p>
         * 
         * @param masterAccountEmail
         *        The email address that is associated with the AWS account that is designated as the master account for
         *        the organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterAccountEmail(String masterAccountEmail);

        /**
         * <p>
         * A list of policy types that are enabled for this organization. For example, if your organization has all
         * features enabled, then service control policies (SCPs) are included in the list.
         * </p>
         * <note>
         * <p>
         * Even if a policy type is shown as available in the organization, you can separately enable and disable them
         * at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use <a>ListRoots</a> to see
         * the status of a policy type in that root.
         * </p>
         * </note>
         * 
         * @param availablePolicyTypes
         *        A list of policy types that are enabled for this organization. For example, if your organization has
         *        all features enabled, then service control policies (SCPs) are included in the list.</p> <note>
         *        <p>
         *        Even if a policy type is shown as available in the organization, you can separately enable and disable
         *        them at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         *        <a>ListRoots</a> to see the status of a policy type in that root.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availablePolicyTypes(Collection<PolicyTypeSummary> availablePolicyTypes);

        /**
         * <p>
         * A list of policy types that are enabled for this organization. For example, if your organization has all
         * features enabled, then service control policies (SCPs) are included in the list.
         * </p>
         * <note>
         * <p>
         * Even if a policy type is shown as available in the organization, you can separately enable and disable them
         * at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use <a>ListRoots</a> to see
         * the status of a policy type in that root.
         * </p>
         * </note>
         * 
         * @param availablePolicyTypes
         *        A list of policy types that are enabled for this organization. For example, if your organization has
         *        all features enabled, then service control policies (SCPs) are included in the list.</p> <note>
         *        <p>
         *        Even if a policy type is shown as available in the organization, you can separately enable and disable
         *        them at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use
         *        <a>ListRoots</a> to see the status of a policy type in that root.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availablePolicyTypes(PolicyTypeSummary... availablePolicyTypes);

        /**
         * <p>
         * A list of policy types that are enabled for this organization. For example, if your organization has all
         * features enabled, then service control policies (SCPs) are included in the list.
         * </p>
         * <note>
         * <p>
         * Even if a policy type is shown as available in the organization, you can separately enable and disable them
         * at the root level by using <a>EnablePolicyType</a> and <a>DisablePolicyType</a>. Use <a>ListRoots</a> to see
         * the status of a policy type in that root.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<PolicyTypeSummary>.Builder}
         * avoiding the need to create one manually via {@link List<PolicyTypeSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PolicyTypeSummary>.Builder#build()} is called immediately
         * and its result is passed to {@link #availablePolicyTypes(List<PolicyTypeSummary>)}.
         * 
         * @param availablePolicyTypes
         *        a consumer that will call methods on {@link List<PolicyTypeSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availablePolicyTypes(List<PolicyTypeSummary>)
         */
        Builder availablePolicyTypes(Consumer<PolicyTypeSummary.Builder>... availablePolicyTypes);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private String featureSet;

        private String masterAccountArn;

        private String masterAccountId;

        private String masterAccountEmail;

        private List<PolicyTypeSummary> availablePolicyTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Organization model) {
            id(model.id);
            arn(model.arn);
            featureSet(model.featureSet);
            masterAccountArn(model.masterAccountArn);
            masterAccountId(model.masterAccountId);
            masterAccountEmail(model.masterAccountEmail);
            availablePolicyTypes(model.availablePolicyTypes);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getFeatureSetAsString() {
            return featureSet;
        }

        @Override
        public final Builder featureSet(String featureSet) {
            this.featureSet = featureSet;
            return this;
        }

        @Override
        public final Builder featureSet(OrganizationFeatureSet featureSet) {
            this.featureSet(featureSet == null ? null : featureSet.toString());
            return this;
        }

        public final void setFeatureSet(String featureSet) {
            this.featureSet = featureSet;
        }

        public final String getMasterAccountArn() {
            return masterAccountArn;
        }

        @Override
        public final Builder masterAccountArn(String masterAccountArn) {
            this.masterAccountArn = masterAccountArn;
            return this;
        }

        public final void setMasterAccountArn(String masterAccountArn) {
            this.masterAccountArn = masterAccountArn;
        }

        public final String getMasterAccountId() {
            return masterAccountId;
        }

        @Override
        public final Builder masterAccountId(String masterAccountId) {
            this.masterAccountId = masterAccountId;
            return this;
        }

        public final void setMasterAccountId(String masterAccountId) {
            this.masterAccountId = masterAccountId;
        }

        public final String getMasterAccountEmail() {
            return masterAccountEmail;
        }

        @Override
        public final Builder masterAccountEmail(String masterAccountEmail) {
            this.masterAccountEmail = masterAccountEmail;
            return this;
        }

        public final void setMasterAccountEmail(String masterAccountEmail) {
            this.masterAccountEmail = masterAccountEmail;
        }

        public final Collection<PolicyTypeSummary.Builder> getAvailablePolicyTypes() {
            return availablePolicyTypes != null ? availablePolicyTypes.stream().map(PolicyTypeSummary::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder availablePolicyTypes(Collection<PolicyTypeSummary> availablePolicyTypes) {
            this.availablePolicyTypes = PolicyTypesCopier.copy(availablePolicyTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availablePolicyTypes(PolicyTypeSummary... availablePolicyTypes) {
            availablePolicyTypes(Arrays.asList(availablePolicyTypes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availablePolicyTypes(Consumer<PolicyTypeSummary.Builder>... availablePolicyTypes) {
            availablePolicyTypes(Stream.of(availablePolicyTypes).map(c -> PolicyTypeSummary.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAvailablePolicyTypes(Collection<PolicyTypeSummary.BuilderImpl> availablePolicyTypes) {
            this.availablePolicyTypes = PolicyTypesCopier.copyFromBuilder(availablePolicyTypes);
        }

        @Override
        public Organization build() {
            return new Organization(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
