/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information that must be exchanged to securely establish a relationship between two accounts (an
 * <i>originator</i> and a <i>recipient</i>). For example, assume that a master account (the originator) invites another
 * account (the recipient) to join its organization. In that case, the two accounts exchange information as a series of
 * handshake requests and responses.
 * </p>
 * <p>
 * <b>Note:</b> Handshakes that are CANCELED, ACCEPTED, or DECLINED show up in lists for only 30 days after entering
 * that state. After that, they are deleted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Handshake implements SdkPojo, Serializable, ToCopyableBuilder<Handshake.Builder, Handshake> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Handshake::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Handshake::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<List<HandshakeParty>> PARTIES_FIELD = SdkField
            .<List<HandshakeParty>> builder(MarshallingType.LIST)
            .getter(getter(Handshake::parties))
            .setter(setter(Builder::parties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HandshakeParty> builder(MarshallingType.SDK_POJO)
                                            .constructor(HandshakeParty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Handshake::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Instant> REQUESTED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Handshake::requestedTimestamp)).setter(setter(Builder::requestedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedTimestamp").build())
            .build();

    private static final SdkField<Instant> EXPIRATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Handshake::expirationTimestamp)).setter(setter(Builder::expirationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationTimestamp").build())
            .build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Handshake::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<List<HandshakeResource>> RESOURCES_FIELD = SdkField
            .<List<HandshakeResource>> builder(MarshallingType.LIST)
            .getter(getter(Handshake::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HandshakeResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(HandshakeResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            PARTIES_FIELD, STATE_FIELD, REQUESTED_TIMESTAMP_FIELD, EXPIRATION_TIMESTAMP_FIELD, ACTION_FIELD, RESOURCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final List<HandshakeParty> parties;

    private final String state;

    private final Instant requestedTimestamp;

    private final Instant expirationTimestamp;

    private final String action;

    private final List<HandshakeResource> resources;

    private Handshake(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.parties = builder.parties;
        this.state = builder.state;
        this.requestedTimestamp = builder.requestedTimestamp;
        this.expirationTimestamp = builder.expirationTimestamp;
        this.action = builder.action;
        this.resources = builder.resources;
    }

    /**
     * <p>
     * The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates the
     * handshake.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-" followed by
     * from 8 to 32 lower-case letters or digits.
     * </p>
     * 
     * @return The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates the
     *         handshake.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-"
     *         followed by from 8 to 32 lower-case letters or digits.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of a handshake.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
     * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of a handshake.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
     *         >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
     */
    public String arn() {
        return arn;
    }

    /**
     * Returns true if the Parties property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParties() {
        return parties != null && !(parties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the two accounts that are participating in the handshake.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParties()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Information about the two accounts that are participating in the handshake.
     */
    public List<HandshakeParty> parties() {
        return parties;
    }

    /**
     * <p>
     * The current state of the handshake. Use the state to trace the flow of the handshake through the process from its
     * creation to its acceptance. The meaning of each of the valid values is as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types) and
     * not all recipients have responded yet. The request stays in this state until all recipients respond.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
     * recipients have responded, allowing the originator to complete the handshake action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any kind
     * from the recipient before the expiration time (15 days).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link HandshakeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the handshake. Use the state to trace the flow of the handshake through the process
     *         from its creation to its acceptance. The meaning of each of the valid values is as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
     *         types) and not all recipients have responded yet. The request stays in this state until all recipients
     *         respond.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
     *         all recipients have responded, allowing the originator to complete the handshake action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of
     *         any kind from the recipient before the expiration time (15 days).
     *         </p>
     *         </li>
     * @see HandshakeState
     */
    public HandshakeState state() {
        return HandshakeState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the handshake. Use the state to trace the flow of the handshake through the process from its
     * creation to its acceptance. The meaning of each of the valid values is as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types) and
     * not all recipients have responded yet. The request stays in this state until all recipients respond.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
     * recipients have responded, allowing the originator to complete the handshake action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any kind
     * from the recipient before the expiration time (15 days).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link HandshakeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the handshake. Use the state to trace the flow of the handshake through the process
     *         from its creation to its acceptance. The meaning of each of the valid values is as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
     *         types) and not all recipients have responded yet. The request stays in this state until all recipients
     *         respond.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
     *         all recipients have responded, allowing the originator to complete the handshake action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of
     *         any kind from the recipient before the expiration time (15 days).
     *         </p>
     *         </li>
     * @see HandshakeState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The date and time that the handshake request was made.
     * </p>
     * 
     * @return The date and time that the handshake request was made.
     */
    public Instant requestedTimestamp() {
        return requestedTimestamp;
    }

    /**
     * <p>
     * The date and time that the handshake expires. If the recipient of the handshake request fails to respond before
     * the specified date and time, the handshake becomes inactive and is no longer valid.
     * </p>
     * 
     * @return The date and time that the handshake expires. If the recipient of the handshake request fails to respond
     *         before the specified date and time, the handshake becomes inactive and is no longer valid.
     */
    public Instant expirationTimestamp() {
        return expirationTimestamp;
    }

    /**
     * <p>
     * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
     * handshake types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from the
     * master account to only non-member accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     * organization. It is always sent from the master account to only <i>invited</i> member accounts. Created accounts
     * do not receive this because those accounts were created by the organization's master account and approval is
     * inferred.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
     * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the master account and
     * signals the master that it can finalize the process to enable all features.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link ActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
     *         following handshake types are supported:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
     *         from the master account to only non-member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     *         organization. It is always sent from the master account to only <i>invited</i> member accounts. Created
     *         accounts do not receive this because those accounts were created by the organization's master account and
     *         approval is inferred.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
     *         member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
     *         master account and signals the master that it can finalize the process to enable all features.
     *         </p>
     *         </li>
     * @see ActionType
     */
    public ActionType action() {
        return ActionType.fromValue(action);
    }

    /**
     * <p>
     * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
     * handshake types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from the
     * master account to only non-member accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     * organization. It is always sent from the master account to only <i>invited</i> member accounts. Created accounts
     * do not receive this because those accounts were created by the organization's master account and approval is
     * inferred.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
     * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the master account and
     * signals the master that it can finalize the process to enable all features.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link ActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
     *         following handshake types are supported:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
     *         from the master account to only non-member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     *         organization. It is always sent from the master account to only <i>invited</i> member accounts. Created
     *         accounts do not receive this because those accounts were created by the organization's master account and
     *         approval is inferred.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
     *         member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
     *         master account and signals the master that it can finalize the process to enable all features.
     *         </p>
     *         </li>
     * @see ActionType
     */
    public String actionAsString() {
        return action;
    }

    /**
     * Returns true if the Resources property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional information that is needed to process the handshake.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Additional information that is needed to process the handshake.
     */
    public List<HandshakeResource> resources() {
        return resources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(parties());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requestedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(expirationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Handshake)) {
            return false;
        }
        Handshake other = (Handshake) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(parties(), other.parties()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(requestedTimestamp(), other.requestedTimestamp())
                && Objects.equals(expirationTimestamp(), other.expirationTimestamp())
                && Objects.equals(actionAsString(), other.actionAsString()) && Objects.equals(resources(), other.resources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Handshake").add("Id", id()).add("Arn", arn()).add("Parties", parties())
                .add("State", stateAsString()).add("RequestedTimestamp", requestedTimestamp())
                .add("ExpirationTimestamp", expirationTimestamp()).add("Action", actionAsString()).add("Resources", resources())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Parties":
            return Optional.ofNullable(clazz.cast(parties()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "RequestedTimestamp":
            return Optional.ofNullable(clazz.cast(requestedTimestamp()));
        case "ExpirationTimestamp":
            return Optional.ofNullable(clazz.cast(expirationTimestamp()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Handshake, T> g) {
        return obj -> g.apply((Handshake) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Handshake> {
        /**
         * <p>
         * The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates the
         * handshake.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-"
         * followed by from 8 to 32 lower-case letters or digits.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates
         *        the handshake.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-"
         *        followed by from 8 to 32 lower-case letters or digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of a handshake.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns">ARN
         * Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of a handshake.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions.html#orgs-permissions-arns"
         *        >ARN Formats Supported by Organizations</a> in the <i>AWS Organizations User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Information about the two accounts that are participating in the handshake.
         * </p>
         * 
         * @param parties
         *        Information about the two accounts that are participating in the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parties(Collection<HandshakeParty> parties);

        /**
         * <p>
         * Information about the two accounts that are participating in the handshake.
         * </p>
         * 
         * @param parties
         *        Information about the two accounts that are participating in the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parties(HandshakeParty... parties);

        /**
         * <p>
         * Information about the two accounts that are participating in the handshake.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HandshakeParty>.Builder} avoiding the need
         * to create one manually via {@link List<HandshakeParty>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HandshakeParty>.Builder#build()} is called immediately and
         * its result is passed to {@link #parties(List<HandshakeParty>)}.
         * 
         * @param parties
         *        a consumer that will call methods on {@link List<HandshakeParty>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parties(List<HandshakeParty>)
         */
        Builder parties(Consumer<HandshakeParty.Builder>... parties);

        /**
         * <p>
         * The current state of the handshake. Use the state to trace the flow of the handshake through the process from
         * its creation to its acceptance. The meaning of each of the valid values is as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types)
         * and not all recipients have responded yet. The request stays in this state until all recipients respond.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
         * recipients have responded, allowing the originator to complete the handshake action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any
         * kind from the recipient before the expiration time (15 days).
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the handshake. Use the state to trace the flow of the handshake through the
         *        process from its creation to its acceptance. The meaning of each of the valid values is as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
         *        types) and not all recipients have responded yet. The request stays in this state until all recipients
         *        respond.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
         *        all recipients have responded, allowing the originator to complete the handshake action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating
         *        account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response
         *        of any kind from the recipient before the expiration time (15 days).
         *        </p>
         *        </li>
         * @see HandshakeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HandshakeState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the handshake. Use the state to trace the flow of the handshake through the process from
         * its creation to its acceptance. The meaning of each of the valid values is as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types)
         * and not all recipients have responded yet. The request stays in this state until all recipients respond.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
         * recipients have responded, allowing the originator to complete the handshake action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any
         * kind from the recipient before the expiration time (15 days).
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the handshake. Use the state to trace the flow of the handshake through the
         *        process from its creation to its acceptance. The meaning of each of the valid values is as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
         *        types) and not all recipients have responded yet. The request stays in this state until all recipients
         *        respond.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
         *        all recipients have responded, allowing the originator to complete the handshake action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating
         *        account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response
         *        of any kind from the recipient before the expiration time (15 days).
         *        </p>
         *        </li>
         * @see HandshakeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HandshakeState
         */
        Builder state(HandshakeState state);

        /**
         * <p>
         * The date and time that the handshake request was made.
         * </p>
         * 
         * @param requestedTimestamp
         *        The date and time that the handshake request was made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedTimestamp(Instant requestedTimestamp);

        /**
         * <p>
         * The date and time that the handshake expires. If the recipient of the handshake request fails to respond
         * before the specified date and time, the handshake becomes inactive and is no longer valid.
         * </p>
         * 
         * @param expirationTimestamp
         *        The date and time that the handshake expires. If the recipient of the handshake request fails to
         *        respond before the specified date and time, the handshake becomes inactive and is no longer valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationTimestamp(Instant expirationTimestamp);

        /**
         * <p>
         * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
         * handshake types are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from
         * the master account to only non-member accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         * organization. It is always sent from the master account to only <i>invited</i> member accounts. Created
         * accounts do not receive this because those accounts were created by the organization's master account and
         * approval is inferred.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
         * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the master account
         * and signals the master that it can finalize the process to enable all features.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
         *        following handshake types are supported:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
         *        from the master account to only non-member accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         *        organization. It is always sent from the master account to only <i>invited</i> member accounts.
         *        Created accounts do not receive this because those accounts were created by the organization's master
         *        account and approval is inferred.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
         *        member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
         *        master account and signals the master that it can finalize the process to enable all features.
         *        </p>
         *        </li>
         * @see ActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionType
         */
        Builder action(String action);

        /**
         * <p>
         * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
         * handshake types are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from
         * the master account to only non-member accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         * organization. It is always sent from the master account to only <i>invited</i> member accounts. Created
         * accounts do not receive this because those accounts were created by the organization's master account and
         * approval is inferred.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
         * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the master account
         * and signals the master that it can finalize the process to enable all features.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
         *        following handshake types are supported:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
         *        from the master account to only non-member accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         *        organization. It is always sent from the master account to only <i>invited</i> member accounts.
         *        Created accounts do not receive this because those accounts were created by the organization's master
         *        account and approval is inferred.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
         *        member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
         *        master account and signals the master that it can finalize the process to enable all features.
         *        </p>
         *        </li>
         * @see ActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionType
         */
        Builder action(ActionType action);

        /**
         * <p>
         * Additional information that is needed to process the handshake.
         * </p>
         * 
         * @param resources
         *        Additional information that is needed to process the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<HandshakeResource> resources);

        /**
         * <p>
         * Additional information that is needed to process the handshake.
         * </p>
         * 
         * @param resources
         *        Additional information that is needed to process the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(HandshakeResource... resources);

        /**
         * <p>
         * Additional information that is needed to process the handshake.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HandshakeResource>.Builder} avoiding the
         * need to create one manually via {@link List<HandshakeResource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HandshakeResource>.Builder#build()} is called immediately
         * and its result is passed to {@link #resources(List<HandshakeResource>)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link List<HandshakeResource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(List<HandshakeResource>)
         */
        Builder resources(Consumer<HandshakeResource.Builder>... resources);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private List<HandshakeParty> parties = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private Instant requestedTimestamp;

        private Instant expirationTimestamp;

        private String action;

        private List<HandshakeResource> resources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Handshake model) {
            id(model.id);
            arn(model.arn);
            parties(model.parties);
            state(model.state);
            requestedTimestamp(model.requestedTimestamp);
            expirationTimestamp(model.expirationTimestamp);
            action(model.action);
            resources(model.resources);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Collection<HandshakeParty.Builder> getParties() {
            return parties != null ? parties.stream().map(HandshakeParty::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parties(Collection<HandshakeParty> parties) {
            this.parties = HandshakePartiesCopier.copy(parties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parties(HandshakeParty... parties) {
            parties(Arrays.asList(parties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parties(Consumer<HandshakeParty.Builder>... parties) {
            parties(Stream.of(parties).map(c -> HandshakeParty.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParties(Collection<HandshakeParty.BuilderImpl> parties) {
            this.parties = HandshakePartiesCopier.copyFromBuilder(parties);
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(HandshakeState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final Instant getRequestedTimestamp() {
            return requestedTimestamp;
        }

        @Override
        public final Builder requestedTimestamp(Instant requestedTimestamp) {
            this.requestedTimestamp = requestedTimestamp;
            return this;
        }

        public final void setRequestedTimestamp(Instant requestedTimestamp) {
            this.requestedTimestamp = requestedTimestamp;
        }

        public final Instant getExpirationTimestamp() {
            return expirationTimestamp;
        }

        @Override
        public final Builder expirationTimestamp(Instant expirationTimestamp) {
            this.expirationTimestamp = expirationTimestamp;
            return this;
        }

        public final void setExpirationTimestamp(Instant expirationTimestamp) {
            this.expirationTimestamp = expirationTimestamp;
        }

        public final String getActionAsString() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(ActionType action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final Collection<HandshakeResource.Builder> getResources() {
            return resources != null ? resources.stream().map(HandshakeResource::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resources(Collection<HandshakeResource> resources) {
            this.resources = HandshakeResourcesCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(HandshakeResource... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Consumer<HandshakeResource.Builder>... resources) {
            resources(Stream.of(resources).map(c -> HandshakeResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setResources(Collection<HandshakeResource.BuilderImpl> resources) {
            this.resources = HandshakeResourcesCopier.copyFromBuilder(resources);
        }

        @Override
        public Handshake build() {
            return new Handshake(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
