/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.observabilityadmin.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines how telemetry should be configured for specific Amazon Web Services resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TelemetryRule implements SdkPojo, Serializable, ToCopyableBuilder<TelemetryRule.Builder, TelemetryRule> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(TelemetryRule::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> TELEMETRY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TelemetryType").getter(getter(TelemetryRule::telemetryTypeAsString))
            .setter(setter(Builder::telemetryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TelemetryType").build()).build();

    private static final SdkField<List<String>> TELEMETRY_SOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TelemetrySourceTypes")
            .getter(getter(TelemetryRule::telemetrySourceTypesAsStrings))
            .setter(setter(Builder::telemetrySourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TelemetrySourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TelemetryDestinationConfiguration> DESTINATION_CONFIGURATION_FIELD = SdkField
            .<TelemetryDestinationConfiguration> builder(MarshallingType.SDK_POJO).memberName("DestinationConfiguration")
            .getter(getter(TelemetryRule::destinationConfiguration)).setter(setter(Builder::destinationConfiguration))
            .constructor(TelemetryDestinationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationConfiguration").build())
            .build();

    private static final SdkField<String> SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Scope")
            .getter(getter(TelemetryRule::scope)).setter(setter(Builder::scope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scope").build()).build();

    private static final SdkField<String> SELECTION_CRITERIA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SelectionCriteria").getter(getter(TelemetryRule::selectionCriteria))
            .setter(setter(Builder::selectionCriteria))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectionCriteria").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            TELEMETRY_TYPE_FIELD, TELEMETRY_SOURCE_TYPES_FIELD, DESTINATION_CONFIGURATION_FIELD, SCOPE_FIELD,
            SELECTION_CRITERIA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final String telemetryType;

    private final List<String> telemetrySourceTypes;

    private final TelemetryDestinationConfiguration destinationConfiguration;

    private final String scope;

    private final String selectionCriteria;

    private TelemetryRule(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.telemetryType = builder.telemetryType;
        this.telemetrySourceTypes = builder.telemetrySourceTypes;
        this.destinationConfiguration = builder.destinationConfiguration;
        this.scope = builder.scope;
        this.selectionCriteria = builder.selectionCriteria;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC", "AWS::EKS::Cluster",
     * "AWS::WAFv2::WebACL").
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC",
     *         "AWS::EKS::Cluster", "AWS::WAFv2::WebACL").
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC", "AWS::EKS::Cluster",
     * "AWS::WAFv2::WebACL").
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC",
     *         "AWS::EKS::Cluster", "AWS::WAFv2::WebACL").
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The type of telemetry to collect (Logs, Metrics, or Traces).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #telemetryType}
     * will return {@link TelemetryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #telemetryTypeAsString}.
     * </p>
     * 
     * @return The type of telemetry to collect (Logs, Metrics, or Traces).
     * @see TelemetryType
     */
    public final TelemetryType telemetryType() {
        return TelemetryType.fromValue(telemetryType);
    }

    /**
     * <p>
     * The type of telemetry to collect (Logs, Metrics, or Traces).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #telemetryType}
     * will return {@link TelemetryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #telemetryTypeAsString}.
     * </p>
     * 
     * @return The type of telemetry to collect (Logs, Metrics, or Traces).
     * @see TelemetryType
     */
    public final String telemetryTypeAsString() {
        return telemetryType;
    }

    /**
     * <p>
     * The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or EKS_AUDIT_LOGS.
     * TelemetrySourceTypes must be correlated with the specific resource type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetrySourceTypes} method.
     * </p>
     * 
     * @return The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or
     *         EKS_AUDIT_LOGS. TelemetrySourceTypes must be correlated with the specific resource type.
     */
    public final List<TelemetrySourceType> telemetrySourceTypes() {
        return TelemetrySourceTypesCopier.copyStringToEnum(telemetrySourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the TelemetrySourceTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTelemetrySourceTypes() {
        return telemetrySourceTypes != null && !(telemetrySourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or EKS_AUDIT_LOGS.
     * TelemetrySourceTypes must be correlated with the specific resource type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetrySourceTypes} method.
     * </p>
     * 
     * @return The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or
     *         EKS_AUDIT_LOGS. TelemetrySourceTypes must be correlated with the specific resource type.
     */
    public final List<String> telemetrySourceTypesAsStrings() {
        return telemetrySourceTypes;
    }

    /**
     * <p>
     * Configuration specifying where and how the telemetry data should be delivered.
     * </p>
     * 
     * @return Configuration specifying where and how the telemetry data should be delivered.
     */
    public final TelemetryDestinationConfiguration destinationConfiguration() {
        return destinationConfiguration;
    }

    /**
     * <p>
     * The organizational scope to which the rule applies, specified using accounts or organizational units.
     * </p>
     * 
     * @return The organizational scope to which the rule applies, specified using accounts or organizational units.
     */
    public final String scope() {
        return scope;
    }

    /**
     * <p>
     * Criteria for selecting which resources the rule applies to, such as resource tags.
     * </p>
     * 
     * @return Criteria for selecting which resources the rule applies to, such as resource tags.
     */
    public final String selectionCriteria() {
        return selectionCriteria;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(telemetryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTelemetrySourceTypes() ? telemetrySourceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(destinationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(scope());
        hashCode = 31 * hashCode + Objects.hashCode(selectionCriteria());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TelemetryRule)) {
            return false;
        }
        TelemetryRule other = (TelemetryRule) obj;
        return Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(telemetryTypeAsString(), other.telemetryTypeAsString())
                && hasTelemetrySourceTypes() == other.hasTelemetrySourceTypes()
                && Objects.equals(telemetrySourceTypesAsStrings(), other.telemetrySourceTypesAsStrings())
                && Objects.equals(destinationConfiguration(), other.destinationConfiguration())
                && Objects.equals(scope(), other.scope()) && Objects.equals(selectionCriteria(), other.selectionCriteria());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TelemetryRule").add("ResourceType", resourceTypeAsString())
                .add("TelemetryType", telemetryTypeAsString())
                .add("TelemetrySourceTypes", hasTelemetrySourceTypes() ? telemetrySourceTypesAsStrings() : null)
                .add("DestinationConfiguration", destinationConfiguration()).add("Scope", scope())
                .add("SelectionCriteria", selectionCriteria()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "TelemetryType":
            return Optional.ofNullable(clazz.cast(telemetryTypeAsString()));
        case "TelemetrySourceTypes":
            return Optional.ofNullable(clazz.cast(telemetrySourceTypesAsStrings()));
        case "DestinationConfiguration":
            return Optional.ofNullable(clazz.cast(destinationConfiguration()));
        case "Scope":
            return Optional.ofNullable(clazz.cast(scope()));
        case "SelectionCriteria":
            return Optional.ofNullable(clazz.cast(selectionCriteria()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("TelemetryType", TELEMETRY_TYPE_FIELD);
        map.put("TelemetrySourceTypes", TELEMETRY_SOURCE_TYPES_FIELD);
        map.put("DestinationConfiguration", DESTINATION_CONFIGURATION_FIELD);
        map.put("Scope", SCOPE_FIELD);
        map.put("SelectionCriteria", SELECTION_CRITERIA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TelemetryRule, T> g) {
        return obj -> g.apply((TelemetryRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TelemetryRule> {
        /**
         * <p>
         * The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC",
         * "AWS::EKS::Cluster", "AWS::WAFv2::WebACL").
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC",
         *        "AWS::EKS::Cluster", "AWS::WAFv2::WebACL").
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC",
         * "AWS::EKS::Cluster", "AWS::WAFv2::WebACL").
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource to configure telemetry for (e.g., "AWS::EC2::VPC",
         *        "AWS::EKS::Cluster", "AWS::WAFv2::WebACL").
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The type of telemetry to collect (Logs, Metrics, or Traces).
         * </p>
         * 
         * @param telemetryType
         *        The type of telemetry to collect (Logs, Metrics, or Traces).
         * @see TelemetryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TelemetryType
         */
        Builder telemetryType(String telemetryType);

        /**
         * <p>
         * The type of telemetry to collect (Logs, Metrics, or Traces).
         * </p>
         * 
         * @param telemetryType
         *        The type of telemetry to collect (Logs, Metrics, or Traces).
         * @see TelemetryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TelemetryType
         */
        Builder telemetryType(TelemetryType telemetryType);

        /**
         * <p>
         * The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or EKS_AUDIT_LOGS.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or
         *        EKS_AUDIT_LOGS. TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypesWithStrings(Collection<String> telemetrySourceTypes);

        /**
         * <p>
         * The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or EKS_AUDIT_LOGS.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or
         *        EKS_AUDIT_LOGS. TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypesWithStrings(String... telemetrySourceTypes);

        /**
         * <p>
         * The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or EKS_AUDIT_LOGS.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or
         *        EKS_AUDIT_LOGS. TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypes(Collection<TelemetrySourceType> telemetrySourceTypes);

        /**
         * <p>
         * The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or EKS_AUDIT_LOGS.
         * TelemetrySourceTypes must be correlated with the specific resource type.
         * </p>
         * 
         * @param telemetrySourceTypes
         *        The specific telemetry source types to configure for the resource, such as VPC_FLOW_LOGS or
         *        EKS_AUDIT_LOGS. TelemetrySourceTypes must be correlated with the specific resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetrySourceTypes(TelemetrySourceType... telemetrySourceTypes);

        /**
         * <p>
         * Configuration specifying where and how the telemetry data should be delivered.
         * </p>
         * 
         * @param destinationConfiguration
         *        Configuration specifying where and how the telemetry data should be delivered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationConfiguration(TelemetryDestinationConfiguration destinationConfiguration);

        /**
         * <p>
         * Configuration specifying where and how the telemetry data should be delivered.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link TelemetryDestinationConfiguration.Builder} avoiding the need to create one manually via
         * {@link TelemetryDestinationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TelemetryDestinationConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #destinationConfiguration(TelemetryDestinationConfiguration)}.
         * 
         * @param destinationConfiguration
         *        a consumer that will call methods on {@link TelemetryDestinationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationConfiguration(TelemetryDestinationConfiguration)
         */
        default Builder destinationConfiguration(Consumer<TelemetryDestinationConfiguration.Builder> destinationConfiguration) {
            return destinationConfiguration(TelemetryDestinationConfiguration.builder().applyMutation(destinationConfiguration)
                    .build());
        }

        /**
         * <p>
         * The organizational scope to which the rule applies, specified using accounts or organizational units.
         * </p>
         * 
         * @param scope
         *        The organizational scope to which the rule applies, specified using accounts or organizational units.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scope(String scope);

        /**
         * <p>
         * Criteria for selecting which resources the rule applies to, such as resource tags.
         * </p>
         * 
         * @param selectionCriteria
         *        Criteria for selecting which resources the rule applies to, such as resource tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectionCriteria(String selectionCriteria);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private String telemetryType;

        private List<String> telemetrySourceTypes = DefaultSdkAutoConstructList.getInstance();

        private TelemetryDestinationConfiguration destinationConfiguration;

        private String scope;

        private String selectionCriteria;

        private BuilderImpl() {
        }

        private BuilderImpl(TelemetryRule model) {
            resourceType(model.resourceType);
            telemetryType(model.telemetryType);
            telemetrySourceTypesWithStrings(model.telemetrySourceTypes);
            destinationConfiguration(model.destinationConfiguration);
            scope(model.scope);
            selectionCriteria(model.selectionCriteria);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getTelemetryType() {
            return telemetryType;
        }

        public final void setTelemetryType(String telemetryType) {
            this.telemetryType = telemetryType;
        }

        @Override
        public final Builder telemetryType(String telemetryType) {
            this.telemetryType = telemetryType;
            return this;
        }

        @Override
        public final Builder telemetryType(TelemetryType telemetryType) {
            this.telemetryType(telemetryType == null ? null : telemetryType.toString());
            return this;
        }

        public final Collection<String> getTelemetrySourceTypes() {
            if (telemetrySourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return telemetrySourceTypes;
        }

        public final void setTelemetrySourceTypes(Collection<String> telemetrySourceTypes) {
            this.telemetrySourceTypes = TelemetrySourceTypesCopier.copy(telemetrySourceTypes);
        }

        @Override
        public final Builder telemetrySourceTypesWithStrings(Collection<String> telemetrySourceTypes) {
            this.telemetrySourceTypes = TelemetrySourceTypesCopier.copy(telemetrySourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder telemetrySourceTypesWithStrings(String... telemetrySourceTypes) {
            telemetrySourceTypesWithStrings(Arrays.asList(telemetrySourceTypes));
            return this;
        }

        @Override
        public final Builder telemetrySourceTypes(Collection<TelemetrySourceType> telemetrySourceTypes) {
            this.telemetrySourceTypes = TelemetrySourceTypesCopier.copyEnumToString(telemetrySourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder telemetrySourceTypes(TelemetrySourceType... telemetrySourceTypes) {
            telemetrySourceTypes(Arrays.asList(telemetrySourceTypes));
            return this;
        }

        public final TelemetryDestinationConfiguration.Builder getDestinationConfiguration() {
            return destinationConfiguration != null ? destinationConfiguration.toBuilder() : null;
        }

        public final void setDestinationConfiguration(TelemetryDestinationConfiguration.BuilderImpl destinationConfiguration) {
            this.destinationConfiguration = destinationConfiguration != null ? destinationConfiguration.build() : null;
        }

        @Override
        public final Builder destinationConfiguration(TelemetryDestinationConfiguration destinationConfiguration) {
            this.destinationConfiguration = destinationConfiguration;
            return this;
        }

        public final String getScope() {
            return scope;
        }

        public final void setScope(String scope) {
            this.scope = scope;
        }

        @Override
        public final Builder scope(String scope) {
            this.scope = scope;
            return this;
        }

        public final String getSelectionCriteria() {
            return selectionCriteria;
        }

        public final void setSelectionCriteria(String selectionCriteria) {
            this.selectionCriteria = selectionCriteria;
        }

        @Override
        public final Builder selectionCriteria(String selectionCriteria) {
            this.selectionCriteria = selectionCriteria;
            return this;
        }

        @Override
        public TelemetryRule build() {
            return new TelemetryRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
