/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.observabilityadmin.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A model representing the state of a resource within an account according to telemetry config.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TelemetryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<TelemetryConfiguration.Builder, TelemetryConfiguration> {
    private static final SdkField<String> ACCOUNT_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountIdentifier").getter(getter(TelemetryConfiguration::accountIdentifier))
            .setter(setter(Builder::accountIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountIdentifier").build()).build();

    private static final SdkField<Map<String, String>> TELEMETRY_CONFIGURATION_STATE_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("TelemetryConfigurationState")
            .getter(getter(TelemetryConfiguration::telemetryConfigurationStateAsStrings))
            .setter(setter(Builder::telemetryConfigurationStateWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TelemetryConfigurationState")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(TelemetryConfiguration::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> RESOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceIdentifier").getter(getter(TelemetryConfiguration::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifier").build())
            .build();

    private static final SdkField<Map<String, String>> RESOURCE_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ResourceTags")
            .getter(getter(TelemetryConfiguration::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Long> LAST_UPDATE_TIME_STAMP_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("LastUpdateTimeStamp").getter(getter(TelemetryConfiguration::lastUpdateTimeStamp))
            .setter(setter(Builder::lastUpdateTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTimeStamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_IDENTIFIER_FIELD,
            TELEMETRY_CONFIGURATION_STATE_FIELD, RESOURCE_TYPE_FIELD, RESOURCE_IDENTIFIER_FIELD, RESOURCE_TAGS_FIELD,
            LAST_UPDATE_TIME_STAMP_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String accountIdentifier;

    private final Map<String, String> telemetryConfigurationState;

    private final String resourceType;

    private final String resourceIdentifier;

    private final Map<String, String> resourceTags;

    private final Long lastUpdateTimeStamp;

    private TelemetryConfiguration(BuilderImpl builder) {
        this.accountIdentifier = builder.accountIdentifier;
        this.telemetryConfigurationState = builder.telemetryConfigurationState;
        this.resourceType = builder.resourceType;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.resourceTags = builder.resourceTags;
        this.lastUpdateTimeStamp = builder.lastUpdateTimeStamp;
    }

    /**
     * <p>
     * The account ID which contains the resource managed in telemetry configuration. An example of a valid account ID
     * is <code>012345678901</code>.
     * </p>
     * 
     * @return The account ID which contains the resource managed in telemetry configuration. An example of a valid
     *         account ID is <code>012345678901</code>.
     */
    public final String accountIdentifier() {
        return accountIdentifier;
    }

    /**
     * <p>
     * The configuration state for the resource, for example
     * <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetryConfigurationState} method.
     * </p>
     * 
     * @return The configuration state for the resource, for example
     *         <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
     */
    public final Map<TelemetryType, TelemetryState> telemetryConfigurationState() {
        return TelemetryConfigurationStateCopier.copyStringToEnum(telemetryConfigurationState);
    }

    /**
     * For responses, this returns true if the service returned a value for the TelemetryConfigurationState property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasTelemetryConfigurationState() {
        return telemetryConfigurationState != null && !(telemetryConfigurationState instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration state for the resource, for example
     * <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTelemetryConfigurationState} method.
     * </p>
     * 
     * @return The configuration state for the resource, for example
     *         <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
     */
    public final Map<String, String> telemetryConfigurationStateAsStrings() {
        return telemetryConfigurationState;
    }

    /**
     * <p>
     * The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
     * <code>Amazon Web Services::EKS::Cluster</code>, etc.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
     *         <code>Amazon Web Services::EKS::Cluster</code>, etc.
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
     * <code>Amazon Web Services::EKS::Cluster</code>, etc.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
     *         <code>Amazon Web Services::EKS::Cluster</code>, etc.
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The identifier of the resource, for example for Amazon VPC, it would be <code>vpc-1a2b3c4d5e6f1a2b3</code>.
     * </p>
     * 
     * @return The identifier of the resource, for example for Amazon VPC, it would be
     *         <code>vpc-1a2b3c4d5e6f1a2b3</code>.
     */
    public final String resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags associated with the resource, for example
     * <code>{ Name: "ExampleInstance", Environment: "Development" }</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTags} method.
     * </p>
     * 
     * @return Tags associated with the resource, for example
     *         <code>{ Name: "ExampleInstance", Environment: "Development" }</code>.
     */
    public final Map<String, String> resourceTags() {
        return resourceTags;
    }

    /**
     * <p>
     * The timestamp of the last change to the telemetry configuration for the resource. For example,
     * <code>1728679196318</code>.
     * </p>
     * 
     * @return The timestamp of the last change to the telemetry configuration for the resource. For example,
     *         <code>1728679196318</code>.
     */
    public final Long lastUpdateTimeStamp() {
        return lastUpdateTimeStamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountIdentifier());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasTelemetryConfigurationState() ? telemetryConfigurationStateAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimeStamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TelemetryConfiguration)) {
            return false;
        }
        TelemetryConfiguration other = (TelemetryConfiguration) obj;
        return Objects.equals(accountIdentifier(), other.accountIdentifier())
                && hasTelemetryConfigurationState() == other.hasTelemetryConfigurationState()
                && Objects.equals(telemetryConfigurationStateAsStrings(), other.telemetryConfigurationStateAsStrings())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && hasResourceTags() == other.hasResourceTags() && Objects.equals(resourceTags(), other.resourceTags())
                && Objects.equals(lastUpdateTimeStamp(), other.lastUpdateTimeStamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("TelemetryConfiguration")
                .add("AccountIdentifier", accountIdentifier())
                .add("TelemetryConfigurationState",
                        hasTelemetryConfigurationState() ? telemetryConfigurationStateAsStrings() : null)
                .add("ResourceType", resourceTypeAsString()).add("ResourceIdentifier", resourceIdentifier())
                .add("ResourceTags", hasResourceTags() ? resourceTags() : null).add("LastUpdateTimeStamp", lastUpdateTimeStamp())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountIdentifier":
            return Optional.ofNullable(clazz.cast(accountIdentifier()));
        case "TelemetryConfigurationState":
            return Optional.ofNullable(clazz.cast(telemetryConfigurationStateAsStrings()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "ResourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        case "LastUpdateTimeStamp":
            return Optional.ofNullable(clazz.cast(lastUpdateTimeStamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AccountIdentifier", ACCOUNT_IDENTIFIER_FIELD);
        map.put("TelemetryConfigurationState", TELEMETRY_CONFIGURATION_STATE_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("ResourceIdentifier", RESOURCE_IDENTIFIER_FIELD);
        map.put("ResourceTags", RESOURCE_TAGS_FIELD);
        map.put("LastUpdateTimeStamp", LAST_UPDATE_TIME_STAMP_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TelemetryConfiguration, T> g) {
        return obj -> g.apply((TelemetryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TelemetryConfiguration> {
        /**
         * <p>
         * The account ID which contains the resource managed in telemetry configuration. An example of a valid account
         * ID is <code>012345678901</code>.
         * </p>
         * 
         * @param accountIdentifier
         *        The account ID which contains the resource managed in telemetry configuration. An example of a valid
         *        account ID is <code>012345678901</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIdentifier(String accountIdentifier);

        /**
         * <p>
         * The configuration state for the resource, for example
         * <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
         * </p>
         * 
         * @param telemetryConfigurationState
         *        The configuration state for the resource, for example
         *        <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetryConfigurationStateWithStrings(Map<String, String> telemetryConfigurationState);

        /**
         * <p>
         * The configuration state for the resource, for example
         * <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
         * </p>
         * 
         * @param telemetryConfigurationState
         *        The configuration state for the resource, for example
         *        <code>{ Logs: NotApplicable; Metrics: Enabled; Traces: NotApplicable; }</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder telemetryConfigurationState(Map<TelemetryType, TelemetryState> telemetryConfigurationState);

        /**
         * <p>
         * The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
         * <code>Amazon Web Services::EKS::Cluster</code>, etc.
         * </p>
         * 
         * @param resourceType
         *        The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
         *        <code>Amazon Web Services::EKS::Cluster</code>, etc.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
         * <code>Amazon Web Services::EKS::Cluster</code>, etc.
         * </p>
         * 
         * @param resourceType
         *        The type of resource, for example <code>Amazon Web Services::EC2::Instance</code>, or
         *        <code>Amazon Web Services::EKS::Cluster</code>, etc.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The identifier of the resource, for example for Amazon VPC, it would be <code>vpc-1a2b3c4d5e6f1a2b3</code>.
         * </p>
         * 
         * @param resourceIdentifier
         *        The identifier of the resource, for example for Amazon VPC, it would be
         *        <code>vpc-1a2b3c4d5e6f1a2b3</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(String resourceIdentifier);

        /**
         * <p>
         * Tags associated with the resource, for example
         * <code>{ Name: "ExampleInstance", Environment: "Development" }</code>.
         * </p>
         * 
         * @param resourceTags
         *        Tags associated with the resource, for example
         *        <code>{ Name: "ExampleInstance", Environment: "Development" }</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Map<String, String> resourceTags);

        /**
         * <p>
         * The timestamp of the last change to the telemetry configuration for the resource. For example,
         * <code>1728679196318</code>.
         * </p>
         * 
         * @param lastUpdateTimeStamp
         *        The timestamp of the last change to the telemetry configuration for the resource. For example,
         *        <code>1728679196318</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimeStamp(Long lastUpdateTimeStamp);
    }

    static final class BuilderImpl implements Builder {
        private String accountIdentifier;

        private Map<String, String> telemetryConfigurationState = DefaultSdkAutoConstructMap.getInstance();

        private String resourceType;

        private String resourceIdentifier;

        private Map<String, String> resourceTags = DefaultSdkAutoConstructMap.getInstance();

        private Long lastUpdateTimeStamp;

        private BuilderImpl() {
        }

        private BuilderImpl(TelemetryConfiguration model) {
            accountIdentifier(model.accountIdentifier);
            telemetryConfigurationStateWithStrings(model.telemetryConfigurationState);
            resourceType(model.resourceType);
            resourceIdentifier(model.resourceIdentifier);
            resourceTags(model.resourceTags);
            lastUpdateTimeStamp(model.lastUpdateTimeStamp);
        }

        public final String getAccountIdentifier() {
            return accountIdentifier;
        }

        public final void setAccountIdentifier(String accountIdentifier) {
            this.accountIdentifier = accountIdentifier;
        }

        @Override
        public final Builder accountIdentifier(String accountIdentifier) {
            this.accountIdentifier = accountIdentifier;
            return this;
        }

        public final Map<String, String> getTelemetryConfigurationState() {
            if (telemetryConfigurationState instanceof SdkAutoConstructMap) {
                return null;
            }
            return telemetryConfigurationState;
        }

        public final void setTelemetryConfigurationState(Map<String, String> telemetryConfigurationState) {
            this.telemetryConfigurationState = TelemetryConfigurationStateCopier.copy(telemetryConfigurationState);
        }

        @Override
        public final Builder telemetryConfigurationStateWithStrings(Map<String, String> telemetryConfigurationState) {
            this.telemetryConfigurationState = TelemetryConfigurationStateCopier.copy(telemetryConfigurationState);
            return this;
        }

        @Override
        public final Builder telemetryConfigurationState(Map<TelemetryType, TelemetryState> telemetryConfigurationState) {
            this.telemetryConfigurationState = TelemetryConfigurationStateCopier.copyEnumToString(telemetryConfigurationState);
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getResourceIdentifier() {
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
        }

        @Override
        public final Builder resourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final Map<String, String> getResourceTags() {
            if (resourceTags instanceof SdkAutoConstructMap) {
                return null;
            }
            return resourceTags;
        }

        public final void setResourceTags(Map<String, String> resourceTags) {
            this.resourceTags = TagMapOutputCopier.copy(resourceTags);
        }

        @Override
        public final Builder resourceTags(Map<String, String> resourceTags) {
            this.resourceTags = TagMapOutputCopier.copy(resourceTags);
            return this;
        }

        public final Long getLastUpdateTimeStamp() {
            return lastUpdateTimeStamp;
        }

        public final void setLastUpdateTimeStamp(Long lastUpdateTimeStamp) {
            this.lastUpdateTimeStamp = lastUpdateTimeStamp;
        }

        @Override
        public final Builder lastUpdateTimeStamp(Long lastUpdateTimeStamp) {
            this.lastUpdateTimeStamp = lastUpdateTimeStamp;
            return this;
        }

        @Override
        public TelemetryConfiguration build() {
            return new TelemetryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
