/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkflowmonitor;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.networkflowmonitor.internal.NetworkFlowMonitorServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.networkflowmonitor.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.networkflowmonitor.model.AccessDeniedException;
import software.amazon.awssdk.services.networkflowmonitor.model.ConflictException;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.CreateScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.DeleteScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryResultsWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetQueryStatusWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.GetScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.GetScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.InternalServerException;
import software.amazon.awssdk.services.networkflowmonitor.model.ListMonitorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.ListMonitorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ListScopesRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.ListScopesResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.NetworkFlowMonitorException;
import software.amazon.awssdk.services.networkflowmonitor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.networkflowmonitor.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StartQueryWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryMonitorTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryMonitorTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsDataRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsDataResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.StopQueryWorkloadInsightsTopContributorsResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.TagResourceRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.TagResourceResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ThrottlingException;
import software.amazon.awssdk.services.networkflowmonitor.model.UntagResourceRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.UntagResourceResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateMonitorRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateMonitorResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateScopeRequest;
import software.amazon.awssdk.services.networkflowmonitor.model.UpdateScopeResponse;
import software.amazon.awssdk.services.networkflowmonitor.model.ValidationException;
import software.amazon.awssdk.services.networkflowmonitor.transform.CreateMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.CreateScopeRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.DeleteMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.DeleteScopeRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryResultsMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryResultsWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryResultsWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryStatusMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryStatusWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetQueryStatusWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.GetScopeRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.ListMonitorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.ListScopesRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StartQueryMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StartQueryWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StartQueryWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StopQueryMonitorTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StopQueryWorkloadInsightsTopContributorsDataRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.StopQueryWorkloadInsightsTopContributorsRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.UpdateMonitorRequestMarshaller;
import software.amazon.awssdk.services.networkflowmonitor.transform.UpdateScopeRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link NetworkFlowMonitorClient}.
 *
 * @see NetworkFlowMonitorClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultNetworkFlowMonitorClient implements NetworkFlowMonitorClient {
    private static final Logger log = Logger.loggerFor(DefaultNetworkFlowMonitorClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultNetworkFlowMonitorClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "NetworkFlowMonitor" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a monitor for specific network flows between local and remote resources, so that you can monitor network
     * performance for one or several of your workloads. For each monitor, Network Flow Monitor publishes detailed
     * end-to-end performance metrics and a network health indicator (NHI) that informs you whether there were Amazon
     * Web Services network issues for one or more of the network flows tracked by a monitor, during a time period that
     * you choose.
     * </p>
     *
     * @param createMonitorRequest
     * @return Result of the CreateMonitor operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/CreateMonitor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMonitorResponse createMonitor(CreateMonitorRequest createMonitorRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMonitor");

            return clientHandler.execute(new ClientExecutionParams<CreateMonitorRequest, CreateMonitorResponse>()
                    .withOperationName("CreateMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * In Network Flow Monitor, you specify a scope for the service to generate metrics for. By using the scope, Network
     * Flow Monitor can generate a topology of all the resources to measure performance metrics for. When you create a
     * scope, you enable permissions for Network Flow Monitor.
     * </p>
     * <p>
     * A scope is a Region-account pair or multiple Region-account pairs. Network Flow Monitor uses your scope to
     * determine all the resources (the topology) where Network Flow Monitor will gather network flow performance
     * metrics for you. To provide performance metrics, Network Flow Monitor uses the data that is sent by the Network
     * Flow Monitor agents you install on the resources.
     * </p>
     * <p>
     * To define the Region-account pairs for your scope, the Network Flow Monitor API uses the following constucts,
     * which allow for future flexibility in defining scopes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Targets</i>, which are arrays of targetResources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Target resources</i>, which are Region-targetIdentifier pairs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Target identifiers</i>, made up of a targetID (currently always an account ID) and a targetType (currently
     * always an account).
     * </p>
     * </li>
     * </ul>
     *
     * @param createScopeRequest
     * @return Result of the CreateScope operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.CreateScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/CreateScope"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateScopeResponse createScope(CreateScopeRequest createScopeRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateScopeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScope");

            return clientHandler.execute(new ClientExecutionParams<CreateScopeRequest, CreateScopeResponse>()
                    .withOperationName("CreateScope").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createScopeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateScopeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a monitor in Network Flow Monitor.
     * </p>
     *
     * @param deleteMonitorRequest
     * @return Result of the DeleteMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/DeleteMonitor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMonitorResponse deleteMonitor(DeleteMonitorRequest deleteMonitorRequest) throws ResourceNotFoundException,
            ConflictException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMonitor");

            return clientHandler.execute(new ClientExecutionParams<DeleteMonitorRequest, DeleteMonitorResponse>()
                    .withOperationName("DeleteMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a scope that has been defined.
     * </p>
     *
     * @param deleteScopeRequest
     * @return Result of the DeleteScope operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.DeleteScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/DeleteScope"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScopeResponse deleteScope(DeleteScopeRequest deleteScopeRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ConflictException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteScopeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScope");

            return clientHandler.execute(new ClientExecutionParams<DeleteScopeRequest, DeleteScopeResponse>()
                    .withOperationName("DeleteScope").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteScopeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteScopeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a monitor in Network Flow Monitor based on a monitor name. The information returned
     * includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor, and
     * status information.
     * </p>
     *
     * @param getMonitorRequest
     * @return Result of the GetMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMonitorResponse getMonitor(GetMonitorRequest getMonitorRequest) throws ResourceNotFoundException,
            ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMonitor");

            return clientHandler.execute(new ClientExecutionParams<GetMonitorRequest, GetMonitorResponse>()
                    .withOperationName("GetMonitor").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getMonitorRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. You specify the query that you want to
     * return results for by providing a query ID and a monitor name. This query returns the top contributors for a
     * specific monitor.
     * </p>
     * <p>
     * Create a query ID for this call by calling the corresponding API call to start the query,
     * <code>StartQueryMonitorTopContributors</code>. Use the scope ID that was returned for your account by
     * <code>CreateScope</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param getQueryResultsMonitorTopContributorsRequest
     * @return Result of the GetQueryResultsMonitorTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetQueryResultsMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryResultsMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryResultsMonitorTopContributorsResponse getQueryResultsMonitorTopContributors(
            GetQueryResultsMonitorTopContributorsRequest getQueryResultsMonitorTopContributorsRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryResultsMonitorTopContributorsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetQueryResultsMonitorTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryResultsMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryResultsMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResultsMonitorTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsMonitorTopContributorsRequest, GetQueryResultsMonitorTopContributorsResponse>()
                            .withOperationName("GetQueryResultsMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getQueryResultsMonitorTopContributorsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetQueryResultsMonitorTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. You specify the query that you want to
     * return results for by providing a query ID and a monitor name.
     * </p>
     * <p>
     * This query returns the top contributors for a scope for workload insights. Workload insights provide a high level
     * view of network flow performance data collected by agents. To return the data for the top contributors, see
     * <code>GetQueryResultsWorkloadInsightsTopContributorsData</code>.
     * </p>
     * <p>
     * Create a query ID for this call by calling the corresponding API call to start the query,
     * <code>StartQueryWorkloadInsightsTopContributors</code>. Use the scope ID that was returned for your account by
     * <code>CreateScope</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param getQueryResultsWorkloadInsightsTopContributorsRequest
     * @return Result of the GetQueryResultsWorkloadInsightsTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetQueryResultsWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryResultsWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryResultsWorkloadInsightsTopContributorsResponse getQueryResultsWorkloadInsightsTopContributors(
            GetQueryResultsWorkloadInsightsTopContributorsRequest getQueryResultsWorkloadInsightsTopContributorsRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryResultsWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetQueryResultsWorkloadInsightsTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryResultsWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryResultsWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResultsWorkloadInsightsTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsWorkloadInsightsTopContributorsRequest, GetQueryResultsWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("GetQueryResultsWorkloadInsightsTopContributors")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getQueryResultsWorkloadInsightsTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetQueryResultsWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. Specify the query that you want to
     * return results for by providing a query ID and a scope ID.
     * </p>
     * <p>
     * This query returns the data for top contributors for workload insights for a specific scope. Workload insights
     * provide a high level view of network flow performance data collected by agents for a scope. To return just the
     * top contributors, see <code>GetQueryResultsWorkloadInsightsTopContributors</code>.
     * </p>
     * <p>
     * Create a query ID for this call by calling the corresponding API call to start the query,
     * <code>StartQueryWorkloadInsightsTopContributorsData</code>. Use the scope ID that was returned for your account
     * by <code>CreateScope</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     * <p>
     * The top contributor network flows overall are for a specific metric type, for example, the number of
     * retransmissions.
     * </p>
     *
     * @param getQueryResultsWorkloadInsightsTopContributorsDataRequest
     * @return Result of the GetQueryResultsWorkloadInsightsTopContributorsData operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetQueryResultsWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryResultsWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryResultsWorkloadInsightsTopContributorsDataResponse getQueryResultsWorkloadInsightsTopContributorsData(
            GetQueryResultsWorkloadInsightsTopContributorsDataRequest getQueryResultsWorkloadInsightsTopContributorsDataRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ValidationException, ThrottlingException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryResultsWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetQueryResultsWorkloadInsightsTopContributorsDataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryResultsWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryResultsWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResultsWorkloadInsightsTopContributorsData");

            return clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsWorkloadInsightsTopContributorsDataRequest, GetQueryResultsWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("GetQueryResultsWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getQueryResultsWorkloadInsightsTopContributorsDataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(
                                    new GetQueryResultsWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current status of a query for the Network Flow Monitor query interface, for a specified query ID and
     * monitor. This call returns the query status for the top contributors for a monitor.
     * </p>
     * <p>
     * When you create a query, use this call to check the status of the query to make sure that it has has
     * <code>SUCCEEDED</code> before you review the results. Use the same query ID that you used for the corresponding
     * API call to start (create) the query, <code>StartQueryMonitorTopContributors</code>.
     * </p>
     * <p>
     * When you run a query, use this call to check the status of the query to make sure that the query has
     * <code>SUCCEEDED</code> before you review the results.
     * </p>
     *
     * @param getQueryStatusMonitorTopContributorsRequest
     * @return Result of the GetQueryStatusMonitorTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetQueryStatusMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryStatusMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryStatusMonitorTopContributorsResponse getQueryStatusMonitorTopContributors(
            GetQueryStatusMonitorTopContributorsRequest getQueryStatusMonitorTopContributorsRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryStatusMonitorTopContributorsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetQueryStatusMonitorTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryStatusMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryStatusMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatusMonitorTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatusMonitorTopContributorsRequest, GetQueryStatusMonitorTopContributorsResponse>()
                            .withOperationName("GetQueryStatusMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getQueryStatusMonitorTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetQueryStatusMonitorTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return the data for a query with the Network Flow Monitor query interface. Specify the query that you want to
     * return results for by providing a query ID and a monitor name. This query returns the top contributors for
     * workload insights.
     * </p>
     * <p>
     * When you start a query, use this call to check the status of the query to make sure that it has has
     * <code>SUCCEEDED</code> before you review the results. Use the same query ID that you used for the corresponding
     * API call to start the query, <code>StartQueryWorkloadInsightsTopContributors</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param getQueryStatusWorkloadInsightsTopContributorsRequest
     * @return Result of the GetQueryStatusWorkloadInsightsTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetQueryStatusWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryStatusWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryStatusWorkloadInsightsTopContributorsResponse getQueryStatusWorkloadInsightsTopContributors(
            GetQueryStatusWorkloadInsightsTopContributorsRequest getQueryStatusWorkloadInsightsTopContributorsRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryStatusWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetQueryStatusWorkloadInsightsTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryStatusWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryStatusWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatusWorkloadInsightsTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatusWorkloadInsightsTopContributorsRequest, GetQueryStatusWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("GetQueryStatusWorkloadInsightsTopContributors")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getQueryStatusWorkloadInsightsTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetQueryStatusWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current status of a query for the Network Flow Monitor query interface, for a specified query ID and
     * monitor. This call returns the query status for the top contributors data for workload insights.
     * </p>
     * <p>
     * When you start a query, use this call to check the status of the query to make sure that it has has
     * <code>SUCCEEDED</code> before you review the results. Use the same query ID that you used for the corresponding
     * API call to start the query, <code>StartQueryWorkloadInsightsTopContributorsData</code>.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     * <p>
     * The top contributor network flows overall are for a specific metric type, for example, the number of
     * retransmissions.
     * </p>
     *
     * @param getQueryStatusWorkloadInsightsTopContributorsDataRequest
     * @return Result of the GetQueryStatusWorkloadInsightsTopContributorsData operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetQueryStatusWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetQueryStatusWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryStatusWorkloadInsightsTopContributorsDataResponse getQueryStatusWorkloadInsightsTopContributorsData(
            GetQueryStatusWorkloadInsightsTopContributorsDataRequest getQueryStatusWorkloadInsightsTopContributorsDataRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryStatusWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetQueryStatusWorkloadInsightsTopContributorsDataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getQueryStatusWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getQueryStatusWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatusWorkloadInsightsTopContributorsData");

            return clientHandler
                    .execute(new ClientExecutionParams<GetQueryStatusWorkloadInsightsTopContributorsDataRequest, GetQueryStatusWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("GetQueryStatusWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getQueryStatusWorkloadInsightsTopContributorsDataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(
                                    new GetQueryStatusWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a scope, including the name, status, tags, and target details. The scope in Network Flow
     * Monitor is an account.
     * </p>
     *
     * @param getScopeRequest
     * @return Result of the GetScope operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.GetScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/GetScope" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetScopeResponse getScope(GetScopeRequest getScopeRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetScopeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetScope");

            return clientHandler.execute(new ClientExecutionParams<GetScopeRequest, GetScopeResponse>()
                    .withOperationName("GetScope").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getScopeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetScopeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all monitors in an account. Optionally, you can list only monitors that have a specific status, by using the
     * <code>STATUS</code> parameter.
     * </p>
     *
     * @param listMonitorsRequest
     * @return Result of the ListMonitors operation returned by the service.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/ListMonitors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMonitorsResponse listMonitors(ListMonitorsRequest listMonitorsRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMonitorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitors");

            return clientHandler.execute(new ClientExecutionParams<ListMonitorsRequest, ListMonitorsResponse>()
                    .withOperationName("ListMonitors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMonitorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMonitorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all the scopes for an account.
     * </p>
     *
     * @param listScopesRequest
     * @return Result of the ListScopes operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.ListScopes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/ListScopes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListScopesResponse listScopes(ListScopesRequest listScopesRequest) throws ServiceQuotaExceededException,
            ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListScopesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListScopesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listScopesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listScopesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListScopes");

            return clientHandler.execute(new ClientExecutionParams<ListScopesRequest, ListScopesResponse>()
                    .withOperationName("ListScopes").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listScopesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListScopesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ConflictException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a query that you can use with the Network Flow Monitor query interface to return the top contributors for
     * a monitor. Specify the monitor that you want to create the query for.
     * </p>
     * <p>
     * The call returns a query ID that you can use with <a href=
     * "https://docs.aws.amazon.com/networkflowmonitor/2.0/APIReference/API_GetQueryResultsMonitorTopContributors.html">
     * GetQueryResultsMonitorTopContributors</a> to run the query and return the top contributors for a specific
     * monitor.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable APIs for the top contributors that you want to be returned.
     * </p>
     *
     * @param startQueryMonitorTopContributorsRequest
     * @return Result of the StartQueryMonitorTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.StartQueryMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StartQueryMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartQueryMonitorTopContributorsResponse startQueryMonitorTopContributors(
            StartQueryMonitorTopContributorsRequest startQueryMonitorTopContributorsRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartQueryMonitorTopContributorsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartQueryMonitorTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startQueryMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startQueryMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryMonitorTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<StartQueryMonitorTopContributorsRequest, StartQueryMonitorTopContributorsResponse>()
                            .withOperationName("StartQueryMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startQueryMonitorTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartQueryMonitorTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a query with the Network Flow Monitor query interface that you can run to return workload insights top
     * contributors. Specify the scope that you want to create a query for.
     * </p>
     * <p>
     * The call returns a query ID that you can use with <a href=
     * "https://docs.aws.amazon.com/networkflowmonitor/2.0/APIReference/API_GetQueryResultsWorkloadInsightsTopContributors.html"
     * > GetQueryResultsWorkloadInsightsTopContributors</a> to run the query and return the top contributors for the
     * workload insights for a scope.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable APIs for the top contributors that you want to be returned.
     * </p>
     *
     * @param startQueryWorkloadInsightsTopContributorsRequest
     * @return Result of the StartQueryWorkloadInsightsTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.StartQueryWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StartQueryWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartQueryWorkloadInsightsTopContributorsResponse startQueryWorkloadInsightsTopContributors(
            StartQueryWorkloadInsightsTopContributorsRequest startQueryWorkloadInsightsTopContributorsRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartQueryWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, StartQueryWorkloadInsightsTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                startQueryWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startQueryWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryWorkloadInsightsTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<StartQueryWorkloadInsightsTopContributorsRequest, StartQueryWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("StartQueryWorkloadInsightsTopContributors")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(startQueryWorkloadInsightsTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartQueryWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a query with the Network Flow Monitor query interface that you can run to return data for workload
     * insights top contributors. Specify the scope that you want to create a query for.
     * </p>
     * <p>
     * The call returns a query ID that you can use with <a href=
     * "https://docs.aws.amazon.com/networkflowmonitor/2.0/APIReference/API_GetQueryResultsWorkloadInsightsTopContributorsData.html"
     * > GetQueryResultsWorkloadInsightsTopContributorsData</a> to run the query and return the data for the top
     * contributors for the workload insights for a scope.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param startQueryWorkloadInsightsTopContributorsDataRequest
     * @return Result of the StartQueryWorkloadInsightsTopContributorsData operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.StartQueryWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StartQueryWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartQueryWorkloadInsightsTopContributorsDataResponse startQueryWorkloadInsightsTopContributorsData(
            StartQueryWorkloadInsightsTopContributorsDataRequest startQueryWorkloadInsightsTopContributorsDataRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartQueryWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, StartQueryWorkloadInsightsTopContributorsDataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                startQueryWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startQueryWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQueryWorkloadInsightsTopContributorsData");

            return clientHandler
                    .execute(new ClientExecutionParams<StartQueryWorkloadInsightsTopContributorsDataRequest, StartQueryWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("StartQueryWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(startQueryWorkloadInsightsTopContributorsDataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartQueryWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stop a top contributors query for a monitor. Specify the query that you want to stop by providing a query ID and
     * a monitor name.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param stopQueryMonitorTopContributorsRequest
     * @return Result of the StopQueryMonitorTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.StopQueryMonitorTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StopQueryMonitorTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopQueryMonitorTopContributorsResponse stopQueryMonitorTopContributors(
            StopQueryMonitorTopContributorsRequest stopQueryMonitorTopContributorsRequest) throws ServiceQuotaExceededException,
            ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopQueryMonitorTopContributorsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopQueryMonitorTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopQueryMonitorTopContributorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                stopQueryMonitorTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQueryMonitorTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<StopQueryMonitorTopContributorsRequest, StopQueryMonitorTopContributorsResponse>()
                            .withOperationName("StopQueryMonitorTopContributors").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(stopQueryMonitorTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopQueryMonitorTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stop a top contributors query for workload insights. Specify the query that you want to stop by providing a query
     * ID and a scope ID.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param stopQueryWorkloadInsightsTopContributorsRequest
     * @return Result of the StopQueryWorkloadInsightsTopContributors operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.StopQueryWorkloadInsightsTopContributors
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StopQueryWorkloadInsightsTopContributors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopQueryWorkloadInsightsTopContributorsResponse stopQueryWorkloadInsightsTopContributors(
            StopQueryWorkloadInsightsTopContributorsRequest stopQueryWorkloadInsightsTopContributorsRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopQueryWorkloadInsightsTopContributorsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, StopQueryWorkloadInsightsTopContributorsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                stopQueryWorkloadInsightsTopContributorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                stopQueryWorkloadInsightsTopContributorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQueryWorkloadInsightsTopContributors");

            return clientHandler
                    .execute(new ClientExecutionParams<StopQueryWorkloadInsightsTopContributorsRequest, StopQueryWorkloadInsightsTopContributorsResponse>()
                            .withOperationName("StopQueryWorkloadInsightsTopContributors").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(stopQueryWorkloadInsightsTopContributorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopQueryWorkloadInsightsTopContributorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stop a top contributors data query for workload insights. Specify the query that you want to stop by providing a
     * query ID and a scope ID.
     * </p>
     * <p>
     * Top contributors in Network Flow Monitor are network flows with the highest values for a specific metric type.
     * Top contributors can be across all workload insights, for a given scope, or for a specific monitor. Use the
     * applicable call for the top contributors that you want to be returned.
     * </p>
     *
     * @param stopQueryWorkloadInsightsTopContributorsDataRequest
     * @return Result of the StopQueryWorkloadInsightsTopContributorsData operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.StopQueryWorkloadInsightsTopContributorsData
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/StopQueryWorkloadInsightsTopContributorsData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopQueryWorkloadInsightsTopContributorsDataResponse stopQueryWorkloadInsightsTopContributorsData(
            StopQueryWorkloadInsightsTopContributorsDataRequest stopQueryWorkloadInsightsTopContributorsDataRequest)
            throws ServiceQuotaExceededException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopQueryWorkloadInsightsTopContributorsDataResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, StopQueryWorkloadInsightsTopContributorsDataResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                stopQueryWorkloadInsightsTopContributorsDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                stopQueryWorkloadInsightsTopContributorsDataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQueryWorkloadInsightsTopContributorsData");

            return clientHandler
                    .execute(new ClientExecutionParams<StopQueryWorkloadInsightsTopContributorsDataRequest, StopQueryWorkloadInsightsTopContributorsDataResponse>()
                            .withOperationName("StopQueryWorkloadInsightsTopContributorsData")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(stopQueryWorkloadInsightsTopContributorsDataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopQueryWorkloadInsightsTopContributorsDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ConflictException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ConflictException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a monitor to add or remove local or remote resources.
     * </p>
     *
     * @param updateMonitorRequest
     * @return Result of the UpdateMonitor operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/UpdateMonitor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMonitorResponse updateMonitor(UpdateMonitorRequest updateMonitorRequest) throws ResourceNotFoundException,
            ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateMonitorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMonitor");

            return clientHandler.execute(new ClientExecutionParams<UpdateMonitorRequest, UpdateMonitorResponse>()
                    .withOperationName("UpdateMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a scope to add or remove resources that you want to be available for Network Flow Monitor to generate
     * metrics for, when you have active agents on those resources sending metrics reports to the Network Flow Monitor
     * backend.
     * </p>
     *
     * @param updateScopeRequest
     * @return Result of the UpdateScope operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded a service quota.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws ValidationException
     *         Invalid request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NetworkFlowMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NetworkFlowMonitorClient.UpdateScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/networkflowmonitor-2023-04-19/UpdateScope"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateScopeResponse updateScope(UpdateScopeRequest updateScopeRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ConflictException, ValidationException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, NetworkFlowMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateScopeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "NetworkFlowMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScope");

            return clientHandler.execute(new ClientExecutionParams<UpdateScopeRequest, UpdateScopeResponse>()
                    .withOperationName("UpdateScope").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateScopeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateScopeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        NetworkFlowMonitorServiceClientConfigurationBuilder serviceConfigBuilder = new NetworkFlowMonitorServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(NetworkFlowMonitorException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    @Override
    public final NetworkFlowMonitorServiceClientConfiguration serviceClientConfiguration() {
        return new NetworkFlowMonitorServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
