/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The firewall defines the configuration settings for an AWS Network Firewall firewall. These settings include the
 * firewall policy, the subnets in your VPC to use for the firewall endpoints, and any tags that are attached to the
 * firewall AWS resource.
 * </p>
 * <p>
 * The status of the firewall, for example whether it's ready to filter network traffic, is provided in the
 * corresponding <a>FirewallStatus</a>. You can retrieve both objects by calling <a>DescribeFirewall</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Firewall implements SdkPojo, Serializable, ToCopyableBuilder<Firewall.Builder, Firewall> {
    private static final SdkField<String> FIREWALL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallName").getter(getter(Firewall::firewallName)).setter(setter(Builder::firewallName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallName").build()).build();

    private static final SdkField<String> FIREWALL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallArn").getter(getter(Firewall::firewallArn)).setter(setter(Builder::firewallArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallArn").build()).build();

    private static final SdkField<String> FIREWALL_POLICY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallPolicyArn").getter(getter(Firewall::firewallPolicyArn))
            .setter(setter(Builder::firewallPolicyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallPolicyArn").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(Firewall::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<SubnetMapping>> SUBNET_MAPPINGS_FIELD = SdkField
            .<List<SubnetMapping>> builder(MarshallingType.LIST)
            .memberName("SubnetMappings")
            .getter(getter(Firewall::subnetMappings))
            .setter(setter(Builder::subnetMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SubnetMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(SubnetMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> DELETE_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteProtection").getter(getter(Firewall::deleteProtection)).setter(setter(Builder::deleteProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteProtection").build()).build();

    private static final SdkField<Boolean> SUBNET_CHANGE_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SubnetChangeProtection").getter(getter(Firewall::subnetChangeProtection))
            .setter(setter(Builder::subnetChangeProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetChangeProtection").build())
            .build();

    private static final SdkField<Boolean> FIREWALL_POLICY_CHANGE_PROTECTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("FirewallPolicyChangeProtection")
            .getter(getter(Firewall::firewallPolicyChangeProtection))
            .setter(setter(Builder::firewallPolicyChangeProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallPolicyChangeProtection")
                    .build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Firewall::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> FIREWALL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallId").getter(getter(Firewall::firewallId)).setter(setter(Builder::firewallId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Firewall::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIREWALL_NAME_FIELD,
            FIREWALL_ARN_FIELD, FIREWALL_POLICY_ARN_FIELD, VPC_ID_FIELD, SUBNET_MAPPINGS_FIELD, DELETE_PROTECTION_FIELD,
            SUBNET_CHANGE_PROTECTION_FIELD, FIREWALL_POLICY_CHANGE_PROTECTION_FIELD, DESCRIPTION_FIELD, FIREWALL_ID_FIELD,
            TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String firewallName;

    private final String firewallArn;

    private final String firewallPolicyArn;

    private final String vpcId;

    private final List<SubnetMapping> subnetMappings;

    private final Boolean deleteProtection;

    private final Boolean subnetChangeProtection;

    private final Boolean firewallPolicyChangeProtection;

    private final String description;

    private final String firewallId;

    private final List<Tag> tags;

    private Firewall(BuilderImpl builder) {
        this.firewallName = builder.firewallName;
        this.firewallArn = builder.firewallArn;
        this.firewallPolicyArn = builder.firewallPolicyArn;
        this.vpcId = builder.vpcId;
        this.subnetMappings = builder.subnetMappings;
        this.deleteProtection = builder.deleteProtection;
        this.subnetChangeProtection = builder.subnetChangeProtection;
        this.firewallPolicyChangeProtection = builder.firewallPolicyChangeProtection;
        this.description = builder.description;
        this.firewallId = builder.firewallId;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The descriptive name of the firewall. You can't change the name of a firewall after you create it.
     * </p>
     * 
     * @return The descriptive name of the firewall. You can't change the name of a firewall after you create it.
     */
    public final String firewallName() {
        return firewallName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the firewall.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the firewall.
     */
    public final String firewallArn() {
        return firewallArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the firewall policy.
     * </p>
     * <p>
     * The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall policy
     * association, and you can use the same firewall policy for multiple firewalls.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the firewall policy.</p>
     *         <p>
     *         The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall
     *         policy association, and you can use the same firewall policy for multiple firewalls.
     */
    public final String firewallPolicyArn() {
        return firewallPolicyArn;
    }

    /**
     * <p>
     * The unique identifier of the VPC where the firewall is in use.
     * </p>
     * 
     * @return The unique identifier of the VPC where the firewall is in use.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * Returns true if the SubnetMappings property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSubnetMappings() {
        return subnetMappings != null && !(subnetMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a different
     * Availability Zone.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetMappings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a
     *         different Availability Zone.
     */
    public final List<SubnetMapping> subnetMappings() {
        return subnetMappings;
    }

    /**
     * <p>
     * A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code> indicates that
     * the firewall is protected against deletion. Use this setting to protect against accidentally deleting a firewall
     * that is in use. When you create a firewall, the operation initializes this flag to <code>TRUE</code>.
     * </p>
     * 
     * @return A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code> indicates
     *         that the firewall is protected against deletion. Use this setting to protect against accidentally
     *         deleting a firewall that is in use. When you create a firewall, the operation initializes this flag to
     *         <code>TRUE</code>.
     */
    public final Boolean deleteProtection() {
        return deleteProtection;
    }

    /**
     * <p>
     * A setting indicating whether the firewall is protected against changes to the subnet associations. Use this
     * setting to protect against accidentally modifying the subnet associations for a firewall that is in use. When you
     * create a firewall, the operation initializes this setting to <code>TRUE</code>.
     * </p>
     * 
     * @return A setting indicating whether the firewall is protected against changes to the subnet associations. Use
     *         this setting to protect against accidentally modifying the subnet associations for a firewall that is in
     *         use. When you create a firewall, the operation initializes this setting to <code>TRUE</code>.
     */
    public final Boolean subnetChangeProtection() {
        return subnetChangeProtection;
    }

    /**
     * <p>
     * A setting indicating whether the firewall is protected against a change to the firewall policy association. Use
     * this setting to protect against accidentally modifying the firewall policy for a firewall that is in use. When
     * you create a firewall, the operation initializes this setting to <code>TRUE</code>.
     * </p>
     * 
     * @return A setting indicating whether the firewall is protected against a change to the firewall policy
     *         association. Use this setting to protect against accidentally modifying the firewall policy for a
     *         firewall that is in use. When you create a firewall, the operation initializes this setting to
     *         <code>TRUE</code>.
     */
    public final Boolean firewallPolicyChangeProtection() {
        return firewallPolicyChangeProtection;
    }

    /**
     * <p>
     * A description of the firewall.
     * </p>
     * 
     * @return A description of the firewall.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The unique identifier for the firewall.
     * </p>
     * 
     * @return The unique identifier for the firewall.
     */
    public final String firewallId() {
        return firewallId;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(firewallName());
        hashCode = 31 * hashCode + Objects.hashCode(firewallArn());
        hashCode = 31 * hashCode + Objects.hashCode(firewallPolicyArn());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetMappings() ? subnetMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deleteProtection());
        hashCode = 31 * hashCode + Objects.hashCode(subnetChangeProtection());
        hashCode = 31 * hashCode + Objects.hashCode(firewallPolicyChangeProtection());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(firewallId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Firewall)) {
            return false;
        }
        Firewall other = (Firewall) obj;
        return Objects.equals(firewallName(), other.firewallName()) && Objects.equals(firewallArn(), other.firewallArn())
                && Objects.equals(firewallPolicyArn(), other.firewallPolicyArn()) && Objects.equals(vpcId(), other.vpcId())
                && hasSubnetMappings() == other.hasSubnetMappings() && Objects.equals(subnetMappings(), other.subnetMappings())
                && Objects.equals(deleteProtection(), other.deleteProtection())
                && Objects.equals(subnetChangeProtection(), other.subnetChangeProtection())
                && Objects.equals(firewallPolicyChangeProtection(), other.firewallPolicyChangeProtection())
                && Objects.equals(description(), other.description()) && Objects.equals(firewallId(), other.firewallId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Firewall").add("FirewallName", firewallName()).add("FirewallArn", firewallArn())
                .add("FirewallPolicyArn", firewallPolicyArn()).add("VpcId", vpcId())
                .add("SubnetMappings", hasSubnetMappings() ? subnetMappings() : null).add("DeleteProtection", deleteProtection())
                .add("SubnetChangeProtection", subnetChangeProtection())
                .add("FirewallPolicyChangeProtection", firewallPolicyChangeProtection()).add("Description", description())
                .add("FirewallId", firewallId()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FirewallName":
            return Optional.ofNullable(clazz.cast(firewallName()));
        case "FirewallArn":
            return Optional.ofNullable(clazz.cast(firewallArn()));
        case "FirewallPolicyArn":
            return Optional.ofNullable(clazz.cast(firewallPolicyArn()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetMappings":
            return Optional.ofNullable(clazz.cast(subnetMappings()));
        case "DeleteProtection":
            return Optional.ofNullable(clazz.cast(deleteProtection()));
        case "SubnetChangeProtection":
            return Optional.ofNullable(clazz.cast(subnetChangeProtection()));
        case "FirewallPolicyChangeProtection":
            return Optional.ofNullable(clazz.cast(firewallPolicyChangeProtection()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "FirewallId":
            return Optional.ofNullable(clazz.cast(firewallId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Firewall, T> g) {
        return obj -> g.apply((Firewall) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Firewall> {
        /**
         * <p>
         * The descriptive name of the firewall. You can't change the name of a firewall after you create it.
         * </p>
         * 
         * @param firewallName
         *        The descriptive name of the firewall. You can't change the name of a firewall after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallName(String firewallName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the firewall.
         * </p>
         * 
         * @param firewallArn
         *        The Amazon Resource Name (ARN) of the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallArn(String firewallArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the firewall policy.
         * </p>
         * <p>
         * The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall policy
         * association, and you can use the same firewall policy for multiple firewalls.
         * </p>
         * 
         * @param firewallPolicyArn
         *        The Amazon Resource Name (ARN) of the firewall policy.</p>
         *        <p>
         *        The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall
         *        policy association, and you can use the same firewall policy for multiple firewalls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallPolicyArn(String firewallPolicyArn);

        /**
         * <p>
         * The unique identifier of the VPC where the firewall is in use.
         * </p>
         * 
         * @param vpcId
         *        The unique identifier of the VPC where the firewall is in use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a different
         * Availability Zone.
         * </p>
         * 
         * @param subnetMappings
         *        The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a
         *        different Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetMappings(Collection<SubnetMapping> subnetMappings);

        /**
         * <p>
         * The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a different
         * Availability Zone.
         * </p>
         * 
         * @param subnetMappings
         *        The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a
         *        different Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetMappings(SubnetMapping... subnetMappings);

        /**
         * <p>
         * The public subnets that Network Firewall is using for the firewall. Each subnet must belong to a different
         * Availability Zone.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SubnetMapping>.Builder} avoiding the need
         * to create one manually via {@link List<SubnetMapping>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SubnetMapping>.Builder#build()} is called immediately and
         * its result is passed to {@link #subnetMappings(List<SubnetMapping>)}.
         * 
         * @param subnetMappings
         *        a consumer that will call methods on {@link List<SubnetMapping>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subnetMappings(List<SubnetMapping>)
         */
        Builder subnetMappings(Consumer<SubnetMapping.Builder>... subnetMappings);

        /**
         * <p>
         * A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code> indicates
         * that the firewall is protected against deletion. Use this setting to protect against accidentally deleting a
         * firewall that is in use. When you create a firewall, the operation initializes this flag to <code>TRUE</code>
         * .
         * </p>
         * 
         * @param deleteProtection
         *        A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code>
         *        indicates that the firewall is protected against deletion. Use this setting to protect against
         *        accidentally deleting a firewall that is in use. When you create a firewall, the operation initializes
         *        this flag to <code>TRUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteProtection(Boolean deleteProtection);

        /**
         * <p>
         * A setting indicating whether the firewall is protected against changes to the subnet associations. Use this
         * setting to protect against accidentally modifying the subnet associations for a firewall that is in use. When
         * you create a firewall, the operation initializes this setting to <code>TRUE</code>.
         * </p>
         * 
         * @param subnetChangeProtection
         *        A setting indicating whether the firewall is protected against changes to the subnet associations. Use
         *        this setting to protect against accidentally modifying the subnet associations for a firewall that is
         *        in use. When you create a firewall, the operation initializes this setting to <code>TRUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetChangeProtection(Boolean subnetChangeProtection);

        /**
         * <p>
         * A setting indicating whether the firewall is protected against a change to the firewall policy association.
         * Use this setting to protect against accidentally modifying the firewall policy for a firewall that is in use.
         * When you create a firewall, the operation initializes this setting to <code>TRUE</code>.
         * </p>
         * 
         * @param firewallPolicyChangeProtection
         *        A setting indicating whether the firewall is protected against a change to the firewall policy
         *        association. Use this setting to protect against accidentally modifying the firewall policy for a
         *        firewall that is in use. When you create a firewall, the operation initializes this setting to
         *        <code>TRUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallPolicyChangeProtection(Boolean firewallPolicyChangeProtection);

        /**
         * <p>
         * A description of the firewall.
         * </p>
         * 
         * @param description
         *        A description of the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The unique identifier for the firewall.
         * </p>
         * 
         * @param firewallId
         *        The unique identifier for the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallId(String firewallId);

        /**
         * <p/>
         * 
         * @param tags
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p/>
         * 
         * @param tags
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String firewallName;

        private String firewallArn;

        private String firewallPolicyArn;

        private String vpcId;

        private List<SubnetMapping> subnetMappings = DefaultSdkAutoConstructList.getInstance();

        private Boolean deleteProtection;

        private Boolean subnetChangeProtection;

        private Boolean firewallPolicyChangeProtection;

        private String description;

        private String firewallId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Firewall model) {
            firewallName(model.firewallName);
            firewallArn(model.firewallArn);
            firewallPolicyArn(model.firewallPolicyArn);
            vpcId(model.vpcId);
            subnetMappings(model.subnetMappings);
            deleteProtection(model.deleteProtection);
            subnetChangeProtection(model.subnetChangeProtection);
            firewallPolicyChangeProtection(model.firewallPolicyChangeProtection);
            description(model.description);
            firewallId(model.firewallId);
            tags(model.tags);
        }

        public final String getFirewallName() {
            return firewallName;
        }

        @Override
        public final Builder firewallName(String firewallName) {
            this.firewallName = firewallName;
            return this;
        }

        public final void setFirewallName(String firewallName) {
            this.firewallName = firewallName;
        }

        public final String getFirewallArn() {
            return firewallArn;
        }

        @Override
        public final Builder firewallArn(String firewallArn) {
            this.firewallArn = firewallArn;
            return this;
        }

        public final void setFirewallArn(String firewallArn) {
            this.firewallArn = firewallArn;
        }

        public final String getFirewallPolicyArn() {
            return firewallPolicyArn;
        }

        @Override
        public final Builder firewallPolicyArn(String firewallPolicyArn) {
            this.firewallPolicyArn = firewallPolicyArn;
            return this;
        }

        public final void setFirewallPolicyArn(String firewallPolicyArn) {
            this.firewallPolicyArn = firewallPolicyArn;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Collection<SubnetMapping.Builder> getSubnetMappings() {
            if (subnetMappings instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetMappings != null ? subnetMappings.stream().map(SubnetMapping::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder subnetMappings(Collection<SubnetMapping> subnetMappings) {
            this.subnetMappings = SubnetMappingsCopier.copy(subnetMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetMappings(SubnetMapping... subnetMappings) {
            subnetMappings(Arrays.asList(subnetMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetMappings(Consumer<SubnetMapping.Builder>... subnetMappings) {
            subnetMappings(Stream.of(subnetMappings).map(c -> SubnetMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSubnetMappings(Collection<SubnetMapping.BuilderImpl> subnetMappings) {
            this.subnetMappings = SubnetMappingsCopier.copyFromBuilder(subnetMappings);
        }

        public final Boolean getDeleteProtection() {
            return deleteProtection;
        }

        @Override
        public final Builder deleteProtection(Boolean deleteProtection) {
            this.deleteProtection = deleteProtection;
            return this;
        }

        public final void setDeleteProtection(Boolean deleteProtection) {
            this.deleteProtection = deleteProtection;
        }

        public final Boolean getSubnetChangeProtection() {
            return subnetChangeProtection;
        }

        @Override
        public final Builder subnetChangeProtection(Boolean subnetChangeProtection) {
            this.subnetChangeProtection = subnetChangeProtection;
            return this;
        }

        public final void setSubnetChangeProtection(Boolean subnetChangeProtection) {
            this.subnetChangeProtection = subnetChangeProtection;
        }

        public final Boolean getFirewallPolicyChangeProtection() {
            return firewallPolicyChangeProtection;
        }

        @Override
        public final Builder firewallPolicyChangeProtection(Boolean firewallPolicyChangeProtection) {
            this.firewallPolicyChangeProtection = firewallPolicyChangeProtection;
            return this;
        }

        public final void setFirewallPolicyChangeProtection(Boolean firewallPolicyChangeProtection) {
            this.firewallPolicyChangeProtection = firewallPolicyChangeProtection;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getFirewallId() {
            return firewallId;
        }

        @Override
        public final Builder firewallId(String firewallId) {
            this.firewallId = firewallId;
            return this;
        }

        public final void setFirewallId(String firewallId) {
            this.firewallId = firewallId;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Firewall build() {
            return new Firewall(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
