/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediatailor.model.BadRequestException;
import software.amazon.awssdk.services.mediatailor.model.DeletePlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.DeletePlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.GetPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.GetPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.MediaTailorException;
import software.amazon.awssdk.services.mediatailor.model.PutPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.PutPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.TagResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.TagResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediatailor.transform.DeletePlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListPlaybackConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.PutPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UntagResourceRequestMarshaller;

/**
 * Internal implementation of {@link MediaTailorClient}.
 *
 * @see MediaTailorClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaTailorClient implements MediaTailorClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaTailorClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the playback configuration for the specified name.
     * </p>
     *
     * @param deletePlaybackConfigurationRequest
     * @return Result of the DeletePlaybackConfiguration operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeletePlaybackConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeletePlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePlaybackConfigurationResponse deletePlaybackConfiguration(
            DeletePlaybackConfigurationRequest deletePlaybackConfigurationRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePlaybackConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeletePlaybackConfigurationRequest, DeletePlaybackConfigurationResponse>()
                        .withOperationName("DeletePlaybackConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deletePlaybackConfigurationRequest)
                        .withMarshaller(new DeletePlaybackConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the playback configuration for the specified name.
     * </p>
     *
     * @param getPlaybackConfigurationRequest
     * @return Result of the GetPlaybackConfiguration operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.GetPlaybackConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPlaybackConfigurationResponse getPlaybackConfiguration(
            GetPlaybackConfigurationRequest getPlaybackConfigurationRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPlaybackConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetPlaybackConfigurationRequest, GetPlaybackConfigurationResponse>()
                        .withOperationName("GetPlaybackConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getPlaybackConfigurationRequest)
                        .withMarshaller(new GetPlaybackConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the playback configurations defined in AWS Elemental MediaTailor. You can specify a maximum
     * number of configurations to return at a time. The default maximum is 50. Results are returned in pagefuls. If
     * MediaTailor has more configurations than the specified maximum, it provides parameters in the response that you
     * can use to retrieve the next pageful.
     * </p>
     *
     * @param listPlaybackConfigurationsRequest
     * @return Result of the ListPlaybackConfigurations operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListPlaybackConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPlaybackConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPlaybackConfigurationsResponse listPlaybackConfigurations(
            ListPlaybackConfigurationsRequest listPlaybackConfigurationsRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPlaybackConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPlaybackConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListPlaybackConfigurationsRequest, ListPlaybackConfigurationsResponse>()
                        .withOperationName("ListPlaybackConfigurations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listPlaybackConfigurationsRequest)
                        .withMarshaller(new ListPlaybackConfigurationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified playback configuration resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Invalid request parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds a new playback configuration to AWS Elemental MediaTailor.
     * </p>
     *
     * @param putPlaybackConfigurationRequest
     * @return Result of the PutPlaybackConfiguration operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.PutPlaybackConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/PutPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutPlaybackConfigurationResponse putPlaybackConfiguration(
            PutPlaybackConfigurationRequest putPlaybackConfigurationRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutPlaybackConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<PutPlaybackConfigurationRequest, PutPlaybackConfigurationResponse>()
                        .withOperationName("PutPlaybackConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(putPlaybackConfigurationRequest)
                        .withMarshaller(new PutPlaybackConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds tags to the specified playback configuration resource. You can specify one or more tags to add.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Invalid request parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes tags from the specified playback configuration resource. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         Invalid request parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaTailorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
