/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastore.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The metric policy that is associated with the container. A metric policy allows AWS Elemental MediaStore to send
 * metrics to Amazon CloudWatch. In the policy, you must indicate whether you want MediaStore to send container-level
 * metrics. You can also include rules to define groups of objects that you want MediaStore to send object-level metrics
 * for.
 * </p>
 * <p>
 * To view examples of how to construct a metric policy for your use case, see <a
 * href="https://docs.aws.amazon.com/mediastore/latest/ug/policies-metric-examples.html">Example Metric Policies</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricPolicy implements SdkPojo, Serializable, ToCopyableBuilder<MetricPolicy.Builder, MetricPolicy> {
    private static final SdkField<String> CONTAINER_LEVEL_METRICS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerLevelMetrics").getter(getter(MetricPolicy::containerLevelMetricsAsString))
            .setter(setter(Builder::containerLevelMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerLevelMetrics").build())
            .build();

    private static final SdkField<List<MetricPolicyRule>> METRIC_POLICY_RULES_FIELD = SdkField
            .<List<MetricPolicyRule>> builder(MarshallingType.LIST)
            .memberName("MetricPolicyRules")
            .getter(getter(MetricPolicy::metricPolicyRules))
            .setter(setter(Builder::metricPolicyRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricPolicyRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricPolicyRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricPolicyRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_LEVEL_METRICS_FIELD,
            METRIC_POLICY_RULES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerLevelMetrics;

    private final List<MetricPolicyRule> metricPolicyRules;

    private MetricPolicy(BuilderImpl builder) {
        this.containerLevelMetrics = builder.containerLevelMetrics;
        this.metricPolicyRules = builder.metricPolicyRules;
    }

    /**
     * <p>
     * A setting to enable or disable metrics at the container level.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #containerLevelMetrics} will return {@link ContainerLevelMetrics#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #containerLevelMetricsAsString}.
     * </p>
     * 
     * @return A setting to enable or disable metrics at the container level.
     * @see ContainerLevelMetrics
     */
    public final ContainerLevelMetrics containerLevelMetrics() {
        return ContainerLevelMetrics.fromValue(containerLevelMetrics);
    }

    /**
     * <p>
     * A setting to enable or disable metrics at the container level.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #containerLevelMetrics} will return {@link ContainerLevelMetrics#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #containerLevelMetricsAsString}.
     * </p>
     * 
     * @return A setting to enable or disable metrics at the container level.
     * @see ContainerLevelMetrics
     */
    public final String containerLevelMetricsAsString() {
        return containerLevelMetrics;
    }

    /**
     * Returns true if the MetricPolicyRules property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasMetricPolicyRules() {
        return metricPolicyRules != null && !(metricPolicyRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A parameter that holds an array of rules that enable metrics at the object level. This parameter is optional, but
     * if you choose to include it, you must also include at least one rule. By default, you can include up to five
     * rules. You can also <a
     * href="https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas">request a
     * quota increase</a> to allow up to 300 rules per policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMetricPolicyRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A parameter that holds an array of rules that enable metrics at the object level. This parameter is
     *         optional, but if you choose to include it, you must also include at least one rule. By default, you can
     *         include up to five rules. You can also <a
     *         href="https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas"
     *         >request a quota increase</a> to allow up to 300 rules per policy.
     */
    public final List<MetricPolicyRule> metricPolicyRules() {
        return metricPolicyRules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerLevelMetricsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricPolicyRules() ? metricPolicyRules() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricPolicy)) {
            return false;
        }
        MetricPolicy other = (MetricPolicy) obj;
        return Objects.equals(containerLevelMetricsAsString(), other.containerLevelMetricsAsString())
                && hasMetricPolicyRules() == other.hasMetricPolicyRules()
                && Objects.equals(metricPolicyRules(), other.metricPolicyRules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricPolicy").add("ContainerLevelMetrics", containerLevelMetricsAsString())
                .add("MetricPolicyRules", hasMetricPolicyRules() ? metricPolicyRules() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerLevelMetrics":
            return Optional.ofNullable(clazz.cast(containerLevelMetricsAsString()));
        case "MetricPolicyRules":
            return Optional.ofNullable(clazz.cast(metricPolicyRules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricPolicy, T> g) {
        return obj -> g.apply((MetricPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricPolicy> {
        /**
         * <p>
         * A setting to enable or disable metrics at the container level.
         * </p>
         * 
         * @param containerLevelMetrics
         *        A setting to enable or disable metrics at the container level.
         * @see ContainerLevelMetrics
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerLevelMetrics
         */
        Builder containerLevelMetrics(String containerLevelMetrics);

        /**
         * <p>
         * A setting to enable or disable metrics at the container level.
         * </p>
         * 
         * @param containerLevelMetrics
         *        A setting to enable or disable metrics at the container level.
         * @see ContainerLevelMetrics
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerLevelMetrics
         */
        Builder containerLevelMetrics(ContainerLevelMetrics containerLevelMetrics);

        /**
         * <p>
         * A parameter that holds an array of rules that enable metrics at the object level. This parameter is optional,
         * but if you choose to include it, you must also include at least one rule. By default, you can include up to
         * five rules. You can also <a
         * href="https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas"
         * >request a quota increase</a> to allow up to 300 rules per policy.
         * </p>
         * 
         * @param metricPolicyRules
         *        A parameter that holds an array of rules that enable metrics at the object level. This parameter is
         *        optional, but if you choose to include it, you must also include at least one rule. By default, you
         *        can include up to five rules. You can also <a href=
         *        "https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas"
         *        >request a quota increase</a> to allow up to 300 rules per policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricPolicyRules(Collection<MetricPolicyRule> metricPolicyRules);

        /**
         * <p>
         * A parameter that holds an array of rules that enable metrics at the object level. This parameter is optional,
         * but if you choose to include it, you must also include at least one rule. By default, you can include up to
         * five rules. You can also <a
         * href="https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas"
         * >request a quota increase</a> to allow up to 300 rules per policy.
         * </p>
         * 
         * @param metricPolicyRules
         *        A parameter that holds an array of rules that enable metrics at the object level. This parameter is
         *        optional, but if you choose to include it, you must also include at least one rule. By default, you
         *        can include up to five rules. You can also <a href=
         *        "https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas"
         *        >request a quota increase</a> to allow up to 300 rules per policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricPolicyRules(MetricPolicyRule... metricPolicyRules);

        /**
         * <p>
         * A parameter that holds an array of rules that enable metrics at the object level. This parameter is optional,
         * but if you choose to include it, you must also include at least one rule. By default, you can include up to
         * five rules. You can also <a
         * href="https://console.aws.amazon.com/servicequotas/home?region=us-east-1#!/services/mediastore/quotas"
         * >request a quota increase</a> to allow up to 300 rules per policy.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MetricPolicyRule>.Builder} avoiding the
         * need to create one manually via {@link List<MetricPolicyRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MetricPolicyRule>.Builder#build()} is called immediately and
         * its result is passed to {@link #metricPolicyRules(List<MetricPolicyRule>)}.
         * 
         * @param metricPolicyRules
         *        a consumer that will call methods on {@link List<MetricPolicyRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricPolicyRules(List<MetricPolicyRule>)
         */
        Builder metricPolicyRules(Consumer<MetricPolicyRule.Builder>... metricPolicyRules);
    }

    static final class BuilderImpl implements Builder {
        private String containerLevelMetrics;

        private List<MetricPolicyRule> metricPolicyRules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricPolicy model) {
            containerLevelMetrics(model.containerLevelMetrics);
            metricPolicyRules(model.metricPolicyRules);
        }

        public final String getContainerLevelMetrics() {
            return containerLevelMetrics;
        }

        @Override
        public final Builder containerLevelMetrics(String containerLevelMetrics) {
            this.containerLevelMetrics = containerLevelMetrics;
            return this;
        }

        @Override
        public final Builder containerLevelMetrics(ContainerLevelMetrics containerLevelMetrics) {
            this.containerLevelMetrics(containerLevelMetrics == null ? null : containerLevelMetrics.toString());
            return this;
        }

        public final void setContainerLevelMetrics(String containerLevelMetrics) {
            this.containerLevelMetrics = containerLevelMetrics;
        }

        public final Collection<MetricPolicyRule.Builder> getMetricPolicyRules() {
            if (metricPolicyRules instanceof SdkAutoConstructList) {
                return null;
            }
            return metricPolicyRules != null ? metricPolicyRules.stream().map(MetricPolicyRule::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder metricPolicyRules(Collection<MetricPolicyRule> metricPolicyRules) {
            this.metricPolicyRules = MetricPolicyRulesCopier.copy(metricPolicyRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricPolicyRules(MetricPolicyRule... metricPolicyRules) {
            metricPolicyRules(Arrays.asList(metricPolicyRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricPolicyRules(Consumer<MetricPolicyRule.Builder>... metricPolicyRules) {
            metricPolicyRules(Stream.of(metricPolicyRules).map(c -> MetricPolicyRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMetricPolicyRules(Collection<MetricPolicyRule.BuilderImpl> metricPolicyRules) {
            this.metricPolicyRules = MetricPolicyRulesCopier.copyFromBuilder(metricPolicyRules);
        }

        @Override
        public MetricPolicy build() {
            return new MetricPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
