/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastore;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediastore.model.ContainerInUseException;
import software.amazon.awssdk.services.mediastore.model.ContainerNotFoundException;
import software.amazon.awssdk.services.mediastore.model.CorsPolicyNotFoundException;
import software.amazon.awssdk.services.mediastore.model.CreateContainerRequest;
import software.amazon.awssdk.services.mediastore.model.CreateContainerResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteMetricPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteMetricPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DescribeContainerRequest;
import software.amazon.awssdk.services.mediastore.model.DescribeContainerResponse;
import software.amazon.awssdk.services.mediastore.model.GetContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetMetricPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetMetricPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediastore.model.LimitExceededException;
import software.amazon.awssdk.services.mediastore.model.ListContainersRequest;
import software.amazon.awssdk.services.mediastore.model.ListContainersResponse;
import software.amazon.awssdk.services.mediastore.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediastore.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediastore.model.MediaStoreException;
import software.amazon.awssdk.services.mediastore.model.MediaStoreRequest;
import software.amazon.awssdk.services.mediastore.model.PolicyNotFoundException;
import software.amazon.awssdk.services.mediastore.model.PutContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutMetricPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutMetricPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.StartAccessLoggingRequest;
import software.amazon.awssdk.services.mediastore.model.StartAccessLoggingResponse;
import software.amazon.awssdk.services.mediastore.model.StopAccessLoggingRequest;
import software.amazon.awssdk.services.mediastore.model.StopAccessLoggingResponse;
import software.amazon.awssdk.services.mediastore.model.TagResourceRequest;
import software.amazon.awssdk.services.mediastore.model.TagResourceResponse;
import software.amazon.awssdk.services.mediastore.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediastore.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediastore.paginators.ListContainersIterable;
import software.amazon.awssdk.services.mediastore.transform.CreateContainerRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteContainerPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteContainerRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteCorsPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteMetricPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DescribeContainerRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetContainerPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetCorsPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetMetricPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.ListContainersRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutContainerPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutCorsPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutMetricPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.StartAccessLoggingRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.StopAccessLoggingRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.UntagResourceRequestMarshaller;

/**
 * Internal implementation of {@link MediaStoreClient}.
 *
 * @see MediaStoreClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaStoreClient implements MediaStoreClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaStoreClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a storage container to hold objects. A container is similar to a bucket in the Amazon S3 service.
     * </p>
     *
     * @param createContainerRequest
     * @return Result of the CreateContainer operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws LimitExceededException
     *         A service limit has been exceeded.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.CreateContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/CreateContainer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateContainerResponse createContainer(CreateContainerRequest createContainerRequest) throws ContainerInUseException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateContainerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateContainerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateContainerRequest, CreateContainerResponse>()
                .withOperationName("CreateContainer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createContainerRequest)
                .withMarshaller(new CreateContainerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified container. Before you make a <code>DeleteContainer</code> request, delete any objects in
     * the container or in any folders in the container. You can delete only empty containers.
     * </p>
     *
     * @param deleteContainerRequest
     * @return Result of the DeleteContainer operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.DeleteContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteContainerResponse deleteContainer(DeleteContainerRequest deleteContainerRequest) throws ContainerInUseException,
            ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContainerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteContainerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteContainerRequest, DeleteContainerResponse>()
                .withOperationName("DeleteContainer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteContainerRequest)
                .withMarshaller(new DeleteContainerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the access policy that is associated with the specified container.
     * </p>
     *
     * @param deleteContainerPolicyRequest
     * @return Result of the DeleteContainerPolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws PolicyNotFoundException
     *         The policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.DeleteContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainerPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteContainerPolicyResponse deleteContainerPolicy(DeleteContainerPolicyRequest deleteContainerPolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, PolicyNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContainerPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteContainerPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteContainerPolicyRequest, DeleteContainerPolicyResponse>()
                .withOperationName("DeleteContainerPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteContainerPolicyRequest)
                .withMarshaller(new DeleteContainerPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:DeleteCorsPolicy</code> action.
     * The container owner has this permission by default and can grant this permission to others.
     * </p>
     *
     * @param deleteCorsPolicyRequest
     * @return Result of the DeleteCorsPolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws CorsPolicyNotFoundException
     *         The CORS policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.DeleteCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteCorsPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteCorsPolicyResponse deleteCorsPolicy(DeleteCorsPolicyRequest deleteCorsPolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, CorsPolicyNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCorsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCorsPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteCorsPolicyRequest, DeleteCorsPolicyResponse>()
                .withOperationName("DeleteCorsPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteCorsPolicyRequest)
                .withMarshaller(new DeleteCorsPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes an object lifecycle policy from a container. It takes up to 20 minutes for the change to take effect.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return Result of the DeleteLifecyclePolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws PolicyNotFoundException
     *         The policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.DeleteLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteLifecyclePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLifecyclePolicyResponse deleteLifecyclePolicy(DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, PolicyNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteLifecyclePolicyRequest, DeleteLifecyclePolicyResponse>()
                .withOperationName("DeleteLifecyclePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteLifecyclePolicyRequest)
                .withMarshaller(new DeleteLifecyclePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the metric policy that is associated with the specified container. If there is no metric policy
     * associated with the container, MediaStore doesn't send metrics to CloudWatch.
     * </p>
     *
     * @param deleteMetricPolicyRequest
     * @return Result of the DeleteMetricPolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws PolicyNotFoundException
     *         The policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.DeleteMetricPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteMetricPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteMetricPolicyResponse deleteMetricPolicy(DeleteMetricPolicyRequest deleteMetricPolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, PolicyNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMetricPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMetricPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteMetricPolicyRequest, DeleteMetricPolicyResponse>()
                .withOperationName("DeleteMetricPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteMetricPolicyRequest)
                .withMarshaller(new DeleteMetricPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the properties of the requested container. This request is commonly used to retrieve the endpoint of a
     * container. An endpoint is a value assigned by the service when a new container is created. A container's endpoint
     * does not change after it has been assigned. The <code>DescribeContainer</code> request returns a single
     * <code>Container</code> object based on <code>ContainerName</code>. To return all <code>Container</code> objects
     * that are associated with a specified AWS account, use <a>ListContainers</a>.
     * </p>
     *
     * @param describeContainerRequest
     * @return Result of the DescribeContainer operation returned by the service.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.DescribeContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DescribeContainer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeContainerResponse describeContainer(DescribeContainerRequest describeContainerRequest)
            throws ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContainerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeContainerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeContainerRequest, DescribeContainerResponse>()
                .withOperationName("DescribeContainer").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeContainerRequest)
                .withMarshaller(new DescribeContainerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the access policy for the specified container. For information about the data that is included in an
     * access policy, see the <a href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management
     * User Guide</a>.
     * </p>
     *
     * @param getContainerPolicyRequest
     * @return Result of the GetContainerPolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws PolicyNotFoundException
     *         The policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.GetContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetContainerPolicyResponse getContainerPolicy(GetContainerPolicyRequest getContainerPolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, PolicyNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContainerPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetContainerPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetContainerPolicyRequest, GetContainerPolicyResponse>()
                .withOperationName("GetContainerPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getContainerPolicyRequest)
                .withMarshaller(new GetContainerPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:GetCorsPolicy</code> action. By
     * default, the container owner has this permission and can grant it to others.
     * </p>
     *
     * @param getCorsPolicyRequest
     * @return Result of the GetCorsPolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws CorsPolicyNotFoundException
     *         The CORS policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.GetCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCorsPolicyResponse getCorsPolicy(GetCorsPolicyRequest getCorsPolicyRequest) throws ContainerInUseException,
            ContainerNotFoundException, CorsPolicyNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCorsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCorsPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetCorsPolicyRequest, GetCorsPolicyResponse>()
                .withOperationName("GetCorsPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getCorsPolicyRequest)
                .withMarshaller(new GetCorsPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the object lifecycle policy that is assigned to a container.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return Result of the GetLifecyclePolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws PolicyNotFoundException
     *         The policy that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.GetLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetLifecyclePolicyResponse getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, PolicyNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetLifecyclePolicyRequest, GetLifecyclePolicyResponse>()
                .withOperationName("GetLifecyclePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getLifecyclePolicyRequest)
                .withMarshaller(new GetLifecyclePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the metric policy for the specified container.
     * </p>
     *
     * @param getMetricPolicyRequest
     * @return Result of the GetMetricPolicy operation returned by the service.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws PolicyNotFoundException
     *         The policy that you specified in the request does not exist.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.GetMetricPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetMetricPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricPolicyResponse getMetricPolicy(GetMetricPolicyRequest getMetricPolicyRequest)
            throws ContainerNotFoundException, PolicyNotFoundException, ContainerInUseException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMetricPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMetricPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetMetricPolicyRequest, GetMetricPolicyResponse>()
                .withOperationName("GetMetricPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMetricPolicyRequest)
                .withMarshaller(new GetMetricPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     *
     * @param listContainersRequest
     * @return Result of the ListContainers operation returned by the service.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContainersResponse listContainers(ListContainersRequest listContainersRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContainersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContainersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListContainersRequest, ListContainersResponse>()
                .withOperationName("ListContainers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listContainersRequest)
                .withMarshaller(new ListContainersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersIterable responses = client.listContainersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediastore.paginators.ListContainersIterable responses = client
     *             .listContainersPaginator(request);
     *     for (software.amazon.awssdk.services.mediastore.model.ListContainersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersIterable responses = client.listContainersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation.</b>
     * </p>
     *
     * @param listContainersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContainersIterable listContainersPaginator(ListContainersRequest listContainersRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreException {
        return new ListContainersIterable(this, applyPaginatorUserAgent(listContainersRequest));
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified container.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ContainerInUseException, ContainerNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an access policy for the specified container to restrict the users and clients that can access it. For
     * information about the data that is included in an access policy, see the <a
     * href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management User Guide</a>.
     * </p>
     * <p>
     * For this release of the REST API, you can create only one policy for a container. If you enter
     * <code>PutContainerPolicy</code> twice, the second command modifies the existing policy.
     * </p>
     *
     * @param putContainerPolicyRequest
     * @return Result of the PutContainerPolicy operation returned by the service.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.PutContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutContainerPolicyResponse putContainerPolicy(PutContainerPolicyRequest putContainerPolicyRequest)
            throws ContainerNotFoundException, ContainerInUseException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutContainerPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutContainerPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutContainerPolicyRequest, PutContainerPolicyResponse>()
                .withOperationName("PutContainerPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putContainerPolicyRequest)
                .withMarshaller(new PutContainerPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the cross-origin resource sharing (CORS) configuration on a container so that the container can service
     * cross-origin requests. For example, you might want to enable a request whose origin is http://www.example.com to
     * access your AWS Elemental MediaStore container at my.example.container.com by using the browser's XMLHttpRequest
     * capability.
     * </p>
     * <p>
     * To enable CORS on a container, you attach a CORS policy to the container. In the CORS policy, you configure rules
     * that identify origins and the HTTP methods that can be executed on your container. The policy can contain up to
     * 398,000 characters. You can add up to 100 rules to a CORS policy. If more than one rule applies, the service uses
     * the first applicable rule listed.
     * </p>
     * <p>
     * To learn more about CORS, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/cors-policy.html">Cross-Origin Resource Sharing (CORS) in
     * AWS Elemental MediaStore</a>.
     * </p>
     *
     * @param putCorsPolicyRequest
     * @return Result of the PutCorsPolicy operation returned by the service.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.PutCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutCorsPolicyResponse putCorsPolicy(PutCorsPolicyRequest putCorsPolicyRequest) throws ContainerNotFoundException,
            ContainerInUseException, InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutCorsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutCorsPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutCorsPolicyRequest, PutCorsPolicyResponse>()
                .withOperationName("PutCorsPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putCorsPolicyRequest)
                .withMarshaller(new PutCorsPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Writes an object lifecycle policy to a container. If the container already has an object lifecycle policy, the
     * service replaces the existing policy with the new policy. It takes up to 20 minutes for the change to take
     * effect.
     * </p>
     * <p>
     * For information about how to construct an object lifecycle policy, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/policies-object-lifecycle-components.html">Components of
     * an Object Lifecycle Policy</a>.
     * </p>
     *
     * @param putLifecyclePolicyRequest
     * @return Result of the PutLifecyclePolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.PutLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutLifecyclePolicyResponse putLifecyclePolicy(PutLifecyclePolicyRequest putLifecyclePolicyRequest)
            throws ContainerInUseException, ContainerNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutLifecyclePolicyRequest, PutLifecyclePolicyResponse>()
                .withOperationName("PutLifecyclePolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putLifecyclePolicyRequest)
                .withMarshaller(new PutLifecyclePolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The metric policy that you want to add to the container. A metric policy allows AWS Elemental MediaStore to send
     * metrics to Amazon CloudWatch. It takes up to 20 minutes for the new policy to take effect.
     * </p>
     *
     * @param putMetricPolicyRequest
     * @return Result of the PutMetricPolicy operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.PutMetricPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutMetricPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricPolicyResponse putMetricPolicy(PutMetricPolicyRequest putMetricPolicyRequest) throws ContainerInUseException,
            ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutMetricPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutMetricPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutMetricPolicyRequest, PutMetricPolicyResponse>()
                .withOperationName("PutMetricPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putMetricPolicyRequest)
                .withMarshaller(new PutMetricPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts access logging on the specified container. When you enable access logging on a container, MediaStore
     * delivers access logs for objects stored in that container to Amazon CloudWatch Logs.
     * </p>
     *
     * @param startAccessLoggingRequest
     * @return Result of the StartAccessLogging operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.StartAccessLogging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StartAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartAccessLoggingResponse startAccessLogging(StartAccessLoggingRequest startAccessLoggingRequest)
            throws ContainerInUseException, ContainerNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartAccessLoggingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartAccessLoggingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartAccessLoggingRequest, StartAccessLoggingResponse>()
                .withOperationName("StartAccessLogging").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startAccessLoggingRequest)
                .withMarshaller(new StartAccessLoggingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stops access logging on the specified container. When you stop access logging on a container, MediaStore stops
     * sending access logs to Amazon CloudWatch Logs. These access logs are not saved and are not retrievable.
     * </p>
     *
     * @param stopAccessLoggingRequest
     * @return Result of the StopAccessLogging operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.StopAccessLogging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StopAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopAccessLoggingResponse stopAccessLogging(StopAccessLoggingRequest stopAccessLoggingRequest)
            throws ContainerInUseException, ContainerNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopAccessLoggingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopAccessLoggingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopAccessLoggingRequest, StopAccessLoggingResponse>()
                .withOperationName("StopAccessLogging").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopAccessLoggingRequest)
                .withMarshaller(new StopAccessLoggingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds tags to the specified AWS Elemental MediaStore container. Tags are key:value pairs that you can associate
     * with AWS resources. For example, the tag key might be "customer" and the tag value might be "companyA." You can
     * specify one or more tags to add to each container. You can add up to 50 tags to each container. For more
     * information about tagging, including naming and usage conventions, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ContainerInUseException,
            ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes tags from the specified container. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ContainerInUseException
     *         The container that you specified in the request already exists or is being updated.
     * @throws ContainerNotFoundException
     *         The container that you specified in the request does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ContainerInUseException,
            ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaStoreException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContainerInUseException")
                                .exceptionBuilderSupplier(ContainerInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CorsPolicyNotFoundException")
                                .exceptionBuilderSupplier(CorsPolicyNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContainerNotFoundException")
                                .exceptionBuilderSupplier(ContainerNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyNotFoundException")
                                .exceptionBuilderSupplier(PolicyNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaStoreRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
