/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastore.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateContainerRequest extends MediaStoreRequest implements
        ToCopyableBuilder<CreateContainerRequest.Builder, CreateContainerRequest> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateContainerRequest::containerName)).setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerName").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateContainerRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_NAME_FIELD,
            TAGS_FIELD));

    private final String containerName;

    private final List<Tag> tags;

    private CreateContainerRequest(BuilderImpl builder) {
        super(builder);
        this.containerName = builder.containerName;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name for the container. The name must be from 1 to 255 characters. Container names must be unique to your AWS
     * account within a specific region. As an example, you could create a container named <code>movies</code> in every
     * region, as long as you don’t have an existing container with that name.
     * </p>
     * 
     * @return The name for the container. The name must be from 1 to 255 characters. Container names must be unique to
     *         your AWS account within a specific region. As an example, you could create a container named
     *         <code>movies</code> in every region, as long as you don’t have an existing container with that name.
     */
    public String containerName() {
        return containerName;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of key:value pairs that you define. These values can be anything that you want. Typically, the tag key
     * represents a category (such as "environment") and the tag value represents a specific value within that category
     * (such as "test," "development," or "production"). You can add up to 50 tags to each container. For more
     * information about tagging, including naming and usage conventions, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of key:value pairs that you define. These values can be anything that you want. Typically, the
     *         tag key represents a category (such as "environment") and the tag value represents a specific value
     *         within that category (such as "test," "development," or "production"). You can add up to 50 tags to each
     *         container. For more information about tagging, including naming and usage conventions, see <a
     *         href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateContainerRequest)) {
            return false;
        }
        CreateContainerRequest other = (CreateContainerRequest) obj;
        return Objects.equals(containerName(), other.containerName()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateContainerRequest").add("ContainerName", containerName()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateContainerRequest, T> g) {
        return obj -> g.apply((CreateContainerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MediaStoreRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateContainerRequest> {
        /**
         * <p>
         * The name for the container. The name must be from 1 to 255 characters. Container names must be unique to your
         * AWS account within a specific region. As an example, you could create a container named <code>movies</code>
         * in every region, as long as you don’t have an existing container with that name.
         * </p>
         * 
         * @param containerName
         *        The name for the container. The name must be from 1 to 255 characters. Container names must be unique
         *        to your AWS account within a specific region. As an example, you could create a container named
         *        <code>movies</code> in every region, as long as you don’t have an existing container with that name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * An array of key:value pairs that you define. These values can be anything that you want. Typically, the tag
         * key represents a category (such as "environment") and the tag value represents a specific value within that
         * category (such as "test," "development," or "production"). You can add up to 50 tags to each container. For
         * more information about tagging, including naming and usage conventions, see <a
         * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
         * </p>
         * 
         * @param tags
         *        An array of key:value pairs that you define. These values can be anything that you want. Typically,
         *        the tag key represents a category (such as "environment") and the tag value represents a specific
         *        value within that category (such as "test," "development," or "production"). You can add up to 50 tags
         *        to each container. For more information about tagging, including naming and usage conventions, see <a
         *        href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in
         *        MediaStore</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * An array of key:value pairs that you define. These values can be anything that you want. Typically, the tag
         * key represents a category (such as "environment") and the tag value represents a specific value within that
         * category (such as "test," "development," or "production"). You can add up to 50 tags to each container. For
         * more information about tagging, including naming and usage conventions, see <a
         * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
         * </p>
         * 
         * @param tags
         *        An array of key:value pairs that you define. These values can be anything that you want. Typically,
         *        the tag key represents a category (such as "environment") and the tag value represents a specific
         *        value within that category (such as "test," "development," or "production"). You can add up to 50 tags
         *        to each container. For more information about tagging, including naming and usage conventions, see <a
         *        href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in
         *        MediaStore</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * An array of key:value pairs that you define. These values can be anything that you want. Typically, the tag
         * key represents a category (such as "environment") and the tag value represents a specific value within that
         * category (such as "test," "development," or "production"). You can add up to 50 tags to each container. For
         * more information about tagging, including naming and usage conventions, see <a
         * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MediaStoreRequest.BuilderImpl implements Builder {
        private String containerName;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateContainerRequest model) {
            super(model);
            containerName(model.containerName);
            tags(model.tags);
        }

        public final String getContainerName() {
            return containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateContainerRequest build() {
            return new CreateContainerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
