/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastore;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.mediastore.model.CreateContainerRequest;
import software.amazon.awssdk.services.mediastore.model.CreateContainerResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DescribeContainerRequest;
import software.amazon.awssdk.services.mediastore.model.DescribeContainerResponse;
import software.amazon.awssdk.services.mediastore.model.GetContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.ListContainersRequest;
import software.amazon.awssdk.services.mediastore.model.ListContainersResponse;
import software.amazon.awssdk.services.mediastore.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediastore.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediastore.model.PutContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.StartAccessLoggingRequest;
import software.amazon.awssdk.services.mediastore.model.StartAccessLoggingResponse;
import software.amazon.awssdk.services.mediastore.model.StopAccessLoggingRequest;
import software.amazon.awssdk.services.mediastore.model.StopAccessLoggingResponse;
import software.amazon.awssdk.services.mediastore.model.TagResourceRequest;
import software.amazon.awssdk.services.mediastore.model.TagResourceResponse;
import software.amazon.awssdk.services.mediastore.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediastore.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher;

/**
 * Service client for accessing MediaStore asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * An AWS Elemental MediaStore container is a namespace that holds folders and objects. You use a container endpoint to
 * create, read, and delete objects.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface MediaStoreAsyncClient extends SdkClient {
    String SERVICE_NAME = "mediastore";

    /**
     * Create a {@link MediaStoreAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MediaStoreAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MediaStoreAsyncClient}.
     */
    static MediaStoreAsyncClientBuilder builder() {
        return new DefaultMediaStoreAsyncClientBuilder();
    }

    /**
     * <p>
     * Creates a storage container to hold objects. A container is similar to a bucket in the Amazon S3 service.
     * </p>
     *
     * @param createContainerRequest
     * @return A Java Future containing the result of the CreateContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>LimitExceededException A service limit has been exceeded.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.CreateContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/CreateContainer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateContainerResponse> createContainer(CreateContainerRequest createContainerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a storage container to hold objects. A container is similar to a bucket in the Amazon S3 service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateContainerRequest.Builder} avoiding the need
     * to create one manually via {@link CreateContainerRequest#builder()}
     * </p>
     *
     * @param createContainerRequest
     *        A {@link Consumer} that will call methods on {@link CreateContainerInput.Builder} to create a request.
     * @return A Java Future containing the result of the CreateContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>LimitExceededException A service limit has been exceeded.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.CreateContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/CreateContainer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateContainerResponse> createContainer(
            Consumer<CreateContainerRequest.Builder> createContainerRequest) {
        return createContainer(CreateContainerRequest.builder().applyMutation(createContainerRequest).build());
    }

    /**
     * <p>
     * Deletes the specified container. Before you make a <code>DeleteContainer</code> request, delete any objects in
     * the container or in any folders in the container. You can delete only empty containers.
     * </p>
     *
     * @param deleteContainerRequest
     * @return A Java Future containing the result of the DeleteContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteContainerResponse> deleteContainer(DeleteContainerRequest deleteContainerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified container. Before you make a <code>DeleteContainer</code> request, delete any objects in
     * the container or in any folders in the container. You can delete only empty containers.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteContainerRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteContainerRequest#builder()}
     * </p>
     *
     * @param deleteContainerRequest
     *        A {@link Consumer} that will call methods on {@link DeleteContainerInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteContainerResponse> deleteContainer(
            Consumer<DeleteContainerRequest.Builder> deleteContainerRequest) {
        return deleteContainer(DeleteContainerRequest.builder().applyMutation(deleteContainerRequest).build());
    }

    /**
     * <p>
     * Deletes the access policy that is associated with the specified container.
     * </p>
     *
     * @param deleteContainerPolicyRequest
     * @return A Java Future containing the result of the DeleteContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainerPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteContainerPolicyResponse> deleteContainerPolicy(
            DeleteContainerPolicyRequest deleteContainerPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the access policy that is associated with the specified container.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteContainerPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteContainerPolicyRequest#builder()}
     * </p>
     *
     * @param deleteContainerPolicyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteContainerPolicyInput.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainerPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteContainerPolicyResponse> deleteContainerPolicy(
            Consumer<DeleteContainerPolicyRequest.Builder> deleteContainerPolicyRequest) {
        return deleteContainerPolicy(DeleteContainerPolicyRequest.builder().applyMutation(deleteContainerPolicyRequest).build());
    }

    /**
     * <p>
     * Deletes the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:DeleteCorsPolicy</code> action.
     * The container owner has this permission by default and can grant this permission to others.
     * </p>
     *
     * @param deleteCorsPolicyRequest
     * @return A Java Future containing the result of the DeleteCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>CorsPolicyNotFoundException The CORS policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteCorsPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteCorsPolicyResponse> deleteCorsPolicy(DeleteCorsPolicyRequest deleteCorsPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:DeleteCorsPolicy</code> action.
     * The container owner has this permission by default and can grant this permission to others.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteCorsPolicyRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteCorsPolicyRequest#builder()}
     * </p>
     *
     * @param deleteCorsPolicyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteCorsPolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>CorsPolicyNotFoundException The CORS policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteCorsPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteCorsPolicyResponse> deleteCorsPolicy(
            Consumer<DeleteCorsPolicyRequest.Builder> deleteCorsPolicyRequest) {
        return deleteCorsPolicy(DeleteCorsPolicyRequest.builder().applyMutation(deleteCorsPolicyRequest).build());
    }

    /**
     * <p>
     * Removes an object lifecycle policy from a container. It takes up to 20 minutes for the change to take effect.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return A Java Future containing the result of the DeleteLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteLifecyclePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteLifecyclePolicyResponse> deleteLifecyclePolicy(
            DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes an object lifecycle policy from a container. It takes up to 20 minutes for the change to take effect.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteLifecyclePolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteLifecyclePolicyRequest#builder()}
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteLifecyclePolicyInput.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteLifecyclePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteLifecyclePolicyResponse> deleteLifecyclePolicy(
            Consumer<DeleteLifecyclePolicyRequest.Builder> deleteLifecyclePolicyRequest) {
        return deleteLifecyclePolicy(DeleteLifecyclePolicyRequest.builder().applyMutation(deleteLifecyclePolicyRequest).build());
    }

    /**
     * <p>
     * Retrieves the properties of the requested container. This request is commonly used to retrieve the endpoint of a
     * container. An endpoint is a value assigned by the service when a new container is created. A container's endpoint
     * does not change after it has been assigned. The <code>DescribeContainer</code> request returns a single
     * <code>Container</code> object based on <code>ContainerName</code>. To return all <code>Container</code> objects
     * that are associated with a specified AWS account, use <a>ListContainers</a>.
     * </p>
     *
     * @param describeContainerRequest
     * @return A Java Future containing the result of the DescribeContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DescribeContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DescribeContainer" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeContainerResponse> describeContainer(DescribeContainerRequest describeContainerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the properties of the requested container. This request is commonly used to retrieve the endpoint of a
     * container. An endpoint is a value assigned by the service when a new container is created. A container's endpoint
     * does not change after it has been assigned. The <code>DescribeContainer</code> request returns a single
     * <code>Container</code> object based on <code>ContainerName</code>. To return all <code>Container</code> objects
     * that are associated with a specified AWS account, use <a>ListContainers</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeContainerRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeContainerRequest#builder()}
     * </p>
     *
     * @param describeContainerRequest
     *        A {@link Consumer} that will call methods on {@link DescribeContainerInput.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DescribeContainer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DescribeContainer" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeContainerResponse> describeContainer(
            Consumer<DescribeContainerRequest.Builder> describeContainerRequest) {
        return describeContainer(DescribeContainerRequest.builder().applyMutation(describeContainerRequest).build());
    }

    /**
     * <p>
     * Retrieves the access policy for the specified container. For information about the data that is included in an
     * access policy, see the <a href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management
     * User Guide</a>.
     * </p>
     *
     * @param getContainerPolicyRequest
     * @return A Java Future containing the result of the GetContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetContainerPolicyResponse> getContainerPolicy(GetContainerPolicyRequest getContainerPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the access policy for the specified container. For information about the data that is included in an
     * access policy, see the <a href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management
     * User Guide</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetContainerPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link GetContainerPolicyRequest#builder()}
     * </p>
     *
     * @param getContainerPolicyRequest
     *        A {@link Consumer} that will call methods on {@link GetContainerPolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetContainerPolicyResponse> getContainerPolicy(
            Consumer<GetContainerPolicyRequest.Builder> getContainerPolicyRequest) {
        return getContainerPolicy(GetContainerPolicyRequest.builder().applyMutation(getContainerPolicyRequest).build());
    }

    /**
     * <p>
     * Returns the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:GetCorsPolicy</code> action. By
     * default, the container owner has this permission and can grant it to others.
     * </p>
     *
     * @param getCorsPolicyRequest
     * @return A Java Future containing the result of the GetCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>CorsPolicyNotFoundException The CORS policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetCorsPolicyResponse> getCorsPolicy(GetCorsPolicyRequest getCorsPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:GetCorsPolicy</code> action. By
     * default, the container owner has this permission and can grant it to others.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetCorsPolicyRequest.Builder} avoiding the need to
     * create one manually via {@link GetCorsPolicyRequest#builder()}
     * </p>
     *
     * @param getCorsPolicyRequest
     *        A {@link Consumer} that will call methods on {@link GetCorsPolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>CorsPolicyNotFoundException The CORS policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetCorsPolicyResponse> getCorsPolicy(Consumer<GetCorsPolicyRequest.Builder> getCorsPolicyRequest) {
        return getCorsPolicy(GetCorsPolicyRequest.builder().applyMutation(getCorsPolicyRequest).build());
    }

    /**
     * <p>
     * Retrieves the object lifecycle policy that is assigned to a container.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return A Java Future containing the result of the GetLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetLifecyclePolicyResponse> getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the object lifecycle policy that is assigned to a container.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLifecyclePolicyRequest.Builder} avoiding the
     * need to create one manually via {@link GetLifecyclePolicyRequest#builder()}
     * </p>
     *
     * @param getLifecyclePolicyRequest
     *        A {@link Consumer} that will call methods on {@link GetLifecyclePolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetLifecyclePolicyResponse> getLifecyclePolicy(
            Consumer<GetLifecyclePolicyRequest.Builder> getLifecyclePolicyRequest) {
        return getLifecyclePolicy(GetLifecyclePolicyRequest.builder().applyMutation(getLifecyclePolicyRequest).build());
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     *
     * @param listContainersRequest
     * @return A Java Future containing the result of the ListContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListContainersResponse> listContainers(ListContainersRequest listContainersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListContainersRequest.Builder} avoiding the need to
     * create one manually via {@link ListContainersRequest#builder()}
     * </p>
     *
     * @param listContainersRequest
     *        A {@link Consumer} that will call methods on {@link ListContainersInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListContainersResponse> listContainers(Consumer<ListContainersRequest.Builder> listContainersRequest) {
        return listContainers(ListContainersRequest.builder().applyMutation(listContainersRequest).build());
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     *
     * @return A Java Future containing the result of the ListContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListContainersResponse> listContainers() {
        return listContainers(ListContainersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher publisher = client.listContainersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher publisher = client.listContainersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediastore.model.ListContainersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediastore.model.ListContainersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListContainersPublisher listContainersPaginator() {
        return listContainersPaginator(ListContainersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher publisher = client.listContainersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher publisher = client.listContainersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediastore.model.ListContainersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediastore.model.ListContainersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation.</b>
     * </p>
     *
     * @param listContainersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListContainersPublisher listContainersPaginator(ListContainersRequest listContainersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher publisher = client.listContainersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediastore.paginators.ListContainersPublisher publisher = client.listContainersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediastore.model.ListContainersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediastore.model.ListContainersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContainers(software.amazon.awssdk.services.mediastore.model.ListContainersRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListContainersRequest.Builder} avoiding the need to
     * create one manually via {@link ListContainersRequest#builder()}
     * </p>
     *
     * @param listContainersRequest
     *        A {@link Consumer} that will call methods on {@link ListContainersInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListContainersPublisher listContainersPaginator(Consumer<ListContainersRequest.Builder> listContainersRequest) {
        return listContainersPaginator(ListContainersRequest.builder().applyMutation(listContainersRequest).build());
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified container.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified container.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Creates an access policy for the specified container to restrict the users and clients that can access it. For
     * information about the data that is included in an access policy, see the <a
     * href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management User Guide</a>.
     * </p>
     * <p>
     * For this release of the REST API, you can create only one policy for a container. If you enter
     * <code>PutContainerPolicy</code> twice, the second command modifies the existing policy.
     * </p>
     *
     * @param putContainerPolicyRequest
     * @return A Java Future containing the result of the PutContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<PutContainerPolicyResponse> putContainerPolicy(PutContainerPolicyRequest putContainerPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an access policy for the specified container to restrict the users and clients that can access it. For
     * information about the data that is included in an access policy, see the <a
     * href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management User Guide</a>.
     * </p>
     * <p>
     * For this release of the REST API, you can create only one policy for a container. If you enter
     * <code>PutContainerPolicy</code> twice, the second command modifies the existing policy.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutContainerPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link PutContainerPolicyRequest#builder()}
     * </p>
     *
     * @param putContainerPolicyRequest
     *        A {@link Consumer} that will call methods on {@link PutContainerPolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the PutContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutContainerPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<PutContainerPolicyResponse> putContainerPolicy(
            Consumer<PutContainerPolicyRequest.Builder> putContainerPolicyRequest) {
        return putContainerPolicy(PutContainerPolicyRequest.builder().applyMutation(putContainerPolicyRequest).build());
    }

    /**
     * <p>
     * Sets the cross-origin resource sharing (CORS) configuration on a container so that the container can service
     * cross-origin requests. For example, you might want to enable a request whose origin is http://www.example.com to
     * access your AWS Elemental MediaStore container at my.example.container.com by using the browser's XMLHttpRequest
     * capability.
     * </p>
     * <p>
     * To enable CORS on a container, you attach a CORS policy to the container. In the CORS policy, you configure rules
     * that identify origins and the HTTP methods that can be executed on your container. The policy can contain up to
     * 398,000 characters. You can add up to 100 rules to a CORS policy. If more than one rule applies, the service uses
     * the first applicable rule listed.
     * </p>
     * <p>
     * To learn more about CORS, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/cors-policy.html">Cross-Origin Resource Sharing (CORS) in
     * AWS Elemental MediaStore</a>.
     * </p>
     *
     * @param putCorsPolicyRequest
     * @return A Java Future containing the result of the PutCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutCorsPolicyResponse> putCorsPolicy(PutCorsPolicyRequest putCorsPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sets the cross-origin resource sharing (CORS) configuration on a container so that the container can service
     * cross-origin requests. For example, you might want to enable a request whose origin is http://www.example.com to
     * access your AWS Elemental MediaStore container at my.example.container.com by using the browser's XMLHttpRequest
     * capability.
     * </p>
     * <p>
     * To enable CORS on a container, you attach a CORS policy to the container. In the CORS policy, you configure rules
     * that identify origins and the HTTP methods that can be executed on your container. The policy can contain up to
     * 398,000 characters. You can add up to 100 rules to a CORS policy. If more than one rule applies, the service uses
     * the first applicable rule listed.
     * </p>
     * <p>
     * To learn more about CORS, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/cors-policy.html">Cross-Origin Resource Sharing (CORS) in
     * AWS Elemental MediaStore</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutCorsPolicyRequest.Builder} avoiding the need to
     * create one manually via {@link PutCorsPolicyRequest#builder()}
     * </p>
     *
     * @param putCorsPolicyRequest
     *        A {@link Consumer} that will call methods on {@link PutCorsPolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the PutCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutCorsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutCorsPolicyResponse> putCorsPolicy(Consumer<PutCorsPolicyRequest.Builder> putCorsPolicyRequest) {
        return putCorsPolicy(PutCorsPolicyRequest.builder().applyMutation(putCorsPolicyRequest).build());
    }

    /**
     * <p>
     * Writes an object lifecycle policy to a container. If the container already has an object lifecycle policy, the
     * service replaces the existing policy with the new policy. It takes up to 20 minutes for the change to take
     * effect.
     * </p>
     * <p>
     * For information about how to construct an object lifecycle policy, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/policies-object-lifecycle-components.html">Components of
     * an Object Lifecycle Policy</a>.
     * </p>
     *
     * @param putLifecyclePolicyRequest
     * @return A Java Future containing the result of the PutLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<PutLifecyclePolicyResponse> putLifecyclePolicy(PutLifecyclePolicyRequest putLifecyclePolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Writes an object lifecycle policy to a container. If the container already has an object lifecycle policy, the
     * service replaces the existing policy with the new policy. It takes up to 20 minutes for the change to take
     * effect.
     * </p>
     * <p>
     * For information about how to construct an object lifecycle policy, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/policies-object-lifecycle-components.html">Components of
     * an Object Lifecycle Policy</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutLifecyclePolicyRequest.Builder} avoiding the
     * need to create one manually via {@link PutLifecyclePolicyRequest#builder()}
     * </p>
     *
     * @param putLifecyclePolicyRequest
     *        A {@link Consumer} that will call methods on {@link PutLifecyclePolicyInput.Builder} to create a request.
     * @return A Java Future containing the result of the PutLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<PutLifecyclePolicyResponse> putLifecyclePolicy(
            Consumer<PutLifecyclePolicyRequest.Builder> putLifecyclePolicyRequest) {
        return putLifecyclePolicy(PutLifecyclePolicyRequest.builder().applyMutation(putLifecyclePolicyRequest).build());
    }

    /**
     * <p>
     * Starts access logging on the specified container. When you enable access logging on a container, MediaStore
     * delivers access logs for objects stored in that container to Amazon CloudWatch Logs.
     * </p>
     *
     * @param startAccessLoggingRequest
     * @return A Java Future containing the result of the StartAccessLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.StartAccessLogging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StartAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<StartAccessLoggingResponse> startAccessLogging(StartAccessLoggingRequest startAccessLoggingRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts access logging on the specified container. When you enable access logging on a container, MediaStore
     * delivers access logs for objects stored in that container to Amazon CloudWatch Logs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartAccessLoggingRequest.Builder} avoiding the
     * need to create one manually via {@link StartAccessLoggingRequest#builder()}
     * </p>
     *
     * @param startAccessLoggingRequest
     *        A {@link Consumer} that will call methods on {@link StartAccessLoggingInput.Builder} to create a request.
     * @return A Java Future containing the result of the StartAccessLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.StartAccessLogging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StartAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<StartAccessLoggingResponse> startAccessLogging(
            Consumer<StartAccessLoggingRequest.Builder> startAccessLoggingRequest) {
        return startAccessLogging(StartAccessLoggingRequest.builder().applyMutation(startAccessLoggingRequest).build());
    }

    /**
     * <p>
     * Stops access logging on the specified container. When you stop access logging on a container, MediaStore stops
     * sending access logs to Amazon CloudWatch Logs. These access logs are not saved and are not retrievable.
     * </p>
     *
     * @param stopAccessLoggingRequest
     * @return A Java Future containing the result of the StopAccessLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.StopAccessLogging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StopAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<StopAccessLoggingResponse> stopAccessLogging(StopAccessLoggingRequest stopAccessLoggingRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Stops access logging on the specified container. When you stop access logging on a container, MediaStore stops
     * sending access logs to Amazon CloudWatch Logs. These access logs are not saved and are not retrievable.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopAccessLoggingRequest.Builder} avoiding the need
     * to create one manually via {@link StopAccessLoggingRequest#builder()}
     * </p>
     *
     * @param stopAccessLoggingRequest
     *        A {@link Consumer} that will call methods on {@link StopAccessLoggingInput.Builder} to create a request.
     * @return A Java Future containing the result of the StopAccessLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.StopAccessLogging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StopAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<StopAccessLoggingResponse> stopAccessLogging(
            Consumer<StopAccessLoggingRequest.Builder> stopAccessLoggingRequest) {
        return stopAccessLogging(StopAccessLoggingRequest.builder().applyMutation(stopAccessLoggingRequest).build());
    }

    /**
     * <p>
     * Adds tags to the specified AWS Elemental MediaStore container. Tags are key:value pairs that you can associate
     * with AWS resources. For example, the tag key might be "customer" and the tag value might be "companyA." You can
     * specify one or more tags to add to each container. You can add up to 50 tags to each container. For more
     * information about tagging, including naming and usage conventions, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds tags to the specified AWS Elemental MediaStore container. Tags are key:value pairs that you can associate
     * with AWS resources. For example, the tag key might be "customer" and the tag value might be "companyA." You can
     * specify one or more tags to add to each container. You can add up to 50 tags to each container. For more
     * information about tagging, including naming and usage conventions, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from the specified container. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from the specified container. You can specify one or more tags to remove.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }
}
