/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagevod;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ForbiddenException;
import software.amazon.awssdk.services.mediapackagevod.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.MediaPackageVodException;
import software.amazon.awssdk.services.mediapackagevod.model.MediaPackageVodRequest;
import software.amazon.awssdk.services.mediapackagevod.model.NotFoundException;
import software.amazon.awssdk.services.mediapackagevod.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediapackagevod.model.TagResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.TagResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediapackagevod.model.UnprocessableEntityException;
import software.amazon.awssdk.services.mediapackagevod.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.UpdatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.UpdatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsPublisher;
import software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsPublisher;
import software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsPublisher;
import software.amazon.awssdk.services.mediapackagevod.transform.CreateAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeleteAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribeAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListAssetsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingGroupsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.UpdatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaPackageVodAsyncClient}.
 *
 * @see MediaPackageVodAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageVodAsyncClient implements MediaPackageVodAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaPackageVodAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageVodAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Creates a new MediaPackage VOD Asset resource.
     *
     * @param createAssetRequest
     *        A new MediaPackage VOD Asset configuration.
     * @return A Java Future containing the result of the CreateAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.CreateAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreateAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAssetResponse> createAsset(CreateAssetRequest createAssetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAssetRequest, CreateAssetResponse>()
                            .withOperationName("CreateAsset").withMarshaller(new CreateAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAssetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAssetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new MediaPackage VOD PackagingConfiguration resource.
     *
     * @param createPackagingConfigurationRequest
     *        A new MediaPackage VOD PackagingConfiguration resource configuration.
     * @return A Java Future containing the result of the CreatePackagingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.CreatePackagingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePackagingConfigurationResponse> createPackagingConfiguration(
            CreatePackagingConfigurationRequest createPackagingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackagingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackagingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePackagingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackagingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackagingConfigurationRequest, CreatePackagingConfigurationResponse>()
                            .withOperationName("CreatePackagingConfiguration")
                            .withMarshaller(new CreatePackagingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPackagingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createPackagingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreatePackagingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new MediaPackage VOD PackagingGroup resource.
     *
     * @param createPackagingGroupRequest
     *        A new MediaPackage VOD PackagingGroup resource configuration.
     * @return A Java Future containing the result of the CreatePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.CreatePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePackagingGroupResponse> createPackagingGroup(
            CreatePackagingGroupRequest createPackagingGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackagingGroupRequest, CreatePackagingGroupResponse>()
                            .withOperationName("CreatePackagingGroup")
                            .withMarshaller(new CreatePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPackagingGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createPackagingGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreatePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an existing MediaPackage VOD Asset resource.
     *
     * @param deleteAssetRequest
     * @return A Java Future containing the result of the DeleteAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DeleteAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeleteAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAssetResponse> deleteAsset(DeleteAssetRequest deleteAssetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAssetRequest, DeleteAssetResponse>()
                            .withOperationName("DeleteAsset").withMarshaller(new DeleteAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAssetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAssetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param deletePackagingConfigurationRequest
     * @return A Java Future containing the result of the DeletePackagingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DeletePackagingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePackagingConfigurationResponse> deletePackagingConfiguration(
            DeletePackagingConfigurationRequest deletePackagingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackagingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackagingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePackagingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackagingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackagingConfigurationRequest, DeletePackagingConfigurationResponse>()
                            .withOperationName("DeletePackagingConfiguration")
                            .withMarshaller(new DeletePackagingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePackagingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePackagingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeletePackagingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes a MediaPackage VOD PackagingGroup resource.
     *
     * @param deletePackagingGroupRequest
     * @return A Java Future containing the result of the DeletePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DeletePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePackagingGroupResponse> deletePackagingGroup(
            DeletePackagingGroupRequest deletePackagingGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackagingGroupRequest, DeletePackagingGroupResponse>()
                            .withOperationName("DeletePackagingGroup")
                            .withMarshaller(new DeletePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePackagingGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePackagingGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeletePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a description of a MediaPackage VOD Asset resource.
     *
     * @param describeAssetRequest
     * @return A Java Future containing the result of the DescribeAsset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DescribeAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribeAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAssetResponse> describeAsset(DescribeAssetRequest describeAssetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAssetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAssetRequest, DescribeAssetResponse>()
                            .withOperationName("DescribeAsset")
                            .withMarshaller(new DescribeAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAssetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeAssetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeAssetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param describePackagingConfigurationRequest
     * @return A Java Future containing the result of the DescribePackagingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DescribePackagingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackagingConfigurationResponse> describePackagingConfiguration(
            DescribePackagingConfigurationRequest describePackagingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePackagingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackagingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackagingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackagingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackagingConfigurationRequest, DescribePackagingConfigurationResponse>()
                            .withOperationName("DescribePackagingConfiguration")
                            .withMarshaller(new DescribePackagingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackagingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describePackagingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribePackagingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingGroup resource.
     *
     * @param describePackagingGroupRequest
     * @return A Java Future containing the result of the DescribePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.DescribePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackagingGroupResponse> describePackagingGroup(
            DescribePackagingGroupRequest describePackagingGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackagingGroupRequest, DescribePackagingGroupResponse>()
                            .withOperationName("DescribePackagingGroup")
                            .withMarshaller(new DescribePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackagingGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describePackagingGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD Asset resources.
     *
     * @param listAssetsRequest
     * @return A Java Future containing the result of the ListAssets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListAssets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssetsResponse> listAssets(ListAssetsRequest listAssetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAssetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssetsRequest, ListAssetsResponse>().withOperationName("ListAssets")
                            .withMarshaller(new ListAssetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAssetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAssetsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListAssetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD Asset resources.<br/>
     * <p>
     * This is a variant of {@link #listAssets(software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsPublisher publisher = client.listAssetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsPublisher publisher = client.listAssetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssets(software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest)} operation.</b>
     * </p>
     *
     * @param listAssetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListAssets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListAssetsPublisher listAssetsPaginator(ListAssetsRequest listAssetsRequest) {
        return new ListAssetsPublisher(this, applyPaginatorUserAgent(listAssetsRequest));
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingConfiguration resources.
     *
     * @param listPackagingConfigurationsRequest
     * @return A Java Future containing the result of the ListPackagingConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListPackagingConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackagingConfigurationsResponse> listPackagingConfigurations(
            ListPackagingConfigurationsRequest listPackagingConfigurationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagingConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagingConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackagingConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagingConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagingConfigurationsRequest, ListPackagingConfigurationsResponse>()
                            .withOperationName("ListPackagingConfigurations")
                            .withMarshaller(new ListPackagingConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackagingConfigurationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPackagingConfigurationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListPackagingConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingConfiguration resources.<br/>
     * <p>
     * This is a variant of
     * {@link #listPackagingConfigurations(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsPublisher publisher = client.listPackagingConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsPublisher publisher = client.listPackagingConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackagingConfigurations(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackagingConfigurationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListPackagingConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPackagingConfigurationsPublisher listPackagingConfigurationsPaginator(
            ListPackagingConfigurationsRequest listPackagingConfigurationsRequest) {
        return new ListPackagingConfigurationsPublisher(this, applyPaginatorUserAgent(listPackagingConfigurationsRequest));
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingGroup resources.
     *
     * @param listPackagingGroupsRequest
     * @return A Java Future containing the result of the ListPackagingGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListPackagingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackagingGroupsResponse> listPackagingGroups(
            ListPackagingGroupsRequest listPackagingGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagingGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagingGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagingGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackagingGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagingGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagingGroupsRequest, ListPackagingGroupsResponse>()
                            .withOperationName("ListPackagingGroups")
                            .withMarshaller(new ListPackagingGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackagingGroupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPackagingGroupsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListPackagingGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingGroup resources.<br/>
     * <p>
     * This is a variant of
     * {@link #listPackagingGroups(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsPublisher publisher = client.listPackagingGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsPublisher publisher = client.listPackagingGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackagingGroups(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackagingGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListPackagingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPackagingGroupsPublisher listPackagingGroupsPaginator(ListPackagingGroupsRequest listPackagingGroupsRequest) {
        return new ListPackagingGroupsPublisher(this, applyPaginatorUserAgent(listPackagingGroupsRequest));
    }

    /**
     * Returns a list of the tags assigned to the specified resource.
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Adds tags to the specified resource. You can specify one or more tags to add.
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes tags from the specified resource. You can specify one or more tags to remove.
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates a specific packaging group. You can't change the id attribute or any other system-generated attributes.
     *
     * @param updatePackagingGroupRequest
     *        A MediaPackage VOD PackagingGroup resource configuration.
     * @return A Java Future containing the result of the UpdatePackagingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageVodException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MediaPackageVodAsyncClient.UpdatePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/UpdatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePackagingGroupResponse> updatePackagingGroup(
            UpdatePackagingGroupRequest updatePackagingGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePackagingGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePackagingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePackagingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePackagingGroupRequest, UpdatePackagingGroupResponse>()
                            .withOperationName("UpdatePackagingGroup")
                            .withMarshaller(new UpdatePackagingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updatePackagingGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updatePackagingGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdatePackagingGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageVodException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MediaPackageVodRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
