/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Rtmp Output Settings
 */
@Generated("software.amazon.awssdk:codegen")
public final class RtmpOutputSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<RtmpOutputSettings.Builder, RtmpOutputSettings> {
    private static final SdkField<String> CERTIFICATE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RtmpOutputSettings::certificateModeAsString)).setter(setter(Builder::certificateMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateMode").build()).build();

    private static final SdkField<Integer> CONNECTION_RETRY_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(RtmpOutputSettings::connectionRetryInterval)).setter(setter(Builder::connectionRetryInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectionRetryInterval").build())
            .build();

    private static final SdkField<OutputLocationRef> DESTINATION_FIELD = SdkField
            .<OutputLocationRef> builder(MarshallingType.SDK_POJO).getter(getter(RtmpOutputSettings::destination))
            .setter(setter(Builder::destination)).constructor(OutputLocationRef::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destination").build()).build();

    private static final SdkField<Integer> NUM_RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(RtmpOutputSettings::numRetries)).setter(setter(Builder::numRetries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numRetries").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CERTIFICATE_MODE_FIELD,
            CONNECTION_RETRY_INTERVAL_FIELD, DESTINATION_FIELD, NUM_RETRIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certificateMode;

    private final Integer connectionRetryInterval;

    private final OutputLocationRef destination;

    private final Integer numRetries;

    private RtmpOutputSettings(BuilderImpl builder) {
        this.certificateMode = builder.certificateMode;
        this.connectionRetryInterval = builder.connectionRetryInterval;
        this.destination = builder.destination;
        this.numRetries = builder.numRetries;
    }

    /**
     * If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority (CA). This will
     * cause rtmps outputs with self-signed certificates to fail.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #certificateMode}
     * will return {@link RtmpOutputCertificateMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #certificateModeAsString}.
     * </p>
     * 
     * @return If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority (CA).
     *         This will cause rtmps outputs with self-signed certificates to fail.
     * @see RtmpOutputCertificateMode
     */
    public RtmpOutputCertificateMode certificateMode() {
        return RtmpOutputCertificateMode.fromValue(certificateMode);
    }

    /**
     * If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority (CA). This will
     * cause rtmps outputs with self-signed certificates to fail.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #certificateMode}
     * will return {@link RtmpOutputCertificateMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #certificateModeAsString}.
     * </p>
     * 
     * @return If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority (CA).
     *         This will cause rtmps outputs with self-signed certificates to fail.
     * @see RtmpOutputCertificateMode
     */
    public String certificateModeAsString() {
        return certificateMode;
    }

    /**
     * Number of seconds to wait before retrying a connection to the Flash Media server if the connection is lost.
     * 
     * @return Number of seconds to wait before retrying a connection to the Flash Media server if the connection is
     *         lost.
     */
    public Integer connectionRetryInterval() {
        return connectionRetryInterval;
    }

    /**
     * The RTMP endpoint excluding the stream name (eg. rtmp://host/appname). For connection to Akamai, a username and
     * password must be supplied. URI fields accept format identifiers.
     * 
     * @return The RTMP endpoint excluding the stream name (eg. rtmp://host/appname). For connection to Akamai, a
     *         username and password must be supplied. URI fields accept format identifiers.
     */
    public OutputLocationRef destination() {
        return destination;
    }

    /**
     * Number of retry attempts.
     * 
     * @return Number of retry attempts.
     */
    public Integer numRetries() {
        return numRetries;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(connectionRetryInterval());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(numRetries());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RtmpOutputSettings)) {
            return false;
        }
        RtmpOutputSettings other = (RtmpOutputSettings) obj;
        return Objects.equals(certificateModeAsString(), other.certificateModeAsString())
                && Objects.equals(connectionRetryInterval(), other.connectionRetryInterval())
                && Objects.equals(destination(), other.destination()) && Objects.equals(numRetries(), other.numRetries());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RtmpOutputSettings").add("CertificateMode", certificateModeAsString())
                .add("ConnectionRetryInterval", connectionRetryInterval()).add("Destination", destination())
                .add("NumRetries", numRetries()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateMode":
            return Optional.ofNullable(clazz.cast(certificateModeAsString()));
        case "ConnectionRetryInterval":
            return Optional.ofNullable(clazz.cast(connectionRetryInterval()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "NumRetries":
            return Optional.ofNullable(clazz.cast(numRetries()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RtmpOutputSettings, T> g) {
        return obj -> g.apply((RtmpOutputSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RtmpOutputSettings> {
        /**
         * If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority (CA). This
         * will cause rtmps outputs with self-signed certificates to fail.
         * 
         * @param certificateMode
         *        If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority
         *        (CA). This will cause rtmps outputs with self-signed certificates to fail.
         * @see RtmpOutputCertificateMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RtmpOutputCertificateMode
         */
        Builder certificateMode(String certificateMode);

        /**
         * If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority (CA). This
         * will cause rtmps outputs with self-signed certificates to fail.
         * 
         * @param certificateMode
         *        If set to verifyAuthenticity, verify the tls certificate chain to a trusted Certificate Authority
         *        (CA). This will cause rtmps outputs with self-signed certificates to fail.
         * @see RtmpOutputCertificateMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RtmpOutputCertificateMode
         */
        Builder certificateMode(RtmpOutputCertificateMode certificateMode);

        /**
         * Number of seconds to wait before retrying a connection to the Flash Media server if the connection is lost.
         * 
         * @param connectionRetryInterval
         *        Number of seconds to wait before retrying a connection to the Flash Media server if the connection is
         *        lost.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionRetryInterval(Integer connectionRetryInterval);

        /**
         * The RTMP endpoint excluding the stream name (eg. rtmp://host/appname). For connection to Akamai, a username
         * and password must be supplied. URI fields accept format identifiers.
         * 
         * @param destination
         *        The RTMP endpoint excluding the stream name (eg. rtmp://host/appname). For connection to Akamai, a
         *        username and password must be supplied. URI fields accept format identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(OutputLocationRef destination);

        /**
         * The RTMP endpoint excluding the stream name (eg. rtmp://host/appname). For connection to Akamai, a username
         * and password must be supplied. URI fields accept format identifiers. This is a convenience that creates an
         * instance of the {@link OutputLocationRef.Builder} avoiding the need to create one manually via
         * {@link OutputLocationRef#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputLocationRef.Builder#build()} is called immediately and its
         * result is passed to {@link #destination(OutputLocationRef)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link OutputLocationRef.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(OutputLocationRef)
         */
        default Builder destination(Consumer<OutputLocationRef.Builder> destination) {
            return destination(OutputLocationRef.builder().applyMutation(destination).build());
        }

        /**
         * Number of retry attempts.
         * 
         * @param numRetries
         *        Number of retry attempts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numRetries(Integer numRetries);
    }

    static final class BuilderImpl implements Builder {
        private String certificateMode;

        private Integer connectionRetryInterval;

        private OutputLocationRef destination;

        private Integer numRetries;

        private BuilderImpl() {
        }

        private BuilderImpl(RtmpOutputSettings model) {
            certificateMode(model.certificateMode);
            connectionRetryInterval(model.connectionRetryInterval);
            destination(model.destination);
            numRetries(model.numRetries);
        }

        public final String getCertificateMode() {
            return certificateMode;
        }

        @Override
        public final Builder certificateMode(String certificateMode) {
            this.certificateMode = certificateMode;
            return this;
        }

        @Override
        public final Builder certificateMode(RtmpOutputCertificateMode certificateMode) {
            this.certificateMode(certificateMode == null ? null : certificateMode.toString());
            return this;
        }

        public final void setCertificateMode(String certificateMode) {
            this.certificateMode = certificateMode;
        }

        public final Integer getConnectionRetryInterval() {
            return connectionRetryInterval;
        }

        @Override
        public final Builder connectionRetryInterval(Integer connectionRetryInterval) {
            this.connectionRetryInterval = connectionRetryInterval;
            return this;
        }

        public final void setConnectionRetryInterval(Integer connectionRetryInterval) {
            this.connectionRetryInterval = connectionRetryInterval;
        }

        public final OutputLocationRef.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        @Override
        public final Builder destination(OutputLocationRef destination) {
            this.destination = destination;
            return this;
        }

        public final void setDestination(OutputLocationRef.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        public final Integer getNumRetries() {
            return numRetries;
        }

        @Override
        public final Builder numRetries(Integer numRetries) {
            this.numRetries = numRetries;
            return this;
        }

        public final void setNumRetries(Integer numRetries) {
            this.numRetries = numRetries;
        }

        @Override
        public RtmpOutputSettings build() {
            return new RtmpOutputSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
