/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Channel class that the channel should be updated to.
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateChannelClassRequest extends MediaLiveRequest implements
        ToCopyableBuilder<UpdateChannelClassRequest.Builder, UpdateChannelClassRequest> {
    private static final SdkField<String> CHANNEL_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateChannelClassRequest::channelClassAsString)).setter(setter(Builder::channelClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("channelClass").build()).build();

    private static final SdkField<String> CHANNEL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateChannelClassRequest::channelId)).setter(setter(Builder::channelId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("channelId").build()).build();

    private static final SdkField<List<OutputDestination>> DESTINATIONS_FIELD = SdkField
            .<List<OutputDestination>> builder(MarshallingType.LIST)
            .getter(getter(UpdateChannelClassRequest::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OutputDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(OutputDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHANNEL_CLASS_FIELD,
            CHANNEL_ID_FIELD, DESTINATIONS_FIELD));

    private final String channelClass;

    private final String channelId;

    private final List<OutputDestination> destinations;

    private UpdateChannelClassRequest(BuilderImpl builder) {
        super(builder);
        this.channelClass = builder.channelClass;
        this.channelId = builder.channelId;
        this.destinations = builder.destinations;
    }

    /**
     * The channel class that you wish to update this channel to use.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #channelClass} will
     * return {@link ChannelClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #channelClassAsString}.
     * </p>
     * 
     * @return The channel class that you wish to update this channel to use.
     * @see ChannelClass
     */
    public ChannelClass channelClass() {
        return ChannelClass.fromValue(channelClass);
    }

    /**
     * The channel class that you wish to update this channel to use.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #channelClass} will
     * return {@link ChannelClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #channelClassAsString}.
     * </p>
     * 
     * @return The channel class that you wish to update this channel to use.
     * @see ChannelClass
     */
    public String channelClassAsString() {
        return channelClass;
    }

    /**
     * Channel Id of the channel whose class should be updated.
     * 
     * @return Channel Id of the channel whose class should be updated.
     */
    public String channelId() {
        return channelId;
    }

    /**
     * Returns true if the Destinations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * A list of output destinations for this channel.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDestinations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of output destinations for this channel.
     */
    public List<OutputDestination> destinations() {
        return destinations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(channelClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(channelId());
        hashCode = 31 * hashCode + Objects.hashCode(destinations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateChannelClassRequest)) {
            return false;
        }
        UpdateChannelClassRequest other = (UpdateChannelClassRequest) obj;
        return Objects.equals(channelClassAsString(), other.channelClassAsString())
                && Objects.equals(channelId(), other.channelId()) && Objects.equals(destinations(), other.destinations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UpdateChannelClassRequest").add("ChannelClass", channelClassAsString())
                .add("ChannelId", channelId()).add("Destinations", destinations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ChannelClass":
            return Optional.ofNullable(clazz.cast(channelClassAsString()));
        case "ChannelId":
            return Optional.ofNullable(clazz.cast(channelId()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateChannelClassRequest, T> g) {
        return obj -> g.apply((UpdateChannelClassRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MediaLiveRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateChannelClassRequest> {
        /**
         * The channel class that you wish to update this channel to use.
         * 
         * @param channelClass
         *        The channel class that you wish to update this channel to use.
         * @see ChannelClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelClass
         */
        Builder channelClass(String channelClass);

        /**
         * The channel class that you wish to update this channel to use.
         * 
         * @param channelClass
         *        The channel class that you wish to update this channel to use.
         * @see ChannelClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelClass
         */
        Builder channelClass(ChannelClass channelClass);

        /**
         * Channel Id of the channel whose class should be updated.
         * 
         * @param channelId
         *        Channel Id of the channel whose class should be updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelId(String channelId);

        /**
         * A list of output destinations for this channel.
         * 
         * @param destinations
         *        A list of output destinations for this channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<OutputDestination> destinations);

        /**
         * A list of output destinations for this channel.
         * 
         * @param destinations
         *        A list of output destinations for this channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(OutputDestination... destinations);

        /**
         * A list of output destinations for this channel. This is a convenience that creates an instance of the
         * {@link List<OutputDestination>.Builder} avoiding the need to create one manually via {@link List
         * <OutputDestination>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<OutputDestination>.Builder#build()} is called immediately
         * and its result is passed to {@link #destinations(List<OutputDestination>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<OutputDestination>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<OutputDestination>)
         */
        Builder destinations(Consumer<OutputDestination.Builder>... destinations);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MediaLiveRequest.BuilderImpl implements Builder {
        private String channelClass;

        private String channelId;

        private List<OutputDestination> destinations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateChannelClassRequest model) {
            super(model);
            channelClass(model.channelClass);
            channelId(model.channelId);
            destinations(model.destinations);
        }

        public final String getChannelClassAsString() {
            return channelClass;
        }

        @Override
        public final Builder channelClass(String channelClass) {
            this.channelClass = channelClass;
            return this;
        }

        @Override
        public final Builder channelClass(ChannelClass channelClass) {
            this.channelClass(channelClass == null ? null : channelClass.toString());
            return this;
        }

        public final void setChannelClass(String channelClass) {
            this.channelClass = channelClass;
        }

        public final String getChannelId() {
            return channelId;
        }

        @Override
        public final Builder channelId(String channelId) {
            this.channelId = channelId;
            return this;
        }

        public final void setChannelId(String channelId) {
            this.channelId = channelId;
        }

        public final Collection<OutputDestination.Builder> getDestinations() {
            return destinations != null ? destinations.stream().map(OutputDestination::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder destinations(Collection<OutputDestination> destinations) {
            this.destinations = ___listOfOutputDestinationCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(OutputDestination... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<OutputDestination.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> OutputDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinations(Collection<OutputDestination.BuilderImpl> destinations) {
            this.destinations = ___listOfOutputDestinationCopier.copyFromBuilder(destinations);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateChannelClassRequest build() {
            return new UpdateChannelClassRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
