/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.mediaconnect.MediaConnectAsyncClient;
import software.amazon.awssdk.services.mediaconnect.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.GetRouterInputRequest;
import software.amazon.awssdk.services.mediaconnect.model.GetRouterInputResponse;
import software.amazon.awssdk.services.mediaconnect.model.GetRouterOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.GetRouterOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectRequest;
import software.amazon.awssdk.services.mediaconnect.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultMediaConnectAsyncWaiter implements MediaConnectAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final MediaConnectAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeFlowResponse> flowActiveWaiter;

    private final AsyncWaiter<DescribeFlowResponse> flowDeletedWaiter;

    private final AsyncWaiter<DescribeFlowResponse> flowStandbyWaiter;

    private final AsyncWaiter<GetRouterInputResponse> inputActiveWaiter;

    private final AsyncWaiter<GetRouterInputResponse> inputDeletedWaiter;

    private final AsyncWaiter<GetRouterInputResponse> inputStandbyWaiter;

    private final AsyncWaiter<GetRouterOutputResponse> outputActiveWaiter;

    private final AsyncWaiter<GetRouterOutputResponse> outputDeletedWaiter;

    private final AsyncWaiter<GetRouterOutputResponse> outputRoutedWaiter;

    private final AsyncWaiter<GetRouterOutputResponse> outputStandbyWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultMediaConnectAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = MediaConnectAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.flowActiveWaiter = AsyncWaiter.builder(DescribeFlowResponse.class).acceptors(flowActiveWaiterAcceptors())
                .overrideConfiguration(flowActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.flowDeletedWaiter = AsyncWaiter.builder(DescribeFlowResponse.class).acceptors(flowDeletedWaiterAcceptors())
                .overrideConfiguration(flowDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.flowStandbyWaiter = AsyncWaiter.builder(DescribeFlowResponse.class).acceptors(flowStandbyWaiterAcceptors())
                .overrideConfiguration(flowStandbyWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.inputActiveWaiter = AsyncWaiter.builder(GetRouterInputResponse.class).acceptors(inputActiveWaiterAcceptors())
                .overrideConfiguration(inputActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.inputDeletedWaiter = AsyncWaiter.builder(GetRouterInputResponse.class).acceptors(inputDeletedWaiterAcceptors())
                .overrideConfiguration(inputDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.inputStandbyWaiter = AsyncWaiter.builder(GetRouterInputResponse.class).acceptors(inputStandbyWaiterAcceptors())
                .overrideConfiguration(inputStandbyWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.outputActiveWaiter = AsyncWaiter.builder(GetRouterOutputResponse.class).acceptors(outputActiveWaiterAcceptors())
                .overrideConfiguration(outputActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.outputDeletedWaiter = AsyncWaiter.builder(GetRouterOutputResponse.class).acceptors(outputDeletedWaiterAcceptors())
                .overrideConfiguration(outputDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.outputRoutedWaiter = AsyncWaiter.builder(GetRouterOutputResponse.class).acceptors(outputRoutedWaiterAcceptors())
                .overrideConfiguration(outputRoutedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.outputStandbyWaiter = AsyncWaiter.builder(GetRouterOutputResponse.class).acceptors(outputStandbyWaiterAcceptors())
                .overrideConfiguration(outputStandbyWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowActive(DescribeFlowRequest describeFlowRequest) {
        return flowActiveWaiter.runAsync(() -> client.describeFlow(applyWaitersUserAgent(describeFlowRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowActive(DescribeFlowRequest describeFlowRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return flowActiveWaiter.runAsync(() -> client.describeFlow(applyWaitersUserAgent(describeFlowRequest)),
                flowActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowDeleted(DescribeFlowRequest describeFlowRequest) {
        return flowDeletedWaiter.runAsync(() -> client.describeFlow(applyWaitersUserAgent(describeFlowRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowDeleted(DescribeFlowRequest describeFlowRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return flowDeletedWaiter.runAsync(() -> client.describeFlow(applyWaitersUserAgent(describeFlowRequest)),
                flowDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowStandby(DescribeFlowRequest describeFlowRequest) {
        return flowStandbyWaiter.runAsync(() -> client.describeFlow(applyWaitersUserAgent(describeFlowRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowStandby(DescribeFlowRequest describeFlowRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return flowStandbyWaiter.runAsync(() -> client.describeFlow(applyWaitersUserAgent(describeFlowRequest)),
                flowStandbyWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterInputResponse>> waitUntilInputActive(
            GetRouterInputRequest getRouterInputRequest) {
        return inputActiveWaiter.runAsync(() -> client.getRouterInput(applyWaitersUserAgent(getRouterInputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterInputResponse>> waitUntilInputActive(
            GetRouterInputRequest getRouterInputRequest, WaiterOverrideConfiguration overrideConfig) {
        return inputActiveWaiter.runAsync(() -> client.getRouterInput(applyWaitersUserAgent(getRouterInputRequest)),
                inputActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterInputResponse>> waitUntilInputDeleted(
            GetRouterInputRequest getRouterInputRequest) {
        return inputDeletedWaiter.runAsync(() -> client.getRouterInput(applyWaitersUserAgent(getRouterInputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterInputResponse>> waitUntilInputDeleted(
            GetRouterInputRequest getRouterInputRequest, WaiterOverrideConfiguration overrideConfig) {
        return inputDeletedWaiter.runAsync(() -> client.getRouterInput(applyWaitersUserAgent(getRouterInputRequest)),
                inputDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterInputResponse>> waitUntilInputStandby(
            GetRouterInputRequest getRouterInputRequest) {
        return inputStandbyWaiter.runAsync(() -> client.getRouterInput(applyWaitersUserAgent(getRouterInputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterInputResponse>> waitUntilInputStandby(
            GetRouterInputRequest getRouterInputRequest, WaiterOverrideConfiguration overrideConfig) {
        return inputStandbyWaiter.runAsync(() -> client.getRouterInput(applyWaitersUserAgent(getRouterInputRequest)),
                inputStandbyWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputActive(
            GetRouterOutputRequest getRouterOutputRequest) {
        return outputActiveWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputActive(
            GetRouterOutputRequest getRouterOutputRequest, WaiterOverrideConfiguration overrideConfig) {
        return outputActiveWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)),
                outputActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputDeleted(
            GetRouterOutputRequest getRouterOutputRequest) {
        return outputDeletedWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputDeleted(
            GetRouterOutputRequest getRouterOutputRequest, WaiterOverrideConfiguration overrideConfig) {
        return outputDeletedWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)),
                outputDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputRouted(
            GetRouterOutputRequest getRouterOutputRequest) {
        return outputRoutedWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputRouted(
            GetRouterOutputRequest getRouterOutputRequest, WaiterOverrideConfiguration overrideConfig) {
        return outputRoutedWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)),
                outputRoutedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputStandby(
            GetRouterOutputRequest getRouterOutputRequest) {
        return outputStandbyWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetRouterOutputResponse>> waitUntilOutputStandby(
            GetRouterOutputRequest getRouterOutputRequest, WaiterOverrideConfiguration overrideConfig) {
        return outputStandbyWaiter.runAsync(() -> client.getRouterOutput(applyWaitersUserAgent(getRouterOutputRequest)),
                outputStandbyWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeFlowResponse>> flowActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFlowResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Flow").field("Status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Flow").field("Status").value(), "STARTING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Flow").field("Status").value(), "UPDATING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Flow").field("Status").value(), "STANDBY");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Flow.Status=STANDBY) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Flow").field("Status").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Flow.Status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFlowResponse>> flowDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFlowResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NotFoundException")));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Flow").field("Status").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Flow").field("Status").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Flow.Status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFlowResponse>> flowStandbyWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFlowResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Flow").field("Status").value(), "STANDBY");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Flow").field("Status").value(), "STOPPING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Flow").field("Status").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Flow.Status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterInputResponse>> inputActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterInput").field("State").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterInput").field("State").value(), "STARTING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterInput").field("State").value(), "UPDATING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RouterInput").field("State").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RouterInput.State=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterInputResponse>> inputDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterInput").field("State").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RouterInput").field("State").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RouterInput.State=ERROR) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterInputResponse>> inputStandbyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterInput").field("State").value(), "STANDBY");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterInput").field("State").value(), "STOPPING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RouterInput").field("State").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RouterInput.State=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterOutputResponse>> outputActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterOutputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("State").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("State").value(), "STARTING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("State").value(), "UPDATING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RouterOutput").field("State").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RouterOutput.State=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterOutputResponse>> outputDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterOutputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("State").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RouterOutput").field("State").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RouterOutput.State=ERROR) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterOutputResponse>> outputRoutedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterOutputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("RoutedState").value(), "ROUTED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("RoutedState").value(), "ROUTING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRouterOutputResponse>> outputStandbyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRouterOutputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("State").value(), "STANDBY");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RouterOutput").field("State").value(), "STOPPING");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InternalServerErrorException")));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ServiceUnavailableException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RouterOutput").field("State").value(), "ERROR");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RouterOutput.State=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration flowActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration flowDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration flowStandbyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputStandbyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration outputActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration outputDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration outputRoutedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration outputStandbyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static MediaConnectAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends MediaConnectRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements MediaConnectAsyncWaiter.Builder {
        private MediaConnectAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public MediaConnectAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public MediaConnectAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public MediaConnectAsyncWaiter.Builder client(MediaConnectAsyncClient client) {
            this.client = client;
            return this;
        }

        public MediaConnectAsyncWaiter build() {
            return new DefaultMediaConnectAsyncWaiter(this);
        }
    }
}
