/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateCodeSigningConfigRequest extends LambdaRequest implements
        ToCopyableBuilder<CreateCodeSigningConfigRequest.Builder, CreateCodeSigningConfigRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateCodeSigningConfigRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<AllowedPublishers> ALLOWED_PUBLISHERS_FIELD = SdkField
            .<AllowedPublishers> builder(MarshallingType.SDK_POJO).memberName("AllowedPublishers")
            .getter(getter(CreateCodeSigningConfigRequest::allowedPublishers)).setter(setter(Builder::allowedPublishers))
            .constructor(AllowedPublishers::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedPublishers").build()).build();

    private static final SdkField<CodeSigningPolicies> CODE_SIGNING_POLICIES_FIELD = SdkField
            .<CodeSigningPolicies> builder(MarshallingType.SDK_POJO).memberName("CodeSigningPolicies")
            .getter(getter(CreateCodeSigningConfigRequest::codeSigningPolicies)).setter(setter(Builder::codeSigningPolicies))
            .constructor(CodeSigningPolicies::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CodeSigningPolicies").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateCodeSigningConfigRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            ALLOWED_PUBLISHERS_FIELD, CODE_SIGNING_POLICIES_FIELD, TAGS_FIELD));

    private final String description;

    private final AllowedPublishers allowedPublishers;

    private final CodeSigningPolicies codeSigningPolicies;

    private final Map<String, String> tags;

    private CreateCodeSigningConfigRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.allowedPublishers = builder.allowedPublishers;
        this.codeSigningPolicies = builder.codeSigningPolicies;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Descriptive name for this code signing configuration.
     * </p>
     * 
     * @return Descriptive name for this code signing configuration.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Signing profiles for this code signing configuration.
     * </p>
     * 
     * @return Signing profiles for this code signing configuration.
     */
    public final AllowedPublishers allowedPublishers() {
        return allowedPublishers;
    }

    /**
     * <p>
     * The code signing policies define the actions to take if the validation checks fail.
     * </p>
     * 
     * @return The code signing policies define the actions to take if the validation checks fail.
     */
    public final CodeSigningPolicies codeSigningPolicies() {
        return codeSigningPolicies;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of tags to add to the code signing configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags to add to the code signing configuration.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(allowedPublishers());
        hashCode = 31 * hashCode + Objects.hashCode(codeSigningPolicies());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateCodeSigningConfigRequest)) {
            return false;
        }
        CreateCodeSigningConfigRequest other = (CreateCodeSigningConfigRequest) obj;
        return Objects.equals(description(), other.description())
                && Objects.equals(allowedPublishers(), other.allowedPublishers())
                && Objects.equals(codeSigningPolicies(), other.codeSigningPolicies()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateCodeSigningConfigRequest").add("Description", description())
                .add("AllowedPublishers", allowedPublishers()).add("CodeSigningPolicies", codeSigningPolicies())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AllowedPublishers":
            return Optional.ofNullable(clazz.cast(allowedPublishers()));
        case "CodeSigningPolicies":
            return Optional.ofNullable(clazz.cast(codeSigningPolicies()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateCodeSigningConfigRequest, T> g) {
        return obj -> g.apply((CreateCodeSigningConfigRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LambdaRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateCodeSigningConfigRequest> {
        /**
         * <p>
         * Descriptive name for this code signing configuration.
         * </p>
         * 
         * @param description
         *        Descriptive name for this code signing configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Signing profiles for this code signing configuration.
         * </p>
         * 
         * @param allowedPublishers
         *        Signing profiles for this code signing configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedPublishers(AllowedPublishers allowedPublishers);

        /**
         * <p>
         * Signing profiles for this code signing configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link AllowedPublishers.Builder} avoiding the
         * need to create one manually via {@link AllowedPublishers#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AllowedPublishers.Builder#build()} is called immediately and its
         * result is passed to {@link #allowedPublishers(AllowedPublishers)}.
         * 
         * @param allowedPublishers
         *        a consumer that will call methods on {@link AllowedPublishers.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #allowedPublishers(AllowedPublishers)
         */
        default Builder allowedPublishers(Consumer<AllowedPublishers.Builder> allowedPublishers) {
            return allowedPublishers(AllowedPublishers.builder().applyMutation(allowedPublishers).build());
        }

        /**
         * <p>
         * The code signing policies define the actions to take if the validation checks fail.
         * </p>
         * 
         * @param codeSigningPolicies
         *        The code signing policies define the actions to take if the validation checks fail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codeSigningPolicies(CodeSigningPolicies codeSigningPolicies);

        /**
         * <p>
         * The code signing policies define the actions to take if the validation checks fail.
         * </p>
         * This is a convenience method that creates an instance of the {@link CodeSigningPolicies.Builder} avoiding the
         * need to create one manually via {@link CodeSigningPolicies#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CodeSigningPolicies.Builder#build()} is called immediately and
         * its result is passed to {@link #codeSigningPolicies(CodeSigningPolicies)}.
         * 
         * @param codeSigningPolicies
         *        a consumer that will call methods on {@link CodeSigningPolicies.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #codeSigningPolicies(CodeSigningPolicies)
         */
        default Builder codeSigningPolicies(Consumer<CodeSigningPolicies.Builder> codeSigningPolicies) {
            return codeSigningPolicies(CodeSigningPolicies.builder().applyMutation(codeSigningPolicies).build());
        }

        /**
         * <p>
         * A list of tags to add to the code signing configuration.
         * </p>
         * 
         * @param tags
         *        A list of tags to add to the code signing configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LambdaRequest.BuilderImpl implements Builder {
        private String description;

        private AllowedPublishers allowedPublishers;

        private CodeSigningPolicies codeSigningPolicies;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateCodeSigningConfigRequest model) {
            super(model);
            description(model.description);
            allowedPublishers(model.allowedPublishers);
            codeSigningPolicies(model.codeSigningPolicies);
            tags(model.tags);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final AllowedPublishers.Builder getAllowedPublishers() {
            return allowedPublishers != null ? allowedPublishers.toBuilder() : null;
        }

        public final void setAllowedPublishers(AllowedPublishers.BuilderImpl allowedPublishers) {
            this.allowedPublishers = allowedPublishers != null ? allowedPublishers.build() : null;
        }

        @Override
        public final Builder allowedPublishers(AllowedPublishers allowedPublishers) {
            this.allowedPublishers = allowedPublishers;
            return this;
        }

        public final CodeSigningPolicies.Builder getCodeSigningPolicies() {
            return codeSigningPolicies != null ? codeSigningPolicies.toBuilder() : null;
        }

        public final void setCodeSigningPolicies(CodeSigningPolicies.BuilderImpl codeSigningPolicies) {
            this.codeSigningPolicies = codeSigningPolicies != null ? codeSigningPolicies.build() : null;
        }

        @Override
        public final Builder codeSigningPolicies(CodeSigningPolicies codeSigningPolicies) {
            this.codeSigningPolicies = codeSigningPolicies;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateCodeSigningConfigRequest build() {
            return new CreateCodeSigningConfigRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
