/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure for a table with columns object. This object is only used when granting a SELECT permission.
 * </p>
 * <p>
 * This object must take a value for at least one of <code>ColumnsNames</code>, <code>ColumnsIndexes</code>, or
 * <code>ColumnsWildcard</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TableWithColumnsResource implements SdkPojo, Serializable,
        ToCopyableBuilder<TableWithColumnsResource.Builder, TableWithColumnsResource> {
    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TableWithColumnsResource::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TableWithColumnsResource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> COLUMN_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(TableWithColumnsResource::columnNames))
            .setter(setter(Builder::columnNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ColumnWildcard> COLUMN_WILDCARD_FIELD = SdkField
            .<ColumnWildcard> builder(MarshallingType.SDK_POJO).getter(getter(TableWithColumnsResource::columnWildcard))
            .setter(setter(Builder::columnWildcard)).constructor(ColumnWildcard::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnWildcard").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATABASE_NAME_FIELD,
            NAME_FIELD, COLUMN_NAMES_FIELD, COLUMN_WILDCARD_FIELD));

    private static final long serialVersionUID = 1L;

    private final String databaseName;

    private final String name;

    private final List<String> columnNames;

    private final ColumnWildcard columnWildcard;

    private TableWithColumnsResource(BuilderImpl builder) {
        this.databaseName = builder.databaseName;
        this.name = builder.name;
        this.columnNames = builder.columnNames;
        this.columnWildcard = builder.columnWildcard;
    }

    /**
     * <p>
     * The name of the database for the table with columns resource. Unique to the Data Catalog. A database is a set of
     * associated table definitions organized into a logical group. You can Grant and Revoke database privileges to a
     * principal.
     * </p>
     * 
     * @return The name of the database for the table with columns resource. Unique to the Data Catalog. A database is a
     *         set of associated table definitions organized into a logical group. You can Grant and Revoke database
     *         privileges to a principal.
     */
    public String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * The name of the table resource. A table is a metadata definition that represents your data. You can Grant and
     * Revoke table privileges to a principal.
     * </p>
     * 
     * @return The name of the table resource. A table is a metadata definition that represents your data. You can Grant
     *         and Revoke table privileges to a principal.
     */
    public String name() {
        return name;
    }

    /**
     * Returns true if the ColumnNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasColumnNames() {
        return columnNames != null && !(columnNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of column names for the table. At least one of <code>ColumnNames</code> or <code>ColumnWildcard</code>
     * is required.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasColumnNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of column names for the table. At least one of <code>ColumnNames</code> or
     *         <code>ColumnWildcard</code> is required.
     */
    public List<String> columnNames() {
        return columnNames;
    }

    /**
     * <p>
     * A wildcard specified by a <code>ColumnWildcard</code> object. At least one of <code>ColumnNames</code> or
     * <code>ColumnWildcard</code> is required.
     * </p>
     * 
     * @return A wildcard specified by a <code>ColumnWildcard</code> object. At least one of <code>ColumnNames</code> or
     *         <code>ColumnWildcard</code> is required.
     */
    public ColumnWildcard columnWildcard() {
        return columnWildcard;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(columnNames());
        hashCode = 31 * hashCode + Objects.hashCode(columnWildcard());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TableWithColumnsResource)) {
            return false;
        }
        TableWithColumnsResource other = (TableWithColumnsResource) obj;
        return Objects.equals(databaseName(), other.databaseName()) && Objects.equals(name(), other.name())
                && Objects.equals(columnNames(), other.columnNames()) && Objects.equals(columnWildcard(), other.columnWildcard());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TableWithColumnsResource").add("DatabaseName", databaseName()).add("Name", name())
                .add("ColumnNames", columnNames()).add("ColumnWildcard", columnWildcard()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ColumnNames":
            return Optional.ofNullable(clazz.cast(columnNames()));
        case "ColumnWildcard":
            return Optional.ofNullable(clazz.cast(columnWildcard()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TableWithColumnsResource, T> g) {
        return obj -> g.apply((TableWithColumnsResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TableWithColumnsResource> {
        /**
         * <p>
         * The name of the database for the table with columns resource. Unique to the Data Catalog. A database is a set
         * of associated table definitions organized into a logical group. You can Grant and Revoke database privileges
         * to a principal.
         * </p>
         * 
         * @param databaseName
         *        The name of the database for the table with columns resource. Unique to the Data Catalog. A database
         *        is a set of associated table definitions organized into a logical group. You can Grant and Revoke
         *        database privileges to a principal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * The name of the table resource. A table is a metadata definition that represents your data. You can Grant and
         * Revoke table privileges to a principal.
         * </p>
         * 
         * @param name
         *        The name of the table resource. A table is a metadata definition that represents your data. You can
         *        Grant and Revoke table privileges to a principal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The list of column names for the table. At least one of <code>ColumnNames</code> or
         * <code>ColumnWildcard</code> is required.
         * </p>
         * 
         * @param columnNames
         *        The list of column names for the table. At least one of <code>ColumnNames</code> or
         *        <code>ColumnWildcard</code> is required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnNames(Collection<String> columnNames);

        /**
         * <p>
         * The list of column names for the table. At least one of <code>ColumnNames</code> or
         * <code>ColumnWildcard</code> is required.
         * </p>
         * 
         * @param columnNames
         *        The list of column names for the table. At least one of <code>ColumnNames</code> or
         *        <code>ColumnWildcard</code> is required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnNames(String... columnNames);

        /**
         * <p>
         * A wildcard specified by a <code>ColumnWildcard</code> object. At least one of <code>ColumnNames</code> or
         * <code>ColumnWildcard</code> is required.
         * </p>
         * 
         * @param columnWildcard
         *        A wildcard specified by a <code>ColumnWildcard</code> object. At least one of <code>ColumnNames</code>
         *        or <code>ColumnWildcard</code> is required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnWildcard(ColumnWildcard columnWildcard);

        /**
         * <p>
         * A wildcard specified by a <code>ColumnWildcard</code> object. At least one of <code>ColumnNames</code> or
         * <code>ColumnWildcard</code> is required.
         * </p>
         * This is a convenience that creates an instance of the {@link ColumnWildcard.Builder} avoiding the need to
         * create one manually via {@link ColumnWildcard#builder()}.
         *
         * When the {@link Consumer} completes, {@link ColumnWildcard.Builder#build()} is called immediately and its
         * result is passed to {@link #columnWildcard(ColumnWildcard)}.
         * 
         * @param columnWildcard
         *        a consumer that will call methods on {@link ColumnWildcard.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #columnWildcard(ColumnWildcard)
         */
        default Builder columnWildcard(Consumer<ColumnWildcard.Builder> columnWildcard) {
            return columnWildcard(ColumnWildcard.builder().applyMutation(columnWildcard).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String databaseName;

        private String name;

        private List<String> columnNames = DefaultSdkAutoConstructList.getInstance();

        private ColumnWildcard columnWildcard;

        private BuilderImpl() {
        }

        private BuilderImpl(TableWithColumnsResource model) {
            databaseName(model.databaseName);
            name(model.name);
            columnNames(model.columnNames);
            columnWildcard(model.columnWildcard);
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<String> getColumnNames() {
            return columnNames;
        }

        @Override
        public final Builder columnNames(Collection<String> columnNames) {
            this.columnNames = ColumnNamesCopier.copy(columnNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder columnNames(String... columnNames) {
            columnNames(Arrays.asList(columnNames));
            return this;
        }

        public final void setColumnNames(Collection<String> columnNames) {
            this.columnNames = ColumnNamesCopier.copy(columnNames);
        }

        public final ColumnWildcard.Builder getColumnWildcard() {
            return columnWildcard != null ? columnWildcard.toBuilder() : null;
        }

        @Override
        public final Builder columnWildcard(ColumnWildcard columnWildcard) {
            this.columnWildcard = columnWildcard;
            return this;
        }

        public final void setColumnWildcard(ColumnWildcard.BuilderImpl columnWildcard) {
            this.columnWildcard = columnWildcard != null ? columnWildcard.build() : null;
        }

        @Override
        public TableWithColumnsResource build() {
            return new TableWithColumnsResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
