/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.util.Map;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.kms.transform.GrantConstraintsMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that you can use to allow certain operations in the grant only when the desired encryption context is
 * present. For more information about encryption context, see <a
 * href="http://docs.aws.amazon.com/kms/latest/developerguide/encryption-context.html">Encryption Context</a> in the
 * <i>AWS Key Management Service Developer Guide</i>.
 * </p>
 * <p>
 * Grant constraints apply only to operations that accept encryption context as input. For example, the
 * <code> <a>DescribeKey</a> </code> operation does not accept encryption context as input. A grant that allows the
 * <code>DescribeKey</code> operation does so regardless of the grant constraints. In constrast, the
 * <code> <a>Encrypt</a> </code> operation accepts encryption context as input. A grant that allows the
 * <code>Encrypt</code> operation does so only when the encryption context of the <code>Encrypt</code> operation
 * satisfies the grant constraints.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GrantConstraints implements StructuredPojo, ToCopyableBuilder<GrantConstraints.Builder, GrantConstraints> {
    private final Map<String, String> encryptionContextSubset;

    private final Map<String, String> encryptionContextEquals;

    private GrantConstraints(BuilderImpl builder) {
        this.encryptionContextSubset = builder.encryptionContextSubset;
        this.encryptionContextEquals = builder.encryptionContextEquals;
    }

    /**
     * <p>
     * A list of key-value pairs, all of which must be present in the encryption context of certain subsequent
     * operations that the grant allows. When certain subsequent operations allowed by the grant include encryption
     * context that matches this list or is a superset of this list, the grant allows the operation. Otherwise, the
     * grant does not allow the operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of key-value pairs, all of which must be present in the encryption context of certain subsequent
     *         operations that the grant allows. When certain subsequent operations allowed by the grant include
     *         encryption context that matches this list or is a superset of this list, the grant allows the operation.
     *         Otherwise, the grant does not allow the operation.
     */
    public Map<String, String> encryptionContextSubset() {
        return encryptionContextSubset;
    }

    /**
     * <p>
     * A list of key-value pairs that must be present in the encryption context of certain subsequent operations that
     * the grant allows. When certain subsequent operations allowed by the grant include encryption context that matches
     * this list, the grant allows the operation. Otherwise, the grant does not allow the operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of key-value pairs that must be present in the encryption context of certain subsequent operations
     *         that the grant allows. When certain subsequent operations allowed by the grant include encryption context
     *         that matches this list, the grant allows the operation. Otherwise, the grant does not allow the
     *         operation.
     */
    public Map<String, String> encryptionContextEquals() {
        return encryptionContextEquals;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((encryptionContextSubset() == null) ? 0 : encryptionContextSubset().hashCode());
        hashCode = 31 * hashCode + ((encryptionContextEquals() == null) ? 0 : encryptionContextEquals().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GrantConstraints)) {
            return false;
        }
        GrantConstraints other = (GrantConstraints) obj;
        if (other.encryptionContextSubset() == null ^ this.encryptionContextSubset() == null) {
            return false;
        }
        if (other.encryptionContextSubset() != null && !other.encryptionContextSubset().equals(this.encryptionContextSubset())) {
            return false;
        }
        if (other.encryptionContextEquals() == null ^ this.encryptionContextEquals() == null) {
            return false;
        }
        if (other.encryptionContextEquals() != null && !other.encryptionContextEquals().equals(this.encryptionContextEquals())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (encryptionContextSubset() != null) {
            sb.append("EncryptionContextSubset: ").append(encryptionContextSubset()).append(",");
        }
        if (encryptionContextEquals() != null) {
            sb.append("EncryptionContextEquals: ").append(encryptionContextEquals()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        GrantConstraintsMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, GrantConstraints> {
        /**
         * <p>
         * A list of key-value pairs, all of which must be present in the encryption context of certain subsequent
         * operations that the grant allows. When certain subsequent operations allowed by the grant include encryption
         * context that matches this list or is a superset of this list, the grant allows the operation. Otherwise, the
         * grant does not allow the operation.
         * </p>
         * 
         * @param encryptionContextSubset
         *        A list of key-value pairs, all of which must be present in the encryption context of certain
         *        subsequent operations that the grant allows. When certain subsequent operations allowed by the grant
         *        include encryption context that matches this list or is a superset of this list, the grant allows the
         *        operation. Otherwise, the grant does not allow the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContextSubset(Map<String, String> encryptionContextSubset);

        /**
         * <p>
         * A list of key-value pairs that must be present in the encryption context of certain subsequent operations
         * that the grant allows. When certain subsequent operations allowed by the grant include encryption context
         * that matches this list, the grant allows the operation. Otherwise, the grant does not allow the operation.
         * </p>
         * 
         * @param encryptionContextEquals
         *        A list of key-value pairs that must be present in the encryption context of certain subsequent
         *        operations that the grant allows. When certain subsequent operations allowed by the grant include
         *        encryption context that matches this list, the grant allows the operation. Otherwise, the grant does
         *        not allow the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContextEquals(Map<String, String> encryptionContextEquals);
    }

    private static final class BuilderImpl implements Builder {
        private Map<String, String> encryptionContextSubset;

        private Map<String, String> encryptionContextEquals;

        private BuilderImpl() {
        }

        private BuilderImpl(GrantConstraints model) {
            setEncryptionContextSubset(model.encryptionContextSubset);
            setEncryptionContextEquals(model.encryptionContextEquals);
        }

        public final Map<String, String> getEncryptionContextSubset() {
            return encryptionContextSubset;
        }

        @Override
        public final Builder encryptionContextSubset(Map<String, String> encryptionContextSubset) {
            this.encryptionContextSubset = EncryptionContextTypeCopier.copy(encryptionContextSubset);
            return this;
        }

        public final void setEncryptionContextSubset(Map<String, String> encryptionContextSubset) {
            this.encryptionContextSubset = EncryptionContextTypeCopier.copy(encryptionContextSubset);
        }

        public final Map<String, String> getEncryptionContextEquals() {
            return encryptionContextEquals;
        }

        @Override
        public final Builder encryptionContextEquals(Map<String, String> encryptionContextEquals) {
            this.encryptionContextEquals = EncryptionContextTypeCopier.copy(encryptionContextEquals);
            return this;
        }

        public final void setEncryptionContextEquals(Map<String, String> encryptionContextEquals) {
            this.encryptionContextEquals = EncryptionContextTypeCopier.copy(encryptionContextEquals);
        }

        @Override
        public GrantConstraints build() {
            return new GrantConstraints(this);
        }
    }
}
