/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.iotsitewise.IoTSiteWiseAsyncClient;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetModelRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetModelResponse;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetResponse;
import software.amazon.awssdk.services.iotsitewise.model.DescribePortalRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribePortalResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface IoTSiteWiseAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeAssetRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetActive(
            DescribeAssetRequest describeAssetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAssetRequest#builder()}
     *
     * @param describeAssetRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetActive(
            Consumer<DescribeAssetRequest.Builder> describeAssetRequest) {
        return waitUntilAssetActive(DescribeAssetRequest.builder().applyMutation(describeAssetRequest).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeAssetRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetActive(
            DescribeAssetRequest describeAssetRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAssetRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetActive(
            Consumer<DescribeAssetRequest.Builder> describeAssetRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAssetActive(DescribeAssetRequest.builder().applyMutation(describeAssetRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition {@code AssetModelActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeAssetModelRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelActive(
            DescribeAssetModelRequest describeAssetModelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition {@code AssetModelActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAssetModelRequest#builder()}
     *
     * @param describeAssetModelRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelActive(
            Consumer<DescribeAssetModelRequest.Builder> describeAssetModelRequest) {
        return waitUntilAssetModelActive(DescribeAssetModelRequest.builder().applyMutation(describeAssetModelRequest).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition {@code AssetModelActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeAssetModelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelActive(
            DescribeAssetModelRequest describeAssetModelRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition {@code AssetModelActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAssetModelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelActive(
            Consumer<DescribeAssetModelRequest.Builder> describeAssetModelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAssetModelActive(DescribeAssetModelRequest.builder().applyMutation(describeAssetModelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition
     * {@code AssetModelNotExists} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeAssetModelRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelNotExists(
            DescribeAssetModelRequest describeAssetModelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition
     * {@code AssetModelNotExists} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAssetModelRequest#builder()}
     *
     * @param describeAssetModelRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelNotExists(
            Consumer<DescribeAssetModelRequest.Builder> describeAssetModelRequest) {
        return waitUntilAssetModelNotExists(DescribeAssetModelRequest.builder().applyMutation(describeAssetModelRequest).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition
     * {@code AssetModelNotExists} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeAssetModelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelNotExists(
            DescribeAssetModelRequest describeAssetModelRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAssetModel} API until the desired condition
     * {@code AssetModelNotExists} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAssetModelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelNotExists(
            Consumer<DescribeAssetModelRequest.Builder> describeAssetModelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAssetModelNotExists(DescribeAssetModelRequest.builder().applyMutation(describeAssetModelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetNotExists} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeAssetRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetNotExists(
            DescribeAssetRequest describeAssetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetNotExists} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAssetRequest#builder()}
     *
     * @param describeAssetRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetNotExists(
            Consumer<DescribeAssetRequest.Builder> describeAssetRequest) {
        return waitUntilAssetNotExists(DescribeAssetRequest.builder().applyMutation(describeAssetRequest).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetNotExists} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeAssetRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetNotExists(
            DescribeAssetRequest describeAssetRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describeAsset} API until the desired condition {@code AssetNotExists} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAssetRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetNotExists(
            Consumer<DescribeAssetRequest.Builder> describeAssetRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAssetNotExists(DescribeAssetRequest.builder().applyMutation(describeAssetRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalActive} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describePortalRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalActive(
            DescribePortalRequest describePortalRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalActive} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribePortalRequest#builder()}
     *
     * @param describePortalRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalActive(
            Consumer<DescribePortalRequest.Builder> describePortalRequest) {
        return waitUntilPortalActive(DescribePortalRequest.builder().applyMutation(describePortalRequest).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalActive} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describePortalRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalActive(
            DescribePortalRequest describePortalRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalActive} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describePortalRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalActive(
            Consumer<DescribePortalRequest.Builder> describePortalRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilPortalActive(DescribePortalRequest.builder().applyMutation(describePortalRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalNotExists} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describePortalRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalNotExists(
            DescribePortalRequest describePortalRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalNotExists} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribePortalRequest#builder()}
     *
     * @param describePortalRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalNotExists(
            Consumer<DescribePortalRequest.Builder> describePortalRequest) {
        return waitUntilPortalNotExists(DescribePortalRequest.builder().applyMutation(describePortalRequest).build());
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalNotExists} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describePortalRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalNotExists(
            DescribePortalRequest describePortalRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IoTSiteWiseAsyncClient#describePortal} API until the desired condition {@code PortalNotExists} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describePortalRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalNotExists(
            Consumer<DescribePortalRequest.Builder> describePortalRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilPortalNotExists(DescribePortalRequest.builder().applyMutation(describePortalRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link IoTSiteWiseAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultIoTSiteWiseAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link IoTSiteWiseAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link IoTSiteWiseAsyncClient} will be created to poll resources. It is recommended to share a
     * single instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link IoTSiteWiseAsyncWaiter}
     */
    static IoTSiteWiseAsyncWaiter create() {
        return DefaultIoTSiteWiseAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link IoTSiteWiseAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(IoTSiteWiseAsyncClient client);

        /**
         * Builds an instance of {@link IoTSiteWiseAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link IoTSiteWiseAsyncWaiter}
         */
        IoTSiteWiseAsyncWaiter build();
    }
}
