/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains an asset metric property. With metrics, you can calculate aggregate functions, such as an average, maximum,
 * or minimum, as specified through an expression. A metric maps several values to a single value (such as a sum).
 * </p>
 * <p>
 * The maximum number of dependent/cascading variables used in any one metric calculation is 10. Therefore, a
 * <i>root</i> metric can have up to 10 cascading metrics in its computational dependency tree. Additionally, a metric
 * can only have a data type of <code>DOUBLE</code> and consume properties with data types of <code>INTEGER</code> or
 * <code>DOUBLE</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/asset-properties.html#metrics">Metrics</a> in the
 * <i>AWS IoT SiteWise User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Metric implements SdkPojo, Serializable, ToCopyableBuilder<Metric.Builder, Metric> {
    private static final SdkField<String> EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Metric::expression)).setter(setter(Builder::expression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expression").build()).build();

    private static final SdkField<List<ExpressionVariable>> VARIABLES_FIELD = SdkField
            .<List<ExpressionVariable>> builder(MarshallingType.LIST)
            .getter(getter(Metric::variables))
            .setter(setter(Builder::variables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExpressionVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExpressionVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<MetricWindow> WINDOW_FIELD = SdkField.<MetricWindow> builder(MarshallingType.SDK_POJO)
            .getter(getter(Metric::window)).setter(setter(Builder::window)).constructor(MetricWindow::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("window").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPRESSION_FIELD,
            VARIABLES_FIELD, WINDOW_FIELD));

    private static final long serialVersionUID = 1L;

    private final String expression;

    private final List<ExpressionVariable> variables;

    private final MetricWindow window;

    private Metric(BuilderImpl builder) {
        this.expression = builder.expression;
        this.variables = builder.variables;
        this.window = builder.window;
    }

    /**
     * <p>
     * The mathematical expression that defines the metric aggregation function. You can specify up to 10 variables per
     * expression. You can specify up to 10 functions per expression.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>AWS IoT
     * SiteWise User Guide</i>.
     * </p>
     * 
     * @return The mathematical expression that defines the metric aggregation function. You can specify up to 10
     *         variables per expression. You can specify up to 10 functions per expression. </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>AWS IoT
     *         SiteWise User Guide</i>.
     */
    public String expression() {
        return expression;
    }

    /**
     * Returns true if the Variables property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasVariables() {
        return variables != null && !(variables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of variables used in the expression.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVariables()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of variables used in the expression.
     */
    public List<ExpressionVariable> variables() {
        return variables;
    }

    /**
     * <p>
     * The window (time interval) over which AWS IoT SiteWise computes the metric's aggregation expression. AWS IoT
     * SiteWise computes one data point per <code>window</code>.
     * </p>
     * 
     * @return The window (time interval) over which AWS IoT SiteWise computes the metric's aggregation expression. AWS
     *         IoT SiteWise computes one data point per <code>window</code>.
     */
    public MetricWindow window() {
        return window;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        hashCode = 31 * hashCode + Objects.hashCode(variables());
        hashCode = 31 * hashCode + Objects.hashCode(window());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Metric)) {
            return false;
        }
        Metric other = (Metric) obj;
        return Objects.equals(expression(), other.expression()) && Objects.equals(variables(), other.variables())
                && Objects.equals(window(), other.window());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Metric").add("Expression", expression()).add("Variables", variables()).add("Window", window())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "expression":
            return Optional.ofNullable(clazz.cast(expression()));
        case "variables":
            return Optional.ofNullable(clazz.cast(variables()));
        case "window":
            return Optional.ofNullable(clazz.cast(window()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Metric, T> g) {
        return obj -> g.apply((Metric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Metric> {
        /**
         * <p>
         * The mathematical expression that defines the metric aggregation function. You can specify up to 10 variables
         * per expression. You can specify up to 10 functions per expression.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>AWS IoT
         * SiteWise User Guide</i>.
         * </p>
         * 
         * @param expression
         *        The mathematical expression that defines the metric aggregation function. You can specify up to 10
         *        variables per expression. You can specify up to 10 functions per expression. </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>AWS
         *        IoT SiteWise User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * 
         * @param variables
         *        The list of variables used in the expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(Collection<ExpressionVariable> variables);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * 
         * @param variables
         *        The list of variables used in the expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(ExpressionVariable... variables);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ExpressionVariable>.Builder} avoiding the
         * need to create one manually via {@link List<ExpressionVariable>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ExpressionVariable>.Builder#build()} is called immediately
         * and its result is passed to {@link #variables(List<ExpressionVariable>)}.
         * 
         * @param variables
         *        a consumer that will call methods on {@link List<ExpressionVariable>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variables(List<ExpressionVariable>)
         */
        Builder variables(Consumer<ExpressionVariable.Builder>... variables);

        /**
         * <p>
         * The window (time interval) over which AWS IoT SiteWise computes the metric's aggregation expression. AWS IoT
         * SiteWise computes one data point per <code>window</code>.
         * </p>
         * 
         * @param window
         *        The window (time interval) over which AWS IoT SiteWise computes the metric's aggregation expression.
         *        AWS IoT SiteWise computes one data point per <code>window</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder window(MetricWindow window);

        /**
         * <p>
         * The window (time interval) over which AWS IoT SiteWise computes the metric's aggregation expression. AWS IoT
         * SiteWise computes one data point per <code>window</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link MetricWindow.Builder} avoiding the need to
         * create one manually via {@link MetricWindow#builder()}.
         *
         * When the {@link Consumer} completes, {@link MetricWindow.Builder#build()} is called immediately and its
         * result is passed to {@link #window(MetricWindow)}.
         * 
         * @param window
         *        a consumer that will call methods on {@link MetricWindow.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #window(MetricWindow)
         */
        default Builder window(Consumer<MetricWindow.Builder> window) {
            return window(MetricWindow.builder().applyMutation(window).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String expression;

        private List<ExpressionVariable> variables = DefaultSdkAutoConstructList.getInstance();

        private MetricWindow window;

        private BuilderImpl() {
        }

        private BuilderImpl(Metric model) {
            expression(model.expression);
            variables(model.variables);
            window(model.window);
        }

        public final String getExpression() {
            return expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        public final Collection<ExpressionVariable.Builder> getVariables() {
            return variables != null ? variables.stream().map(ExpressionVariable::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder variables(Collection<ExpressionVariable> variables) {
            this.variables = ExpressionVariablesCopier.copy(variables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(ExpressionVariable... variables) {
            variables(Arrays.asList(variables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(Consumer<ExpressionVariable.Builder>... variables) {
            variables(Stream.of(variables).map(c -> ExpressionVariable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setVariables(Collection<ExpressionVariable.BuilderImpl> variables) {
            this.variables = ExpressionVariablesCopier.copyFromBuilder(variables);
        }

        public final MetricWindow.Builder getWindow() {
            return window != null ? window.toBuilder() : null;
        }

        @Override
        public final Builder window(MetricWindow window) {
            this.window = window;
            return this;
        }

        public final void setWindow(MetricWindow.BuilderImpl window) {
            this.window = window != null ? window.build() : null;
        }

        @Override
        public Metric build() {
            return new Metric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
