/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotjobsdataplane;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.iotjobsdataplane.internal.IotJobsDataPlaneServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotjobsdataplane.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.iotjobsdataplane.model.CertificateValidationException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ConflictException;
import software.amazon.awssdk.services.iotjobsdataplane.model.DescribeJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.DescribeJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.GetPendingJobExecutionsRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.GetPendingJobExecutionsResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.InternalServerException;
import software.amazon.awssdk.services.iotjobsdataplane.model.InvalidRequestException;
import software.amazon.awssdk.services.iotjobsdataplane.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.iotjobsdataplane.model.IotJobsDataPlaneException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartCommandExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartCommandExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartNextPendingJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartNextPendingJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.TerminalStateException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ThrottlingException;
import software.amazon.awssdk.services.iotjobsdataplane.model.UpdateJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.UpdateJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.ValidationException;
import software.amazon.awssdk.services.iotjobsdataplane.transform.DescribeJobExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.GetPendingJobExecutionsRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.StartCommandExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.StartNextPendingJobExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.UpdateJobExecutionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotJobsDataPlaneAsyncClient}.
 *
 * @see IotJobsDataPlaneAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotJobsDataPlaneAsyncClient implements IotJobsDataPlaneAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotJobsDataPlaneAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotJobsDataPlaneAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "IoT_Jobs_Data_Plane" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Gets details of a job execution.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DescribeJobExecution</a> action.
     * </p>
     *
     * @param describeJobExecutionRequest
     * @return A Java Future containing the result of the DescribeJobExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>TerminalStateException The job is in a terminal state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.DescribeJobExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/DescribeJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobExecutionResponse> describeJobExecution(
            DescribeJobExecutionRequest describeJobExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeJobExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "TerminalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TerminalStateException").httpStatusCode(410)
                            .exceptionBuilderSupplier(TerminalStateException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "CertificateValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CertificateValidationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(CertificateValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InvalidStateTransitionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                            .httpStatusCode(409).exceptionBuilderSupplier(InvalidStateTransitionException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeJobExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobExecutionRequest, DescribeJobExecutionResponse>()
                            .withOperationName("DescribeJobExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeJobExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeJobExecutionRequest));
            CompletableFuture<DescribeJobExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the list of all jobs for a thing that are not in a terminal status.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetPendingJobExecutions</a> action.
     * </p>
     *
     * @param getPendingJobExecutionsRequest
     * @return A Java Future containing the result of the GetPendingJobExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.GetPendingJobExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/GetPendingJobExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPendingJobExecutionsResponse> getPendingJobExecutions(
            GetPendingJobExecutionsRequest getPendingJobExecutionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPendingJobExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPendingJobExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPendingJobExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPendingJobExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPendingJobExecutionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "TerminalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TerminalStateException").httpStatusCode(410)
                            .exceptionBuilderSupplier(TerminalStateException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "CertificateValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CertificateValidationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(CertificateValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InvalidStateTransitionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                            .httpStatusCode(409).exceptionBuilderSupplier(InvalidStateTransitionException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPendingJobExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPendingJobExecutionsRequest, GetPendingJobExecutionsResponse>()
                            .withOperationName("GetPendingJobExecutions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPendingJobExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPendingJobExecutionsRequest));
            CompletableFuture<GetPendingJobExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Using the command created with the <code>CreateCommand</code> API, start a command execution on a specific
     * device.
     * </p>
     *
     * @param startCommandExecutionRequest
     * @return A Java Future containing the result of the StartCommandExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException A validation error occurred when performing the API request.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ConflictException A conflict has occurred when performing the API request.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this request.</li>
     *         <li>InternalServerException An internal server error occurred when performing the API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.StartCommandExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/StartCommandExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartCommandExecutionResponse> startCommandExecution(
            StartCommandExecutionRequest startCommandExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCommandExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCommandExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCommandExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCommandExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartCommandExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "TerminalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TerminalStateException").httpStatusCode(410)
                            .exceptionBuilderSupplier(TerminalStateException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "CertificateValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CertificateValidationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(CertificateValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InvalidStateTransitionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                            .httpStatusCode(409).exceptionBuilderSupplier(InvalidStateTransitionException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartCommandExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCommandExecutionRequest, StartCommandExecutionResponse>()
                            .withOperationName("StartCommandExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartCommandExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startCommandExecutionRequest));
            CompletableFuture<StartCommandExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets and starts the next pending (status IN_PROGRESS or QUEUED) job execution for a thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >StartNextPendingJobExecution</a> action.
     * </p>
     *
     * @param startNextPendingJobExecutionRequest
     * @return A Java Future containing the result of the StartNextPendingJobExecution operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.StartNextPendingJobExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/StartNextPendingJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartNextPendingJobExecutionResponse> startNextPendingJobExecution(
            StartNextPendingJobExecutionRequest startNextPendingJobExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startNextPendingJobExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startNextPendingJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartNextPendingJobExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartNextPendingJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartNextPendingJobExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "TerminalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TerminalStateException").httpStatusCode(410)
                            .exceptionBuilderSupplier(TerminalStateException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "CertificateValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CertificateValidationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(CertificateValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InvalidStateTransitionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                            .httpStatusCode(409).exceptionBuilderSupplier(InvalidStateTransitionException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartNextPendingJobExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartNextPendingJobExecutionRequest, StartNextPendingJobExecutionResponse>()
                            .withOperationName("StartNextPendingJobExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartNextPendingJobExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startNextPendingJobExecutionRequest));
            CompletableFuture<StartNextPendingJobExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status of a job execution.
     * </p>
     * <p>
     * Requires permission to access the <a
     * href="https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotjobsdataplane.html"
     * >UpdateJobExecution</a> action.
     * </p>
     *
     * @param updateJobExecutionRequest
     * @return A Java Future containing the result of the UpdateJobExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>InvalidStateTransitionException An update attempted to change the job execution to a state that is
     *         invalid because of the job execution's current state (for example, an attempt to change a request in
     *         state SUCCESS to state IN_PROGRESS). In this case, the body of the error message also contains the
     *         executionState field.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.UpdateJobExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/UpdateJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobExecutionResponse> updateJobExecution(UpdateJobExecutionRequest updateJobExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateJobExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateJobExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "TerminalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TerminalStateException").httpStatusCode(410)
                            .exceptionBuilderSupplier(TerminalStateException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "CertificateValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CertificateValidationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(CertificateValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InvalidStateTransitionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                            .httpStatusCode(409).exceptionBuilderSupplier(InvalidStateTransitionException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateJobExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobExecutionRequest, UpdateJobExecutionResponse>()
                            .withOperationName("UpdateJobExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateJobExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateJobExecutionRequest));
            CompletableFuture<UpdateJobExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IotJobsDataPlaneServiceClientConfiguration serviceClientConfiguration() {
        return new IotJobsDataPlaneServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotJobsDataPlaneException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        IotJobsDataPlaneServiceClientConfigurationBuilder serviceConfigBuilder = new IotJobsDataPlaneServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
