/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ioteventsdata;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ioteventsdata.model.BatchPutMessageRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchPutMessageResponse;
import software.amazon.awssdk.services.ioteventsdata.model.BatchUpdateDetectorRequest;
import software.amazon.awssdk.services.ioteventsdata.model.BatchUpdateDetectorResponse;
import software.amazon.awssdk.services.ioteventsdata.model.DescribeDetectorRequest;
import software.amazon.awssdk.services.ioteventsdata.model.DescribeDetectorResponse;
import software.amazon.awssdk.services.ioteventsdata.model.InternalFailureException;
import software.amazon.awssdk.services.ioteventsdata.model.InvalidRequestException;
import software.amazon.awssdk.services.ioteventsdata.model.IotEventsDataException;
import software.amazon.awssdk.services.ioteventsdata.model.ListDetectorsRequest;
import software.amazon.awssdk.services.ioteventsdata.model.ListDetectorsResponse;
import software.amazon.awssdk.services.ioteventsdata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ioteventsdata.model.ServiceUnavailableException;
import software.amazon.awssdk.services.ioteventsdata.model.ThrottlingException;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchPutMessageRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.BatchUpdateDetectorRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.DescribeDetectorRequestMarshaller;
import software.amazon.awssdk.services.ioteventsdata.transform.ListDetectorsRequestMarshaller;

/**
 * Internal implementation of {@link IotEventsDataClient}.
 *
 * @see IotEventsDataClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotEventsDataClient implements IotEventsDataClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotEventsDataClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Sends a set of messages to the AWS IoT Events system. Each message payload is transformed into the input you
     * specify (<code>"inputName"</code>) and ingested into any detectors that monitor that input. If multiple messages
     * are sent, the order in which the messages are processed isn't guaranteed. To guarantee ordering, you must send
     * messages one at a time and wait for a successful response.
     * </p>
     *
     * @param batchPutMessageRequest
     * @return Result of the BatchPutMessage operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws InternalFailureException
     *         An internal failure occured.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsDataClient.BatchPutMessage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchPutMessage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchPutMessageResponse batchPutMessage(BatchPutMessageRequest batchPutMessageRequest) throws InvalidRequestException,
            InternalFailureException, ServiceUnavailableException, ThrottlingException, AwsServiceException, SdkClientException,
            IotEventsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchPutMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchPutMessageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchPutMessageRequest, BatchPutMessageResponse>()
                .withOperationName("BatchPutMessage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchPutMessageRequest)
                .withMarshaller(new BatchPutMessageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the state, variable values, and timer settings of one or more detectors (instances) of a specified
     * detector model.
     * </p>
     *
     * @param batchUpdateDetectorRequest
     * @return Result of the BatchUpdateDetector operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws InternalFailureException
     *         An internal failure occured.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsDataClient.BatchUpdateDetector
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/BatchUpdateDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchUpdateDetectorResponse batchUpdateDetector(BatchUpdateDetectorRequest batchUpdateDetectorRequest)
            throws InvalidRequestException, InternalFailureException, ServiceUnavailableException, ThrottlingException,
            AwsServiceException, SdkClientException, IotEventsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUpdateDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchUpdateDetectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchUpdateDetectorRequest, BatchUpdateDetectorResponse>()
                .withOperationName("BatchUpdateDetector").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchUpdateDetectorRequest)
                .withMarshaller(new BatchUpdateDetectorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about the specified detector (instance).
     * </p>
     *
     * @param describeDetectorRequest
     * @return Result of the DescribeDetector operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occured.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsDataClient.DescribeDetector
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/DescribeDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDetectorResponse describeDetector(DescribeDetectorRequest describeDetectorRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDetectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDetectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeDetectorRequest, DescribeDetectorResponse>()
                .withOperationName("DescribeDetector").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeDetectorRequest)
                .withMarshaller(new DescribeDetectorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists detectors (the instances of a detector model).
     * </p>
     *
     * @param listDetectorsRequest
     * @return Result of the ListDetectors operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occured.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsDataClient.ListDetectors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-data-2018-10-23/ListDetectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDetectorsResponse listDetectors(ListDetectorsRequest listDetectorsRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDetectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDetectorsRequest, ListDetectorsResponse>()
                .withOperationName("ListDetectors").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDetectorsRequest)
                .withMarshaller(new ListDetectorsRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotEventsDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
