/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdeviceadvisor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Shows tests in a test group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TestCase implements SdkPojo, Serializable, ToCopyableBuilder<TestCase.Builder, TestCase> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(TestCase::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Map<String, String>> CONFIGURATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("configuration")
            .getter(getter(TestCase::configuration))
            .setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<TestCaseDefinition> TEST_FIELD = SdkField
            .<TestCaseDefinition> builder(MarshallingType.SDK_POJO).memberName("test").getter(getter(TestCase::test))
            .setter(setter(Builder::test)).constructor(TestCaseDefinition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("test").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONFIGURATION_FIELD, TEST_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Map<String, String> configuration;

    private final TestCaseDefinition test;

    private TestCase(BuilderImpl builder) {
        this.name = builder.name;
        this.configuration = builder.configuration;
        this.test = builder.test;
    }

    /**
     * <p>
     * Shows test case name.
     * </p>
     * 
     * @return Shows test case name.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns true if the Configuration property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasConfiguration() {
        return configuration != null && !(configuration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Shows test case configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfiguration()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Shows test case configuration.
     */
    public final Map<String, String> configuration() {
        return configuration;
    }

    /**
     * <p>
     * Specifies a test.
     * </p>
     * 
     * @return Specifies a test.
     */
    public final TestCaseDefinition test() {
        return test;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfiguration() ? configuration() : null);
        hashCode = 31 * hashCode + Objects.hashCode(test());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TestCase)) {
            return false;
        }
        TestCase other = (TestCase) obj;
        return Objects.equals(name(), other.name()) && hasConfiguration() == other.hasConfiguration()
                && Objects.equals(configuration(), other.configuration()) && Objects.equals(test(), other.test());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TestCase").add("Name", name()).add("Configuration", hasConfiguration() ? configuration() : null)
                .add("Test", test()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "test":
            return Optional.ofNullable(clazz.cast(test()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TestCase, T> g) {
        return obj -> g.apply((TestCase) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TestCase> {
        /**
         * <p>
         * Shows test case name.
         * </p>
         * 
         * @param name
         *        Shows test case name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Shows test case configuration.
         * </p>
         * 
         * @param configuration
         *        Shows test case configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(Map<String, String> configuration);

        /**
         * <p>
         * Specifies a test.
         * </p>
         * 
         * @param test
         *        Specifies a test.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder test(TestCaseDefinition test);

        /**
         * <p>
         * Specifies a test.
         * </p>
         * This is a convenience that creates an instance of the {@link TestCaseDefinition.Builder} avoiding the need to
         * create one manually via {@link TestCaseDefinition#builder()}.
         *
         * When the {@link Consumer} completes, {@link TestCaseDefinition.Builder#build()} is called immediately and its
         * result is passed to {@link #test(TestCaseDefinition)}.
         * 
         * @param test
         *        a consumer that will call methods on {@link TestCaseDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #test(TestCaseDefinition)
         */
        default Builder test(Consumer<TestCaseDefinition.Builder> test) {
            return test(TestCaseDefinition.builder().applyMutation(test).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Map<String, String> configuration = DefaultSdkAutoConstructMap.getInstance();

        private TestCaseDefinition test;

        private BuilderImpl() {
        }

        private BuilderImpl(TestCase model) {
            name(model.name);
            configuration(model.configuration);
            test(model.test);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Map<String, String> getConfiguration() {
            if (configuration instanceof SdkAutoConstructMap) {
                return null;
            }
            return configuration;
        }

        @Override
        public final Builder configuration(Map<String, String> configuration) {
            this.configuration = TestConfigurationCopier.copy(configuration);
            return this;
        }

        public final void setConfiguration(Map<String, String> configuration) {
            this.configuration = TestConfigurationCopier.copy(configuration);
        }

        public final TestCaseDefinition.Builder getTest() {
            return test != null ? test.toBuilder() : null;
        }

        @Override
        public final Builder test(TestCaseDefinition test) {
            this.test = test;
            return this;
        }

        public final void setTest(TestCaseDefinition.BuilderImpl test) {
            this.test = test != null ? test.build() : null;
        }

        @Override
        public TestCase build() {
            return new TestCase(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
