/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type with the configuration information that determines the threshold and other conditions for when
 * Internet Monitor creates a health event for an overall performance or availability issue, across an application's
 * geographies.
 * </p>
 * <p>
 * Defines the percentages, for overall performance scores and availability scores for an application, that are the
 * thresholds for when Amazon CloudWatch Internet Monitor creates a health event. You can override the defaults to set a
 * custom threshold for overall performance or availability scores, or both.
 * </p>
 * <p>
 * You can also set thresholds for local health scores,, where Internet Monitor creates a health event when scores cross
 * a threshold for one or more city-networks, in addition to creating an event when an overall score crosses a
 * threshold.
 * </p>
 * <p>
 * If you don't set a health event threshold, the default value is 95%.
 * </p>
 * <p>
 * For local thresholds, you also set a minimum percentage of overall traffic that is impacted by an issue before
 * Internet Monitor creates an event. In addition, you can disable local thresholds, for performance scores,
 * availability scores, or both.
 * </p>
 * <p>
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
 * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HealthEventsConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HealthEventsConfig.Builder, HealthEventsConfig> {
    private static final SdkField<Double> AVAILABILITY_SCORE_THRESHOLD_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("AvailabilityScoreThreshold")
            .getter(getter(HealthEventsConfig::availabilityScoreThreshold))
            .setter(setter(Builder::availabilityScoreThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityScoreThreshold").build())
            .build();

    private static final SdkField<Double> PERFORMANCE_SCORE_THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("PerformanceScoreThreshold").getter(getter(HealthEventsConfig::performanceScoreThreshold))
            .setter(setter(Builder::performanceScoreThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformanceScoreThreshold").build())
            .build();

    private static final SdkField<LocalHealthEventsConfig> AVAILABILITY_LOCAL_HEALTH_EVENTS_CONFIG_FIELD = SdkField
            .<LocalHealthEventsConfig> builder(MarshallingType.SDK_POJO)
            .memberName("AvailabilityLocalHealthEventsConfig")
            .getter(getter(HealthEventsConfig::availabilityLocalHealthEventsConfig))
            .setter(setter(Builder::availabilityLocalHealthEventsConfig))
            .constructor(LocalHealthEventsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("AvailabilityLocalHealthEventsConfig").build()).build();

    private static final SdkField<LocalHealthEventsConfig> PERFORMANCE_LOCAL_HEALTH_EVENTS_CONFIG_FIELD = SdkField
            .<LocalHealthEventsConfig> builder(MarshallingType.SDK_POJO)
            .memberName("PerformanceLocalHealthEventsConfig")
            .getter(getter(HealthEventsConfig::performanceLocalHealthEventsConfig))
            .setter(setter(Builder::performanceLocalHealthEventsConfig))
            .constructor(LocalHealthEventsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformanceLocalHealthEventsConfig")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AVAILABILITY_SCORE_THRESHOLD_FIELD, PERFORMANCE_SCORE_THRESHOLD_FIELD, AVAILABILITY_LOCAL_HEALTH_EVENTS_CONFIG_FIELD,
            PERFORMANCE_LOCAL_HEALTH_EVENTS_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Double availabilityScoreThreshold;

    private final Double performanceScoreThreshold;

    private final LocalHealthEventsConfig availabilityLocalHealthEventsConfig;

    private final LocalHealthEventsConfig performanceLocalHealthEventsConfig;

    private HealthEventsConfig(BuilderImpl builder) {
        this.availabilityScoreThreshold = builder.availabilityScoreThreshold;
        this.performanceScoreThreshold = builder.performanceScoreThreshold;
        this.availabilityLocalHealthEventsConfig = builder.availabilityLocalHealthEventsConfig;
        this.performanceLocalHealthEventsConfig = builder.performanceLocalHealthEventsConfig;
    }

    /**
     * <p>
     * The health event threshold percentage set for availability scores.
     * </p>
     * 
     * @return The health event threshold percentage set for availability scores.
     */
    public final Double availabilityScoreThreshold() {
        return availabilityScoreThreshold;
    }

    /**
     * <p>
     * The health event threshold percentage set for performance scores.
     * </p>
     * 
     * @return The health event threshold percentage set for performance scores.
     */
    public final Double performanceScoreThreshold() {
        return performanceScoreThreshold;
    }

    /**
     * <p>
     * The configuration that determines the threshold and other conditions for when Internet Monitor creates a health
     * event for a local availability issue.
     * </p>
     * 
     * @return The configuration that determines the threshold and other conditions for when Internet Monitor creates a
     *         health event for a local availability issue.
     */
    public final LocalHealthEventsConfig availabilityLocalHealthEventsConfig() {
        return availabilityLocalHealthEventsConfig;
    }

    /**
     * <p>
     * The configuration that determines the threshold and other conditions for when Internet Monitor creates a health
     * event for a local performance issue.
     * </p>
     * 
     * @return The configuration that determines the threshold and other conditions for when Internet Monitor creates a
     *         health event for a local performance issue.
     */
    public final LocalHealthEventsConfig performanceLocalHealthEventsConfig() {
        return performanceLocalHealthEventsConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(availabilityScoreThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(performanceScoreThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityLocalHealthEventsConfig());
        hashCode = 31 * hashCode + Objects.hashCode(performanceLocalHealthEventsConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HealthEventsConfig)) {
            return false;
        }
        HealthEventsConfig other = (HealthEventsConfig) obj;
        return Objects.equals(availabilityScoreThreshold(), other.availabilityScoreThreshold())
                && Objects.equals(performanceScoreThreshold(), other.performanceScoreThreshold())
                && Objects.equals(availabilityLocalHealthEventsConfig(), other.availabilityLocalHealthEventsConfig())
                && Objects.equals(performanceLocalHealthEventsConfig(), other.performanceLocalHealthEventsConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HealthEventsConfig").add("AvailabilityScoreThreshold", availabilityScoreThreshold())
                .add("PerformanceScoreThreshold", performanceScoreThreshold())
                .add("AvailabilityLocalHealthEventsConfig", availabilityLocalHealthEventsConfig())
                .add("PerformanceLocalHealthEventsConfig", performanceLocalHealthEventsConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvailabilityScoreThreshold":
            return Optional.ofNullable(clazz.cast(availabilityScoreThreshold()));
        case "PerformanceScoreThreshold":
            return Optional.ofNullable(clazz.cast(performanceScoreThreshold()));
        case "AvailabilityLocalHealthEventsConfig":
            return Optional.ofNullable(clazz.cast(availabilityLocalHealthEventsConfig()));
        case "PerformanceLocalHealthEventsConfig":
            return Optional.ofNullable(clazz.cast(performanceLocalHealthEventsConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AvailabilityScoreThreshold", AVAILABILITY_SCORE_THRESHOLD_FIELD);
        map.put("PerformanceScoreThreshold", PERFORMANCE_SCORE_THRESHOLD_FIELD);
        map.put("AvailabilityLocalHealthEventsConfig", AVAILABILITY_LOCAL_HEALTH_EVENTS_CONFIG_FIELD);
        map.put("PerformanceLocalHealthEventsConfig", PERFORMANCE_LOCAL_HEALTH_EVENTS_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HealthEventsConfig, T> g) {
        return obj -> g.apply((HealthEventsConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HealthEventsConfig> {
        /**
         * <p>
         * The health event threshold percentage set for availability scores.
         * </p>
         * 
         * @param availabilityScoreThreshold
         *        The health event threshold percentage set for availability scores.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityScoreThreshold(Double availabilityScoreThreshold);

        /**
         * <p>
         * The health event threshold percentage set for performance scores.
         * </p>
         * 
         * @param performanceScoreThreshold
         *        The health event threshold percentage set for performance scores.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceScoreThreshold(Double performanceScoreThreshold);

        /**
         * <p>
         * The configuration that determines the threshold and other conditions for when Internet Monitor creates a
         * health event for a local availability issue.
         * </p>
         * 
         * @param availabilityLocalHealthEventsConfig
         *        The configuration that determines the threshold and other conditions for when Internet Monitor creates
         *        a health event for a local availability issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityLocalHealthEventsConfig(LocalHealthEventsConfig availabilityLocalHealthEventsConfig);

        /**
         * <p>
         * The configuration that determines the threshold and other conditions for when Internet Monitor creates a
         * health event for a local availability issue.
         * </p>
         * This is a convenience method that creates an instance of the {@link LocalHealthEventsConfig.Builder} avoiding
         * the need to create one manually via {@link LocalHealthEventsConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LocalHealthEventsConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #availabilityLocalHealthEventsConfig(LocalHealthEventsConfig)}.
         * 
         * @param availabilityLocalHealthEventsConfig
         *        a consumer that will call methods on {@link LocalHealthEventsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availabilityLocalHealthEventsConfig(LocalHealthEventsConfig)
         */
        default Builder availabilityLocalHealthEventsConfig(
                Consumer<LocalHealthEventsConfig.Builder> availabilityLocalHealthEventsConfig) {
            return availabilityLocalHealthEventsConfig(LocalHealthEventsConfig.builder()
                    .applyMutation(availabilityLocalHealthEventsConfig).build());
        }

        /**
         * <p>
         * The configuration that determines the threshold and other conditions for when Internet Monitor creates a
         * health event for a local performance issue.
         * </p>
         * 
         * @param performanceLocalHealthEventsConfig
         *        The configuration that determines the threshold and other conditions for when Internet Monitor creates
         *        a health event for a local performance issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceLocalHealthEventsConfig(LocalHealthEventsConfig performanceLocalHealthEventsConfig);

        /**
         * <p>
         * The configuration that determines the threshold and other conditions for when Internet Monitor creates a
         * health event for a local performance issue.
         * </p>
         * This is a convenience method that creates an instance of the {@link LocalHealthEventsConfig.Builder} avoiding
         * the need to create one manually via {@link LocalHealthEventsConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LocalHealthEventsConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #performanceLocalHealthEventsConfig(LocalHealthEventsConfig)}.
         * 
         * @param performanceLocalHealthEventsConfig
         *        a consumer that will call methods on {@link LocalHealthEventsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #performanceLocalHealthEventsConfig(LocalHealthEventsConfig)
         */
        default Builder performanceLocalHealthEventsConfig(
                Consumer<LocalHealthEventsConfig.Builder> performanceLocalHealthEventsConfig) {
            return performanceLocalHealthEventsConfig(LocalHealthEventsConfig.builder()
                    .applyMutation(performanceLocalHealthEventsConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Double availabilityScoreThreshold;

        private Double performanceScoreThreshold;

        private LocalHealthEventsConfig availabilityLocalHealthEventsConfig;

        private LocalHealthEventsConfig performanceLocalHealthEventsConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(HealthEventsConfig model) {
            availabilityScoreThreshold(model.availabilityScoreThreshold);
            performanceScoreThreshold(model.performanceScoreThreshold);
            availabilityLocalHealthEventsConfig(model.availabilityLocalHealthEventsConfig);
            performanceLocalHealthEventsConfig(model.performanceLocalHealthEventsConfig);
        }

        public final Double getAvailabilityScoreThreshold() {
            return availabilityScoreThreshold;
        }

        public final void setAvailabilityScoreThreshold(Double availabilityScoreThreshold) {
            this.availabilityScoreThreshold = availabilityScoreThreshold;
        }

        @Override
        public final Builder availabilityScoreThreshold(Double availabilityScoreThreshold) {
            this.availabilityScoreThreshold = availabilityScoreThreshold;
            return this;
        }

        public final Double getPerformanceScoreThreshold() {
            return performanceScoreThreshold;
        }

        public final void setPerformanceScoreThreshold(Double performanceScoreThreshold) {
            this.performanceScoreThreshold = performanceScoreThreshold;
        }

        @Override
        public final Builder performanceScoreThreshold(Double performanceScoreThreshold) {
            this.performanceScoreThreshold = performanceScoreThreshold;
            return this;
        }

        public final LocalHealthEventsConfig.Builder getAvailabilityLocalHealthEventsConfig() {
            return availabilityLocalHealthEventsConfig != null ? availabilityLocalHealthEventsConfig.toBuilder() : null;
        }

        public final void setAvailabilityLocalHealthEventsConfig(
                LocalHealthEventsConfig.BuilderImpl availabilityLocalHealthEventsConfig) {
            this.availabilityLocalHealthEventsConfig = availabilityLocalHealthEventsConfig != null ? availabilityLocalHealthEventsConfig
                    .build() : null;
        }

        @Override
        public final Builder availabilityLocalHealthEventsConfig(LocalHealthEventsConfig availabilityLocalHealthEventsConfig) {
            this.availabilityLocalHealthEventsConfig = availabilityLocalHealthEventsConfig;
            return this;
        }

        public final LocalHealthEventsConfig.Builder getPerformanceLocalHealthEventsConfig() {
            return performanceLocalHealthEventsConfig != null ? performanceLocalHealthEventsConfig.toBuilder() : null;
        }

        public final void setPerformanceLocalHealthEventsConfig(
                LocalHealthEventsConfig.BuilderImpl performanceLocalHealthEventsConfig) {
            this.performanceLocalHealthEventsConfig = performanceLocalHealthEventsConfig != null ? performanceLocalHealthEventsConfig
                    .build() : null;
        }

        @Override
        public final Builder performanceLocalHealthEventsConfig(LocalHealthEventsConfig performanceLocalHealthEventsConfig) {
            this.performanceLocalHealthEventsConfig = performanceLocalHealthEventsConfig;
            return this;
        }

        @Override
        public HealthEventsConfig build() {
            return new HealthEventsConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
