/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.inspector.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.inspector.InspectorClient;
import software.amazon.awssdk.services.inspector.model.GetExclusionsPreviewRequest;
import software.amazon.awssdk.services.inspector.model.GetExclusionsPreviewResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.inspector.InspectorClient#getExclusionsPreviewPaginator(software.amazon.awssdk.services.inspector.model.GetExclusionsPreviewRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.inspector.model.GetExclusionsPreviewResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.inspector.paginators.GetExclusionsPreviewIterable responses = client.getExclusionsPreviewPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.inspector.paginators.GetExclusionsPreviewIterable responses = client
 *             .getExclusionsPreviewPaginator(request);
 *     for (software.amazon.awssdk.services.inspector.model.GetExclusionsPreviewResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.inspector.paginators.GetExclusionsPreviewIterable responses = client.getExclusionsPreviewPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getExclusionsPreview(software.amazon.awssdk.services.inspector.model.GetExclusionsPreviewRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetExclusionsPreviewIterable implements SdkIterable<GetExclusionsPreviewResponse> {
    private final InspectorClient client;

    private final GetExclusionsPreviewRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public GetExclusionsPreviewIterable(InspectorClient client, GetExclusionsPreviewRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new GetExclusionsPreviewResponseFetcher();
    }

    @Override
    public Iterator<GetExclusionsPreviewResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class GetExclusionsPreviewResponseFetcher implements SyncPageFetcher<GetExclusionsPreviewResponse> {
        @Override
        public boolean hasNextPage(GetExclusionsPreviewResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public GetExclusionsPreviewResponse nextPage(GetExclusionsPreviewResponse previousPage) {
            if (previousPage == null) {
                return client.getExclusionsPreview(firstRequest);
            }
            return client.getExclusionsPreview(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
